//
// NAME: TestCIDLib_FileSystem.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  file and file system classes to make sure that they are functioning
//  correctly.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/28/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    <windows.h>
#undef      DeleteFile
#undef      RemoveDirectory


// -----------------------------------------------------------------------------
//  Local constant data
//
//  Note the extension "TestFile", which I've set up in my build shell's
//  scrub list. This way it will get cleaned up when I scrub this project.
// -----------------------------------------------------------------------------
static const TString    strBinFileName(L"BinaryFileTest.TestFile");


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __TestXXXX
//
// DESCRIPTION:
//
//  These methods all test various aspects of the file system classes.
// ---------------------------------------
//   INPUT: strmOut is the text stream to send messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestBasicFileSystem(TTextStream& strmOut)
{
    tCIDLib::EFileAttrs eAttrs;
    TFileSys            flsysTest;

    // Check for some files that we know exist
    if (!flsysTest.bExists(L"C:\\Config.Sys"))
        strmOut <<  _CurLn_ << L"Did not find C:\\Config.Sys" << NewLn;

    // Use the version that concatenates the path for us
    if (!flsysTest.bExists(L"C:\\WinNT\\System32", L"Help.Exe"))
        strmOut <<  _CurLn_ << L"Did not find C:\\WinNT\\System32\\Help.Exe" << NewLn;

    // Try some that check for particular attributes
    if (!flsysTest.bIsDirectory(L"C:\\WinNT"))
    {
        strmOut << _CurLn_
                << L"Directory C:\\WinNT not found." << NewLn;
    }

    eAttrs = tCIDLib::EFileAttr_HiddenSystemRead;
    if (flsysTest.bExists(L"C:\\IO.SYS", eAttrs))
    {
        if (!(eAttrs & tCIDLib::EFileAttr_Hidden)
        ||  !(eAttrs & tCIDLib::EFileAttr_System)
        ||  !(eAttrs & tCIDLib::EFileAttr_ReadOnly))
        {
            strmOut << _CurLn_
                    << L"Found file C:\\IO.SYS did not have "
                        L"the HSR attributes." << NewLn;
        }
    }
     else
    {
        strmOut <<  _CurLn_ << L"Did not find C:\\IO.SYS" << NewLn;
    }

    eAttrs = tCIDLib::EFileAttr_SystemReadOnly;
    if (flsysTest.bExists(L"C:\\Boot.Ini", eAttrs))
    {
        if (!(eAttrs & tCIDLib::EFileAttr_ReadOnly)
        ||  !(eAttrs & tCIDLib::EFileAttr_System))
        {
            strmOut << _CurLn_
                    << L"Found file 'C:\\Boot.Ini' did not have "
                        L"the SR attributes." << NewLn;
        }
    }
     else
    {
        strmOut <<  _CurLn_ << L"Did not find 'C:\\Boot.Ini'" << NewLn;
    }
}


tCIDLib::TVoid __TestBinaryFiles(TTextStream& strmOut)
{
    tCIDLib::TCard4         ac4Test[100], c4Ind;
    tCIDLib::TCard4         c4Tmp;

    // Create a test binary file object
    TBinaryFile bflTest(strBinFileName);

    // Try to create it
    bflTest.Open
    (
        tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_ReplaceIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_RandomAccess
    );

    // Fill in the array with test data
    for (c4Ind = 0; c4Ind < 100; c4Ind++)
        ac4Test[c4Ind] = c4Ind;

    // Write out the data
    bflTest.WriteBuf(ac4Test, sizeof(ac4Test));

    // Seek back to the start of the file
    bflTest.Seek(0, tCIDLib::ESeekFrom_Start);

    //
    //  Read in the data again, but zero out the buffer first to make sure we
    //  really read it.
    //
    TRawMem::SetMemBuf(ac4Test, tCIDLib::TCard1(0), sizeof(ac4Test));
    bflTest.ReadBuf(ac4Test, sizeof(ac4Test));

    // Test the contents
    for (c4Ind = 0; c4Ind < 100; c4Ind++)
    {
        if (ac4Test[c4Ind] != c4Ind)
        {
            strmOut << _CurLn_
                    << L"Buffer on disk (" << ac4Test[c4Ind]
                    << L" != data written (" << c4Ind << L")" << NewLn;
            break;
        }
    }

    // Now close the file
    bflTest.Close();

    // Now do an open on the file now, since we know it exists
    bflTest.Open
    (
        tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_OpenIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_RandomAccess
    );

    //
    //  Read in the data again, but zero out the buffer first to make sure we
    //  really read it.
    //
    TRawMem::SetMemBuf(ac4Test, tCIDLib::TCard1(0), sizeof(ac4Test));
    bflTest.ReadBuf(ac4Test, sizeof(ac4Test));

    // Test the contents
    for (c4Ind = 0; c4Ind < 100; c4Ind++)
    {
        if (ac4Test[c4Ind] != c4Ind)
        {
            strmOut << _CurLn_
                    << L"Buffer on disk (" << ac4Test[c4Ind]
                    << L" != data written (" << c4Ind << L")" << NewLn;
            return;
        }
    }

    // Seek to the 10th item and read it in
    bflTest.Seek(10 * sizeof(tCIDLib::TCard4), tCIDLib::ESeekFrom_Start);
    bflTest.ReadBuf(&c4Tmp, sizeof(tCIDLib::TCard4));

    // Check the value
    if (c4Tmp != 10)
    {
        strmOut << _CurLn_
                << L"Read of 10th byte was not correct value failed. Value="
                << c4Tmp << NewLn;
    }

    // Now close the file and delete it
    bflTest.Close();
}


static tCIDLib::TVoid __TestBinStreamIO(TTextStream& strmOut)
{
    //
    //  Create a binary file stream object to use for the test. This guy
    //  does all the work for us of creating or opening the file and setting
    //  up the stream to use the file as its data sink/source.
    //
    TBinaryFileStream strmTest
    (
        strBinFileName
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_OpenIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_RandomAccess
    );

    // Write out some objects.
    TError errToIO
    (
        L"TestCIDLib"
        , __FILE__
        , __LINE__
        , L"This is a test message"
    );
    strmTest << errToIO;

    // Seek back to the start and read them in
    strmTest.Reset();

    // Create another object to read into and stream back in
    TError      errToRead;
    strmTest >> errToRead;

    if (errToRead != errToIO)
    {
        strmOut << _CurLn_
                << L"Stream I/O of monomorphic TError object failed" << NewLn;
    }

    //
    //  Now lets test polymorphic streaming. We seek back on the stream
    //  and write out a set of strings of various types, via the base
    //  string buffer pointer.
    //
    strmTest.Reset();

    //
    //  Create an object and stream it out polymorphically. We create the
    //  initial size of the string as less than the initialize string on
    //  purpose. It should expand up to the 16 and get it all!
    //
    TString strTest(L"String8-8", 8, 16);
    ::PolymorphicWrite(&strTest, strmTest);

    // And do another object, this time an area object
    TArea   areaTest(10, 10, 100, 100);
    ::PolymorphicWrite(&areaTest, strmTest);

    // And now a date object
    TTime  tmTest;
    ::PolymorphicWrite(&tmTest, strmTest);

    //
    //  Seek it back now and read back in the objects, and make sure we
    //  get back the types we wrote.
    //
    TString*    pstrTest = 0;
    TArea*      pareaTest = 0;
    TTime*      ptmTest = 0;
    strmTest.Reset();

    ::PolymorphicRead(pstrTest, strmTest);
    if (!pstrTest->bIsA(TString::clsThis))
        strmOut << _CurLn_ << L"Polymorphic read of TString failed." << NewLn;
    if (*pstrTest != TString(L"String8-8", 8, 16))
    {
        strmOut << _CurLn_
                << L"Polymorphic read of TString had wrong contents" << NewLn;
    }
    delete pstrTest;

    ::PolymorphicRead(pareaTest, strmTest);
    if (!pareaTest->bIsA(TArea::clsThis))
        strmOut << _CurLn_ << L"Polymorphic read of TArea failed." << NewLn;
    if (*pareaTest != TArea(10, 10, 100, 100))
    {
        strmOut << _CurLn_
                << L"Polymorphic read of TArea had wrong contents" << NewLn;
    }
    delete pareaTest;

    ::PolymorphicRead(ptmTest, strmTest);
    if (!ptmTest->bIsA(TTime::clsThis))
        strmOut << _CurLn_ << L"Polymorphic read of TTime failed." << NewLn;
    if (*ptmTest != tmTest)
    {
        strmOut << _CurLn_
                << L"Polymorphic read of TDate had wrong contents" << NewLn;
    }
    delete ptmTest;
}


//
// FUNCTION/METHOD NAME: _TestComponents
//
// DESCRIPTION:
//
//  This function is called to test the components of a path string. We test
//  the information in the PathInfo structure to see if it matches the passed
//  path components.
// ---------------------------------------
//   INPUT: strmOut is the stream to output messages on
//          pathTst is the path string to test
//          PathInfo is the result of a call to bParseComponents()
//          pszNode is the node component
//          pszPath is the path component
//          pszName is the name component
//          pszExt is the extension component
//
//  OUTPUT: None
//
//  RETURN: None
//
static
tCIDLib::TVoid __TestComponents(        TTextStream&        strmOut
                                , const TPathStr&           pathTst
                                , const TPathStr::TPathComp& PathInfo
                                ,       tCIDLib::Tch        chDrive
                                , const tCIDLib::Tch*       pszNode
                                , const tCIDLib::Tch*       pszPath
                                , const tCIDLib::Tch*       pszName
                                , const tCIDLib::Tch*       pszExt)
{
    TString  strTmp(kCIDLib::pszEmptyZStr, 256);

    // Test the drive
    if (chDrive != PathInfo.chDrive)
    {
        strmOut << _CurLn_<< L"Drive does not match" << NewLn;
        return;
    }

    // Test the node
    if (PathInfo.c4NodeOfs != kCIDLib::c4MaxCard)
    {
        strTmp.CopyInSubStr
        (
            pathTst
            , PathInfo.c4NodeOfs
            , (PathInfo.c4NodeEnd-PathInfo.c4NodeOfs)+1
        );

        if (strTmp != pszNode)
        {
            strmOut << _CurLn_<< L"Node did not match" << NewLn;
            return;
        }
    }

    // Test the path
    if (PathInfo.c4PathOfs != kCIDLib::c4MaxCard)
    {
        strTmp.CopyInSubStr
        (
            pathTst
            , PathInfo.c4PathOfs
            , (PathInfo.c4PathEnd-PathInfo.c4PathOfs)+1
        );

        if (strTmp != pszPath)
        {
            strmOut << _CurLn_<< L"Path did not match" << NewLn;
            return;
        }
    }

    // Test the name
    if (PathInfo.c4NameOfs != kCIDLib::c4MaxCard)
    {
        strTmp.CopyInSubStr
        (
            pathTst
            , PathInfo.c4NameOfs
            , (PathInfo.c4NameEnd-PathInfo.c4NameOfs)+1
        );

        if (strTmp != pszName)
        {
            strmOut << _CurLn_<< L"Name did not match" << NewLn;
            return;
        }
    }

    // Test the extension
    if (PathInfo.c4ExtOfs != kCIDLib::c4MaxCard)
    {
        strTmp.CopyInSubStr
        (
            pathTst
            , PathInfo.c4ExtOfs
            , (PathInfo.c4ExtEnd-PathInfo.c4ExtOfs)+1
        );

        if (strTmp != pszExt)
        {
            strmOut << _CurLn_<< L"Extension did not match" << NewLn;
            return;
        }
    }
}


static tCIDLib::TVoid __TestDirectories(TTextStream& strmOut)
{
    //
    //  Try to create a directory underneath this directory. Start by
    //  querying the current directory and adding a subdirectory to it.
    //
    TPathStr    pathDir;
    TFileSys::QueryCurrentDir(pathDir);
    pathDir.AddLevel(L"Testing");

    //
    //  If the path exists, then delete it, so that we can create it again
    //  for the test.
    //
    if (TFileSys::bExists(pathDir))
        TFileSys::RemoveDirectory(pathDir);

    // Try to make this directory
    TFileSys::MakeDirectory(pathDir);

    // Now see if it exists
    if (!TFileSys::bIsDirectory(pathDir))
    {
        strmOut << _CurLn_ << L"Just created directory, but couldn't find it"
                << NewLn;
        return;
    }

    //
    //  Now delete the path and test to make sure that it does not exist
    //  any more.
    //
    TFileSys::RemoveDirectory(pathDir);

    if (TFileSys::bExists(pathDir))
    {
        strmOut << _CurLn_ << L"Just deleted directory, buts its still there"
                << NewLn;
        return;
    }
}

static tCIDLib::TVoid __TestDirectorySearch(TTextStream& strmOut)
{
    // This is the file search we do for the test
    const tCIDLib::Tch* const pszWildcard = L"C:\\*.*";

    // Create a directory iterator and a file system object
    TDirIter    diterTest;
    TFileSys    flsysTest;

    //
    //  Do a search of the root of C:, via both of the objects simultaneously.
    //  Then compare the results. They should be identical. The file system
    //  object just gives us back a whole collection at a time, so we do
    //  it first, then use the iterator's first/next scheme to find the same
    //  files.
    //
    //  The root of C: is chosen because it has a good cross section of
    //  files and directories with various attributes.
    //
    TBag<TFindBuf>    colMatches;
    if (!flsysTest.c4SearchDir(pszWildcard, colMatches))
    {
        strmOut << _CurLn_ << L"No files were found in C:\\*.*" << NewLn;
        return;
    }

    TFindBuf    fndbTmp;
    if (!diterTest.bFindFirst(pszWildcard, fndbTmp, tCIDLib::EFileAttr_All))
    {
        strmOut << _CurLn_
                << L"No files found through iterator, though file system did"
                << NewLn;
        return;
    }

    //
    //  Get a cursor for the bag of matches we got via the file system
    //  object. This will let us iterate along the bag as we iterate through
    //  the directory iterator.
    //
    TBag<TFindBuf>::TCursor cursDir(&colMatches);

    //
    //  Now also start a search via the raw Win32 API and make sure that
    //  our stuff matches it too.
    //
    WIN32_FIND_DATA FileData;
    HANDLE hTmp = FindFirstFile(pszWildcard, &FileData);
    if (hTmp == INVALID_HANDLE_VALUE)
    {
        strmOut << _CurLn_
                << L"Could not start search via Win32 API" << NewLn;
        return;
    }

    // Continue searching for the rest of the matches
    tCIDLib::TBoolean   bRanOut = kCIDLib::False;
    tCIDLib::TCard4     c4Count = 0;

    do
    {
        if (bRanOut)
        {
            strmOut << _CurLn_
                    << L"Either the raw Win32 or bag iterator ran out"
                        L" before the directory iterator did" << NewLn;
            break;
        }

        // If the directory and bag iterators are not on the same object
        if (fndbTmp != cursDir.objCur())
        {
            strmOut << _CurLn_
                    << L"Mismatch in directory search" << NewLn
                    << L"    Bag Name:" << cursDir.objCur().pathFileName()
                    << L", Iter Name:" << fndbTmp.pathFileName()
                    << L". LoopCount=" << c4Count << NewLn;
            return;
        }

        //
        //  Compare the raw Win32 find buffer values to our values
        //  in the find buffer.
        //
        TPathStr    pathWin32Name(FileData.cFileName);
        pathWin32Name.Prepend(L"C:\\");
        if (!fndbTmp.pathFileName().bSameText(pathWin32Name))
        {
            strmOut << _CurLn_
                    << L"Mismatch in Win32 directory search" << NewLn
                    << L"  Win32 Name:" << pathWin32Name
                    << L", CIDLib Name:" << fndbTmp.pathFileName()
                    << L". LoopCount=" << c4Count << NewLn;
        }

        tCIDLib::TInt8 i8Size = TRawBits::i8From32
        (
            FileData.nFileSizeLow
             , FileData.nFileSizeHigh
        );

        if (fndbTmp.fposSize() != i8Size)
        {
            strmOut << _CurLn_
                    << L"Mismatch in Win32 directory search" << NewLn
                    << L"  Win32 Size:" << i8Size
                    << L", CIDLib Size:" << fndbTmp.fposSize()
                    << L". LoopCount=" << c4Count << NewLn;
        }

        // Find the next file via Win32
        if (!FindNextFile(hTmp, &FileData))
            bRanOut = kCIDLib::True;

        // Find the next file via the directory cursor
        if (!cursDir.bNext())
            bRanOut = kCIDLib::True;

        // Bump up our loop counter
        c4Count++;
    }   while (diterTest.bFindNext(fndbTmp));

    if (cursDir.bIsValid())
    {
        strmOut << _CurLn_
                << L"Iterator list ran out before bag iterator or Win32"
                    L" search did. Next bag item:"
                << cursDir.objCur().pathFileName()
                << NewLn;
        return;
    }
}


//
// FUNCTION/METHOD NAME: __TestPathStr()
//
// DESCRIPTION:
//
//  This method will test the TPathStr class, which is a specialized string
//  that can manipulate standard OS/2 paths.
// ---------------------------------------
//   INPUT: strmOut is the stream to send messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestPathStr(TTextStream& strmOut)
{
    TPathStr            pathTst;
    TPathStr::TPathComp PathInfo;

    //
    //  The bParseComponents() method is the linch pin of the whole scheme
    //  so we need to check it out closely. It is used to parse out node,
    //  drive, path, name, and extension components of a path.
    //
    pathTst = L"*.*";
    if (!pathTst.bParseComponents(PathInfo))
    {
        strmOut << _CurLn_<< L"Could not parse components" << NewLn;
        return;
    }

    __TestComponents
    (
        strmOut
        , pathTst
        , PathInfo
        , 0
        , kCIDLib::pszEmptyZStr
        , kCIDLib::pszEmptyZStr
        , L"*"
        , L".*"
    );

    pathTst = L"C:\\ABCD\\EFG\\123.456";
    if (!pathTst.bParseComponents(PathInfo))
    {
        strmOut << _CurLn_<< L"Could not parse components" << NewLn;
        return;
    }

    __TestComponents
    (
        strmOut
        , pathTst
        , PathInfo
        , L'C'
        , kCIDLib::pszEmptyZStr
        , L"\\ABCD\\EFG\\"
        , L"123"
        , L".456"
    );

    pathTst = L"\\.";
    if (!pathTst.bParseComponents(PathInfo))
    {
        strmOut << _CurLn_<< L"Could not parse components" << NewLn;
        return;
    }

    __TestComponents
    (
        strmOut
        , pathTst
        , PathInfo
        , 0
        , kCIDLib::pszEmptyZStr
        , L"\\"
        , L"."
        , kCIDLib::pszEmptyZStr
    );

    pathTst = L"\\*.";
    if (!pathTst.bParseComponents(PathInfo))
    {
        strmOut << _CurLn_<< L"Could not parse components" << NewLn;
        return;
    }

    __TestComponents
    (
        strmOut
        , pathTst
        , PathInfo
        , 0
        , kCIDLib::pszEmptyZStr
        , L"\\"
        , L"*"
        , L"."
    );

    pathTst = L"\\\\Wizard\\Develop\\QS2_Dev\\Result\\*.Dll";
    if (!pathTst.bParseComponents(PathInfo))
    {
        strmOut << _CurLn_<< L"Could not parse components" << NewLn;
        return;
    }

    __TestComponents
    (
        strmOut
        , pathTst
        , PathInfo
        , 0
        , L"Wizard"
        , L"\\Develop\\QS2_Dev\\Result\\"
        , L"*"
        , L".Dll"
    );
}



// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestFiles
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the file and file system classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestFiles()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"path string";
        __TestPathStr(strmOut());

        pszCurTest = L"basic file system";
        __TestBasicFileSystem(strmOut());

        pszCurTest = L"binary file";
        __TestBinaryFiles(strmOut());

        pszCurTest = L"directories";
        __TestDirectories(strmOut());

        pszCurTest = L"directory search";
        __TestDirectorySearch(strmOut());

        pszCurTest = L"binary stream";
        __TestBinStreamIO(strmOut());

        // Clean up the temporary file
        pszCurTest = L"Temp file cleanup";
        TFileSys    flsysTmp;
        flsysTmp.DeleteFile(strBinFileName);
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
