//
// NAME: TestCIDLib_Environment.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program. This module is called
//  from the program's main function. The functions in this module test the
//  environment classes.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 08/16/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"



// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------

static tCIDLib::TVoid __BasicProcEnvTest(TTextStream& strmOut)
{
    //
    //  First we test out the process environment class, which works on the
    //  process wide environment.
    //
    //  We start by looking for some values that we know should be there.
    //
    tCIDLib::Tch* apszToFind[] = 
    {
        L"Path"
        , L"CIDDRIVE"
        , L"COMPUTERNAME"
        , L"INCLUDE"
        , L"SystemRoot"
    };
    const tCIDLib::TCard4 c4Count = c4ArrayElems(apszToFind);

    TString strValue;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
    {
        if (!TProcEnvironment::bFind(apszToFind[c4Index], strValue))
        {
            strmOut << _CurLn_ << L"Could not find " << apszToFind[c4Index]
                    << L" in the process environment" << NewLn;
        }
    }

    // And make sure that we don't find some that we don't exists.
    if (TProcEnvironment::bFind("DontExist", strValue))
        strmOut << _CurLn_ << L"Seemed to find var that does not exist" << NewLn;


    //
    //  Test for some operations that should fail if the variable does not
    //  already exists.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        TProcEnvironment::Update(L"DontExist", L"Dummy Value");
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcEnv_VarNotFound))
        {
            strmOut << _CurLn_ << L"Expected VarNotFound exception" << NewLn;
            throw;
        }
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
        strmOut << _CurLn_ << L"Did not get exception on non-existent var" << NewLn;


    //
    //  Now check a couple of test values that we know are in the environment
    //  and make sure that their values are correct.
    //
    if (!TProcEnvironment::bFind(L"TestVar1", strValue))
    {
        strmOut << _CurLn_ << L"Did not find TestVar1 variable" << NewLn;
    }
     else
    {
        if (strValue != L"The value of TestVar1")
            strmOut << _CurLn_ << L"Value of TestVar1 was not correct" << NewLn;
    }

    if (!TProcEnvironment::bFind(L"TestVar2", strValue))
    {
        strmOut << _CurLn_ << L"Did not find TestVar2 variable" << NewLn;
    }
     else
    {
        if (strValue != L"The value of TestVar2")
            strmOut << _CurLn_ << L"Value of TestVar2 was not correct" << NewLn;
    }
}


static tCIDLib::TVoid __ProcEnvTest2(TTextStream& strmOut)
{
    TString strValue;

    //
    //  Add a value and make sure that we can read it back. Get the element
    //  count first, then check that it went up by one.
    //
    const tCIDLib::Tch* const pszDummyKey = L"ADummyKey";
    const tCIDLib::Tch* const pszDummyVal = L"A Dummy Value for Dummy Key";
    tCIDLib::TCard4 c4EnvCount = TProcEnvironment::c4Entries();
    TProcEnvironment::Add(pszDummyKey, pszDummyVal);
    if (TProcEnvironment::c4Entries() != c4EnvCount+1)
        strmOut << _CurLn_ << L"Adding a key did not bump up entry count" << NewLn;

    if (!TProcEnvironment::bFind(pszDummyKey, strValue))
        strmOut << _CurLn_ << L"Did not find a key that was just added" << NewLn;

    //
    //  Now try to add the same element again. This should cause an exception
    //  that the variable already exists.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        TProcEnvironment::Add(pszDummyKey, pszDummyVal);
    }

    catch(const TError& errToCatch)
    {
        if (!errToCatch.bCheckError(facCIDLib, kCIDErrs::errcEnv_VarAlreadyExists))
        {
            strmOut << _CurLn_ << L"Got wrong exception when adding existing key"
                    << NewLn;
            throw;
        }
        bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
        strmOut << _CurLn_ << L"Failed to catch addition of existing key" << NewLn;

    //
    //  Now do an 'add or update' on that key, which should cause the value
    //  to be modified. Make sure that it returns false, which indicates that
    //  it did not add the element, just updated it.
    //
    if (TProcEnvironment::bAddOrUpdate
    (
        pszDummyKey, L"1 Dummy Value for Dummy Key"))
    {
        strmOut << _CurLn_ << L"AddOrUpdate added when it should have updated"
                << NewLn;
    }

    // And make sure that the new value is what it should be
    if (TProcEnvironment::strFind(pszDummyKey) != L"1 Dummy Value for Dummy Key")
    {
        strmOut << _CurLn_ << L"AddOrUpdate did not update value" << NewLn;
    }
}

static tCIDLib::TVoid __BasicEnvTest(TTextStream& strmOut)
{
    tCIDLib::Tch* apszKeys[] = 
    {
        L"Jimi_Hendrix"
        , L"Janis_Joplin"
        , L"Jim_Morrison"
        , L"Mama_Cass"
        , L"John_Lennnon"
    };

    tCIDLib::Tch* apszValues[] = 
    {
        L"Greatest blues rock guitarist"
        , L"A white chick what can sang"
        , L"Prentitious but interesting singer for Doors"
        , L"One of the Mamas and Papas"
        , L"The soul of the beatles"
    };

    const tCIDLib::TCard4 c4Count = c4ArrayElems(apszKeys);


    //
    //  Create an empty environment first and just do some basic testing
    //  on it.
    //
    TEnvironment envTest;

    //
    //  Add a couple of values and keys to it.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4Count; c4Index++)
        envTest.Add(apszKeys[c4Index], apszValues[c4Index]);

    // Check that the entry count is value
    if (envTest.c4Entries() != c4Count)
        strmOut << _CurLn_ << L"The entry count is not correct" << NewLn;

    //
    //  Check that we can find each one and that its value is the same
    //  as what we put in there.
    //
    for (c4Index = 0; c4Index < c4Count; c4Index++)
    {
        if (envTest.strFind(apszKeys[c4Index]) != apszValues[c4Index])
        {
            strmOut << _CurLn_
                    << L"Value was not as originally set for key: "
                    << apszKeys[c4Index] << NewLn;
        }
    }
}

static tCIDLib::TVoid __AdvEnvTest(TTextStream& strmOut)
{
    // Create an environment that inherits the current process environment
    TEnvironment envTest(kCIDLib::True);

    // Make sure it has the same number of entries as the main environment
    if (envTest.c4Entries() != TProcEnvironment::c4Entries())
    {
        strmOut << _CurLn_
                << L"Test env did not inherit same number of entries as process has"
                << NewLn;
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestEnvironment
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the environment classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestEnvironment()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();

        //
        //  Invoke the tests that we want to do memory checking on. These
        //  only do stuff that doesn't cause spurious leak reports.
        //
        {
            TKrnlMemCheck kmchkTest;

            pszCurTest = L"basic process environment";
            __BasicProcEnvTest(strmOut());

            pszCurTest = L"basic environment";
            __BasicEnvTest(strmOut());

            pszCurTest = L"advanced environment";
            __AdvEnvTest(strmOut());
        }

        //
        //  Now do the tests that interfere with the heap checking stuff.
        //
        pszCurTest = L"more process environment";
        __ProcEnvTest2(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Environment tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" environment test" << NewLn;
        throw;
    }
}
