//
// NAME: TestCIDLib_Deques.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  deque collection classes.
//
//  This module brings in a common set of collection tests that are implemented
//  as templates themselves so that they can be invoked for all types of
//  collections without redundancy. But some of the testing is always specific
//  to the particular collection type.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Typedef some instantiations that we want to test
// -----------------------------------------------------------------------------
typedef TDeque<TArea>       TDequeOfTArea;
typedef TDeque<TInteger>    TDequeOfTInteger;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class TDeque<TArea>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
DummyTmplImpls(TDeque<TArea>)



// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid __TestDequeCommon(TTextStream& strmOut)
{
    // Create a bag of cardinal objects. It can hold up to 16 objects
    TDequeOfTArea colTest(16);

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"TDequeOfTArea")))
    {
        strmOut << _CurLn_ << L"Deque class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    // Throw in some area objects
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.PushTop(TArea(c4Ind+1, c4Ind+2, c4Ind+16, c4Ind+17));

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TArea>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, 16);

    //
    //  Do the basic check again but this time with only 1 element, to
    //  check for any boundary conditions.
    //
    colTest.PushBottom(TArea());
    TestColBasics(strmOut, colTest, 1);

    // Throw back some area objects
    for (c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.PushTop(TArea(c4Ind+1, c4Ind+2, c4Ind+16, c4Ind+17));

    // And do the second basic collection test
    TestColBasics2(strmOut, colTest, 16);
}


static tCIDLib::TVoid __TestDequeBasics(TTextStream& strmOut)
{
    //
    //  Create a deques of integer objects. It can hold up to 16 objects
    //
    TDequeOfTInteger deqTest(16);

    // Check the automatic class name generation
    if (!deqTest.bIsA(TClass("TDequeOfTInteger")))
    {
        strmOut << _CurLn_ << L"Deque class name was bad, It was: "
                << deqTest.clsIsA() << NewLn;
    }

    // Push one on the bottom and one the top
    deqTest.PushBottom(1);
    deqTest.PushTop(16);

    // There should be 2 elements in it
    if (deqTest.c4ElemCount() != 2)
        strmOut << _CurLn_
                << L"Deque count was not 2 after pushing two objects" << NewLn;

    // Peek at the too objects and make sure they are correct
    if (deqTest.objPeekBottom() != 1L)
        strmOut << _CurLn_ << L"Peek at bottom of deque was incorrect" << NewLn;

    if (deqTest.objPeekTop() != 16L)
        strmOut << _CurLn_ << L"Peek at top of deque was incorrect" << NewLn;

    // There should still be 2 elements in it
    if (deqTest.c4ElemCount() != 2)
        strmOut << _CurLn_ << L"Deque could was not 2 after peeking" << NewLn;

    // Now pop them off and check their values
    if (deqTest.objPopBottom() != 1L)
        strmOut << _CurLn_ << L"Pop of bottom of deque was incorrect" << NewLn;

    if (deqTest.objPopTop() != 16L)
        strmOut << _CurLn_ << L"Pop at top of deque was incorrect" << NewLn;

    // There should be no elements
    if (deqTest.c4ElemCount())
        strmOut << _CurLn_ << L"Deque was not empty after popping" << NewLn;

    // Test the other way
    if (!deqTest.bIsEmpty())
        strmOut << _CurLn_
                << L"Deque's bIsEmpty is eTrue but elem count was 0" << NewLn;

    // Push a bunch on
    for (tCIDLib::TInt4 i4Ind = 0; i4Ind < 16; i4Ind++)
        deqTest.PushBottom(i4Ind+1);

    // Make sure that there are 16 elements
    if (deqTest.c4ElemCount() != 16)
        strmOut << _CurLn_
                << L"Deque count was not 16 after pushing 16 objects" << NewLn;

    // Pop them the other way and check them
    for (i4Ind = 0; i4Ind < 16; i4Ind++)
    {
        if (deqTest.objPopTop() != i4Ind+1L)
            strmOut << _CurLn_ << L"Deque element not equal to test index" << NewLn;
    }
}

// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestDeques
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the deque collections.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestDeques()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();
        
        // Invoke the bag tests
        pszCurTest = L"common";
        __TestDequeCommon(strmOut());

        pszCurTest = L"basic";
        __TestDequeBasics(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Deque tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" deque test" << NewLn;
        throw;
    }
}
