//
// NAME: TestCIDLib_Coordinates.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the coordinate
//  oriented classes of the facility.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/04/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

static tCIDLib::TBoolean __bCheckArea(  const   TArea&          areaToCheck
                                        , const tCIDLib::TInt4  i4X
                                        , const tCIDLib::TInt4  i4Y
                                        , const tCIDLib::TCard4 c4CX
                                        , const tCIDLib::TCard4 c4CY)
{
    if ((areaToCheck.i4X() != i4X)
    ||  (areaToCheck.i4Y() != i4Y)
    ||  (areaToCheck.c4Width() != c4CX)
    ||  (areaToCheck.c4Height() != c4CY))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}

static tCIDLib::TBoolean __bCheckPoint( const   TPoint&         pntToCheck
                                        , const tCIDLib::TInt4  i4X
                                        , const tCIDLib::TInt4  i4Y)
{
    if ((pntToCheck.i4X() != i4X) || (pntToCheck.i4Y() != i4Y))
        return kCIDLib::False;
    return kCIDLib::True;
}

static tCIDLib::TBoolean __bCheckSize(  const   TSize&          szToCheck
                                        , const tCIDLib::TCard4 c4X
                                        , const tCIDLib::TCard4 c4Y)
{
    if ((szToCheck.c4Width() != c4X) || (szToCheck.c4Height() != c4Y))
        return kCIDLib::False;
    return kCIDLib::True;
}


static tCIDLib::TVoid __ConstructorTests(TTextStream& strmOut)
{
    // First check out the default constructors
    {
        TArea   areaDefault;
        if (!__bCheckArea(areaDefault, 0, 0, 0, 0))
            strmOut << _CurLn_ << L"Area default ctor failed" << NewLn;
    }

    {
        TPoint  pntDefault;
        if (!__bCheckPoint(pntDefault, 0, 0))
            strmOut << _CurLn_ << L"Point default ctor failed" << NewLn;
    }

    {
        TSize   szDefault;
        if (!__bCheckSize(szDefault, 0, 0))
            strmOut << _CurLn_ << L"Size default ctor failed" << NewLn;
    }

    //
    //  Now test all of TPoint's non-default constructors
    //
    {
        TPoint pntTest(15, 16);
        if (!__bCheckPoint(pntTest, 15, 16))
            strmOut << _CurLn_ << L"Point ctor failed" << NewLn;
    }

    {
        TPoint pntTest(75);
        if (!__bCheckPoint(pntTest, 75, 75))
            strmOut << _CurLn_ << L"Point ctor failed" << NewLn;
    }

    {
        TPoint pntTest(TPoint(80, 90));
        if (!__bCheckPoint(pntTest, 80, 90))
            strmOut << _CurLn_ << L"Point ctor failed" << NewLn;
    }

    {
        const tCIDLib::THostPoint HostPnt = { -1, 95 };
        TPoint pntTest(HostPnt);
        if (!__bCheckPoint(pntTest, -1, 95))
            strmOut << _CurLn_ << L"Point ctor failed" << NewLn;
    }

    //
    //  Now test all of TSize's non-default constructors
    //
    {
        TSize szTest(101, 202);
        if (!__bCheckSize(szTest, 101, 202))
            strmOut << _CurLn_ << L"Size ctor failed" << NewLn;
    }

    {
        TSize szTest(44);
        if (!__bCheckSize(szTest, 44, 44))
            strmOut << _CurLn_ << L"Size ctor failed" << NewLn;
    }

    {
        TSize szTest(TSize(999, 606));
        if (!__bCheckSize(szTest, 999, 606))
            strmOut << _CurLn_ << L"Size ctor failed" << NewLn;
    }

    {
        const tCIDLib::THostSize HostSize = { 1024, 2048 };
        TSize szTest(HostSize);
        if (!__bCheckSize(szTest, 1024, 2048))
            strmOut << _CurLn_ << L"Size ctor failed" << NewLn;
    }

    //
    //  Now test all of TArea's non-default constructors
    //
    {
        TArea areaTest(10, 20, 30, 40);
        if (!__bCheckArea(areaTest, 10, 20, 30, 40))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        TArea areaTest(TPoint(50, 60), 30, 40);
        if (!__bCheckArea(areaTest, 50, 60, 30, 40))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        TArea areaTest(TPoint(10, 20), TSize(30, 40));
        if (!__bCheckArea(areaTest, 10, 20, 30, 40))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        tCIDLib::THostPoint HostPnt1 = { 100, 100 };
        tCIDLib::THostPoint HostPnt2 = { 500, 200 };

        TArea areaTest1(HostPnt1, HostPnt2);
        if (!__bCheckArea(areaTest1, 100, 100, 400, 100))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        TArea areaTest(TPoint(10, 20), TPoint(30, 40));
        if (!__bCheckArea(areaTest, 10, 20, 20, 20))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        TArea areaTest(TArea(99, 100, 101, 102));
        if (!__bCheckArea(areaTest, 99, 100, 101, 102))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }

    {
        tCIDLib::THostRectl HostRectl = { 10, 10, 100, 100 };
        TArea areaTest(HostRectl, tCIDLib::ERectl_Inclusive);
        if (!__bCheckArea(areaTest, 10, 10, 91, 91))
            strmOut << _CurLn_ << L"Area ctor failed" << NewLn;
    }
}


static tCIDLib::TVoid __DataAccessTests(TTextStream& strmOut)
{
    //
    //  Test some synonym methods to make sure that they agree.
    //
    {
        const TArea areaTest(11, 12, 13, 14);
        const TSize szTest(50, 60);

        //
        //  Test some of the synonym methods to make sure that they return the
        //  same thing.
        //
        if (areaTest.i4Top() != areaTest.i4Y())
            strmOut << _CurLn_ << L"i4Bottom != i4Y" << NewLn;

        if (areaTest.i4Left() != areaTest.i4X())
            strmOut << _CurLn_ << L"i4Left != i4X" << NewLn;

        if (szTest.c4X() != szTest.c4Width())
            strmOut << _CurLn_ << L"c4X != c4Width" << NewLn;

        if (szTest.c4Y() != szTest.c4Height())
            strmOut << _CurLn_ << L"c4Y != c4Height" << NewLn;
    }

    // Test all of the data access methods of TPoint
    {
        const TPoint pntTest(1111, 2222);

        if (!pntTest.bInArea(TArea(1110, 2220, 8, 8)))
            strmOut << _CurLn_ << L"Point should have beenin area" << NewLn;

        if (pntTest.c4Packed() != TRawBits::c4From16(1111, 2222))
            strmOut << _CurLn_ << L"Packed point was not correct" << NewLn;

        if (pntTest.i4X() != 1111)
            strmOut << _CurLn_ << L"Point X value was incorrect" << NewLn;

        if (pntTest.i4Y() != 2222)
            strmOut << _CurLn_ << L"Point Y value was incorrect" << NewLn;
    }

    // Test all of the data access methods of TSize
    {
        const TSize szTest(64, 32);

        if (szTest.c4SquareUnits() != 64 * 32)
            strmOut << _CurLn_ << L"Size square units calc failed" << NewLn;

        if (szTest.c4Packed() != TRawBits::c4From16(64, 32))
            strmOut << _CurLn_ << L"Packed size was incorrect" << NewLn;

        if ((szTest.c4X() != 64) || (szTest.c4Width() != 64))
            strmOut << _CurLn_ << L"Size cx was incorrect" << NewLn;

        if ((szTest.c4Y() != 32) || (szTest.c4Height() != 32))
            strmOut << _CurLn_ << L"Size cy was incorrect" << NewLn;
    }

    // Test out all of the access methods of TArea
    {
        const TArea areaTest(128, 32, 64, 256);

        if (!__bCheckArea(areaTest, 128, 32, 64, 256))
            strmOut << _CurLn_ << L"Initial area contents incorrect" << NewLn;

        if (!areaTest.bContainsPoint(TPoint(132, 68)))
            strmOut << _CurLn_ << L"Area should have contained point" << NewLn;

        if (!areaTest.bContainsPoint(132, 68))
            strmOut << _CurLn_ << L"Area should have contained point" << NewLn;

        if (areaTest.bEmpty())
            strmOut << _CurLn_ << L"Area should not have been empty" << NewLn;

        if (areaTest.bIntersects(TArea(0, 0, 32, 32)))
            strmOut << _CurLn_ << L"Areas should not have intersected" << NewLn;

        if (areaTest.bIntersects(TArea(0, 0, 33, 33)))
            strmOut << _CurLn_ << L"Areas should have intersected" << NewLn;

        tCIDLib::TFloat4 f4PercentX;
        tCIDLib::TFloat4 f4PercentY;
        areaTest.bPercentFromOrg
        (
            TPoint(128 + 32, 32 + 128)
            , f4PercentX, f4PercentY
        );
        if ((f4PercentX != 0.5) || (f4PercentY != 0.5))
            strmOut << _CurLn_ << L"Area percent from org calc failed" << NewLn;

        if (!areaTest.bSameSize(TArea(0, 0, 64, 256)))
            strmOut << _CurLn_ << L"Area same size test failed" << NewLn;

        // Check the calculation of the bottom and right sides
        if (areaTest.i4Right() != 191)
            strmOut << _CurLn_ << L"i4Right() returned incorrect value" << NewLn;

        if (areaTest.i4Bottom() != 287)
            strmOut << _CurLn_ << L"i4Bottom() returned incorrect value" << NewLn;

        if (areaTest.pntCenter() != TPoint(128 + 32, 32 + 128))
            strmOut << _CurLn_ << L"Area center point was incorrect" << NewLn;

        if (areaTest.pntOrg() != TPoint(128, 32))
            strmOut << _CurLn_ << L"Area origin was incorrect" << NewLn;

        if (areaTest.szSize() != TSize(64, 256))
            strmOut << _CurLn_ << L"Area size was incorrect" << NewLn;

    }

    // Test the point data modification methods
    {
        TPoint pntTest(64, 96);

        pntTest.Adjust(1, 1);
        if (!__bCheckPoint(pntTest, 65, 97))
            strmOut << _CurLn_ << L"Point adjust failed" << NewLn;

        pntTest.Adjust(-2, -2);
        if (!__bCheckPoint(pntTest, 63, 95))
            strmOut << _CurLn_ << L"Point adjust failed" << NewLn;

        pntTest.AdjustX(6);
        pntTest.AdjustY(-6);
        if (!__bCheckPoint(pntTest, 69, 89))
            strmOut << _CurLn_ << L"Point adjust failed" << NewLn;

        pntTest.Exchange();
        if (!__bCheckPoint(pntTest, 89, 69))
            strmOut << _CurLn_ << L"Point adjust failed" << NewLn;

        pntTest.Negate();
        if (!__bCheckPoint(pntTest, -89, -69))
            strmOut << _CurLn_ << L"Point negation failed" << NewLn;

        pntTest.NegateX();
        pntTest.NegateY();
        if (!__bCheckPoint(pntTest, 89, 69))
            strmOut << _CurLn_ << L"Point negation failed" << NewLn;

        pntTest.Adjust(1, 1);
        pntTest.Scale(0.5, 0.5);
        if (!__bCheckPoint(pntTest, 45, 35))
            strmOut << _CurLn_ << L"Point scale failed" << NewLn;

        pntTest.Set(55, 100);
        if (!__bCheckPoint(pntTest, 55, 100))
            strmOut << _CurLn_ << L"Point set failed" << NewLn;

        // Now do the indivdual sets
        pntTest.i4X(9);
        pntTest.i4Y(10);
        if (pntTest != TPoint(9, 10))
            strmOut << _CurLn_ << L"Point member sets failed" << NewLn;

        pntTest.Zero();
        if (!pntTest.bAtOrg())
            strmOut << _CurLn_ << L"Point zero out failed" << NewLn;
    }

    // Test the size data modification methods
    {
        TSize szTest(50, 60);

        szTest.Adjust(1, 1);
        if (!__bCheckSize(szTest, 51, 61))
            strmOut << _CurLn_ << L"Size adjust failed" << NewLn;

        szTest.Adjust(-1, -1);
        if (!__bCheckSize(szTest, 50, 60))
            strmOut << _CurLn_ << L"Size adjust failed" << NewLn;

        szTest.Adjust(1);
        if (!__bCheckSize(szTest, 51, 61))
            strmOut << _CurLn_ << L"Size adjust failed" << NewLn;

        szTest.Adjust(-1);
        if (!__bCheckSize(szTest, 50, 60))
            strmOut << _CurLn_ << L"Size adjust failed" << NewLn;

        szTest.AdjustCX(-2);
        szTest.AdjustCY(2);
        if (!__bCheckSize(szTest, 48, 62))
            strmOut << _CurLn_ << L"Size field adjust failed" << NewLn;

        szTest.Exchange();
        if (!__bCheckSize(szTest, 62, 48))
            strmOut << _CurLn_ << L"Size exchange failed" << NewLn;

        szTest.Set(90, 180);
        if (!__bCheckSize(szTest, 90, 180))
            strmOut << _CurLn_ << L"Size set failed" << NewLn;

        szTest.c4Width(180);
        szTest.c4Height(90);
        if (!__bCheckSize(szTest, 180, 90))
            strmOut << _CurLn_ << L"Size member set failed" << NewLn;

        szTest.Zero();
        if (!szTest.bAllZero())
            strmOut << _CurLn_ << L"Size zero out failed" << NewLn;
    }

    // Test the area data modification methods
    {
        TArea areaTest(64, 128, 256, 512);

        // Adjust the origin via different methods
        areaTest.AdjustOrg(-32, -64);
        if (!__bCheckArea(areaTest, 32, 64, 256, 512))
            strmOut << _CurLn_ << L"Area origin adjust failed" << NewLn;

        areaTest.AdjustOrg(TPoint(32, 64));
        if (!__bCheckArea(areaTest, 64, 128, 256, 512))
            strmOut << _CurLn_ << L"Area origin adjust failed" << NewLn;

        // Adjust the size via different methods
        areaTest.AdjustSize(256, 512);
        if (!__bCheckArea(areaTest, 64, 128, 512, 1024))
            strmOut << _CurLn_ << L"Area size adjust failed" << NewLn;

        areaTest.AdjustSize(TPoint(-256, -512));
        if (!__bCheckArea(areaTest, 64, 128, 256, 512))
            strmOut << _CurLn_ << L"Area size adjust failed" << NewLn;

        // Adjust the sides
        areaTest.AdjustSides(TPoint(1, 1));
        if (!__bCheckArea(areaTest, 63, 127, 258, 514))
            strmOut << _CurLn_ << L"Area side adjust failed" << NewLn;

        // Force it within another area
        areaTest.ForceWithin(TArea(64, 128, 256, 512), kCIDLib::True);
        if (!__bCheckArea(areaTest, 64, 128, 256, 512))
            strmOut << _CurLn_ << L"Area force within failed" << NewLn;

        // Inflate and deflate and test
        areaTest.Inflate(1, 1);
        if (!__bCheckArea(areaTest, 63, 127, 258, 514))
            strmOut << _CurLn_ << L"Area inflate failed" << NewLn;

        areaTest.Deflate(1, 1);
        if (!__bCheckArea(areaTest, 64, 128, 256, 512))
            strmOut << _CurLn_ << L"Area deflate failed" << NewLn;

        areaTest.Inflate(1);
        if (!__bCheckArea(areaTest, 63, 127, 258, 514))
            strmOut << _CurLn_ << L"Area inflate failed" << NewLn;

        areaTest.Deflate(1);
        if (!__bCheckArea(areaTest, 64, 128, 256, 512))
            strmOut << _CurLn_ << L"Area deflate failed" << NewLn;

        // Justify the area within a larger area, using a couple justifications
        areaTest.JustifyIn
        (
            TArea(32, 32, 512, 1024)
            , tCIDLib::EHJustify_Left
            , tCIDLib::EVJustify_Top
        );
        if (!__bCheckArea(areaTest, 32, 32, 256, 512))
            strmOut << _CurLn_ << L"Area justify in failed" << NewLn;

        areaTest.JustifyIn
        (
            TArea(32, 32, 512, 1024)
            , tCIDLib::EHJustify_Right
            , tCIDLib::EVJustify_Bottom
        );
        if (!__bCheckArea(areaTest, 287, 543, 256, 512))
            strmOut << _CurLn_ << L"Area justify in failed" << NewLn;

        // Negate the origin
        areaTest.NegateOrg();
        if (!__bCheckArea(areaTest, -287, -543, 256, 512))
            strmOut << _CurLn_ << L"Area justify in failed" << NewLn;

        // Scale the size
        areaTest.ScaleSize(0.5, 0.5);
        if (!__bCheckArea(areaTest, -287, -543, 128, 256))
            strmOut << _CurLn_ << L"Area justify in failed" << NewLn;

        // Set the area via the Set method
        areaTest.Set(1, 1, 5, 5);
        if (!__bCheckArea(areaTest, 1, 1, 5, 5))
            strmOut << _CurLn_ << L"Area set failed" << NewLn;

        // Set just the size
        areaTest.SetSizes(10, 10);
        if (!__bCheckArea(areaTest, 1, 1, 10, 10))
            strmOut << _CurLn_ << L"Area set size failed" << NewLn;

        // Do the various zero methods
        areaTest.ZeroOrg();
        if (!__bCheckArea(areaTest, 0, 0, 10, 10))
            strmOut << _CurLn_ << L"Area zero org failed" << NewLn;

        areaTest.ZeroSizes();
        if (!__bCheckArea(areaTest, 0, 0, 0, 0))
            strmOut << _CurLn_ << L"Area zero size failed" << NewLn;

        areaTest = TArea(1, 2, 3, 4);
        areaTest.ZeroAll();
        if (!__bCheckArea(areaTest, 0, 0, 0, 0))
            strmOut << _CurLn_ << L"Area zero all failed" << NewLn;

    }

    // Test all of the conversions to and from host structures
    {
        // Convert a point to a host point
        TPoint pntTest(1111, 2222);
        tCIDLib::THostPoint HostPoint;
        pntTest.ToHostPt(HostPoint);
        if ((HostPoint.i4X != 1111) || (HostPoint.i4Y != 2222))
            strmOut << _CurLn_ << L"Conversion to host point failed" << NewLn;

        // Create another point from the host point and compare to original
        TPoint pntTest2(HostPoint);
        if (pntTest != pntTest2)
            strmOut << _CurLn_ << L"Conversion back from host point failed" << NewLn;

        TSize szTest(64, 32);
        tCIDLib::THostSize HostSize;
        szTest.ToHostSize(HostSize);
        if ((HostSize.i4CX != 64) || (HostSize.i4CY != 32))
            strmOut << _CurLn_ << L"Conversion to host size failed" << NewLn;

        // Create another size and convert back and compare to original
        TSize szTest2(HostSize);
        if (szTest != szTest2)
            strmOut << _CurLn_ << L"Conversion back from host size failed" << NewLn;

        // Convert an area to a host rect
        TArea areaTest(10, 10, 20, 30);
        tCIDLib::THostRectl HostRectl;

        areaTest.ToRectl(HostRectl, tCIDLib::ERectl_Inclusive);
        if ((HostRectl.i4Left != 10)
        ||  (HostRectl.i4Top != 10)
        ||  (HostRectl.i4Right != 29)
        ||  (HostRectl.i4Bottom != 39))
        {
            strmOut << _CurLn_ << L"Convert to host rectl failed" << NewLn;
        }

        // Construct another area from it and compare to original
        TArea areaTest2(HostRectl, tCIDLib::ERectl_Inclusive);
        if (areaTest2 != areaTest)
            strmOut << _CurLn_ << L"Conversion back from rectl failed" << NewLn;

        areaTest.ToRectl(HostRectl, tCIDLib::ERectl_NonInclusive);
        if ((HostRectl.i4Left != 10)
        ||  (HostRectl.i4Top != 10)
        ||  (HostRectl.i4Right != 30)
        ||  (HostRectl.i4Bottom != 40))
        {
            strmOut << _CurLn_ << L"Convert to host rectl failed" << NewLn;
        }

        // Convert back from the rectangle
        areaTest2.FromRectl(HostRectl, tCIDLib::ERectl_NonInclusive);
        if (areaTest2 != areaTest)
            strmOut << _CurLn_ << L"Conversion back from rectl failed" << NewLn;
    }
}


static tCIDLib::TVoid __OperatorTests(TTextStream& strmOut)
{
    // Test TSize operators
    {
        TSize szTest1(100, 200);
        TSize szTest2;

        // Do an assignment and test for equality
        szTest2 = szTest1;
        if (szTest1 != szTest2)
            strmOut << _CurLn_ << L"Assignment of TSize object != to original" << NewLn;

        // Some some addition and subtraction and test results
        if (TSize(1, 1) + TSize(10, 10) != TSize(11, 11))
            strmOut << _CurLn_ << L"Addition of two size objects failed" << NewLn;

        if (TSize(10, 10) - TSize(5, 5) != TSize(5, 5))
            strmOut << _CurLn_ << L"Subtraction of two size objects failed" << NewLn;

        szTest2 -= TSize(50, 50);
        if (szTest2 == szTest1)
            strmOut << _CurLn_ << L"Modified size object still == to original" << NewLn;

        if (szTest2 != TSize(50, 150))
            strmOut << _CurLn_ << L"Subtraction of size failed" << NewLn;

        szTest1 += TSize(60, 40);
        if (szTest1 != TSize(160, 240))
            strmOut << _CurLn_ << L"Addition of size failed" << NewLn;
    }

    // Test TPoint operators
    {
        TPoint pntTest1(10, 11);
        TPoint pntTest2;

        // Do an assignment and test for equality
        pntTest2 = pntTest1;
        if (pntTest1 != pntTest2)
            strmOut << _CurLn_ << L"Assignment of TPoint object != to original" << NewLn;

        // Add and subtract them
        pntTest1 += pntTest2;
        if (pntTest1 != TPoint(20, 22))
            strmOut << _CurLn_ << L"Point addition failed" << NewLn;

        pntTest1 -= pntTest2;
        if (pntTest1 != TPoint(10, 11))
            strmOut << _CurLn_ << L"Point subtraction failed" << NewLn;

        if ((pntTest1 + pntTest2) != TPoint(20, 22))
            strmOut << _CurLn_ << L"Point addition failed" << NewLn;

        if ((pntTest1 - pntTest2) != TPoint(0, 0))
            strmOut << _CurLn_ << L"Point addition failed" << NewLn;
    }

    // Test TArea operators
    {
        TArea areaTest1(1, 2, 10, 11);
        TArea areaTest2;

        // Do an assignment and test for equality
        areaTest2 = areaTest1;
        if (areaTest1 != areaTest2)
            strmOut << _CurLn_ << L"Assignment of TArea object != to original" << NewLn;

        //
        //  Do all of the logical operations and insure that they create the
        //  correct results.
        //
        if ((TArea(10, 10, 20, 20) | TArea(15, 15, 25, 25)) != TArea(10, 10, 30, 30))
            strmOut << _CurLn_ << L"Logical OR of two areas failed" << NewLn;

        if ((TArea(10, 10, 20, 20) & TArea(15, 15, 25, 25)) != TArea(15, 15, 15, 15))
            strmOut << _CurLn_ << L"Logical AND of two areas failed" << NewLn;

        areaTest1.Set(10, 10, 20, 20);
        areaTest1 |= TArea(15, 15, 25, 25);
        if (areaTest1 != TArea(10, 10, 30, 30))
            strmOut << _CurLn_ << L"Logical OR of area failed" << NewLn;

        areaTest1.Set(10, 10, 20, 20);
        areaTest1 &= TArea(15, 15, 25, 25);
        if (areaTest1 != TArea(15, 15, 15, 15))
            strmOut << _CurLn_ << L"Logical AND of area failed" << NewLn;
    }
}


static tCIDLib::TVoid __TestRange(TTextStream& strmOut)
{
    TRange  rngTest(5, 0, 0, 99);

    if (rngTest.c4FullRange() != 100)
        strmOut << _CurLn_ << L"Full range of range object was not 100";

    if (rngTest.c4CurPercent() != 0)
        strmOut << _CurLn_ << L"Current percent of range object is not 0";

    rngTest.i4CurValue(50);
    if (rngTest.c4CurPercent() != 50)
        strmOut << _CurLn_ << L"Current percent of range object is not 50";

    rngTest.i4CurValue(95);
    if (rngTest.c4CurPercent() != 95)
        strmOut << _CurLn_ << L"Current percent of range object is not 95";

    if (rngTest.i4MaxLegalValue() != 95)
        strmOut << _CurLn_ << L"Max legal value of range is not 95";

    if (rngTest.c4CalcPercent(75) != 75)
        strmOut << _CurLn_ << L"Calc'd percent of range is not 75";

    // Set it all differently now
    rngTest.SetAll(995, 5, 0, 999);

    if (rngTest.c4FullRange() != 1000)
        strmOut << _CurLn_ << L"Full range of range object was not 1000";

    if (rngTest.i4MaxLegalValue() != 5)
        strmOut << _CurLn_ << L"Max legal value of range is not 5";

    // Bump up the max value
    rngTest.i4MaxValue(9999);

    if (rngTest.i4MaxLegalValue() != 9005)
        strmOut << _CurLn_ << L"Max legal value of range is not 9005";

    // Create another with the known current values and compare
    TRange  rngTest2(995, 5, 0, 9999);

    if (rngTest != rngTest2)
        strmOut << _CurLn_ << L"Range objects did not equate properly";

    // Copy construct another from the first and compare
    TRange  rngTest3(rngTest);

    if (rngTest != rngTest3)
        strmOut << _CurLn_ << L"Range objects did not equate properly";
}


static tCIDLib::TVoid __UnderOverFlow(TTextStream& strmOut)
{
    tCIDLib::TBoolean   bCaughtIt;
    tCIDLib::TCard4     c4Test = 1;
    while (c4Test)
    {
        const tCIDLib::Tch* pszTest = L"Uknown";

        bCaughtIt = kCIDLib::False;
        try
        {
            if (c4Test == 1)
            {
                pszTest = L"adding two points";
                TPoint pntTest(kCIDLib::i4MaxInt-2, kCIDLib::i4MaxInt-2);
                pntTest += TPoint(16, 16);
            }
             else if (c4Test == 2)
            {
                pszTest = L"adding two points";
                TPoint pntTest(kCIDLib::i4MaxInt-2, kCIDLib::i4MaxInt-2);
                TPoint pntTmp = pntTest + TPoint(16, 16);
            }   
             else if (c4Test == 3)
            {
                pszTest = L"adding two sizes";
                TSize szTest(kCIDLib::c4MaxCard, kCIDLib::c4MaxCard);
                szTest += TSize(1, 1);
            }
             else if (c4Test == 4)
            {
                pszTest = L"adding two sizes";
                TSize szTest(kCIDLib::c4MaxCard, kCIDLib::c4MaxCard);
                TSize szTmp = szTest + TSize(1, 1);
            }
             else if (c4Test == 5)
            {
                pszTest = L"subtracting two points";
                TPoint pntTest(kCIDLib::i4MinInt, kCIDLib::i4MinInt);
                pntTest -= TPoint(16, 16);
            }
             else if (c4Test == 6)
            {
                pszTest = L"subtracting two points";
                TPoint pntTest(kCIDLib::i4MinInt, kCIDLib::i4MinInt);
                TPoint pntTmp = pntTest - TPoint(16, 16);
            }
             else if (c4Test == 7)
            {
                pszTest = L"subtracting two sizes";
                TSize szTest(0, 0);
                szTest -= TSize(1, 1);
            }
             else if (c4Test == 8)
            {
                pszTest = L"subtracting two sizes";
                TSize szTest(0, 0);
                TSize szTmp = szTest - TSize(1,1);
            }
             else if (c4Test == 9)
            {
                pszTest = L"Adjust size CX";
                TSize szTest(0, 0);
                szTest.AdjustCX(-1);
            }
             else if (c4Test == 10)
            {
                pszTest = L"Adjust size CY";
                TSize szTest(0, 0);
                szTest.AdjustCY(-1);
            }
             else if (c4Test == 11)
            {
                pszTest = L"Adjust size fields";
                TSize szTest(0, 0);
                szTest.Adjust(-1, -1);
            }
             else if (c4Test == 12)
            {
                pszTest = L"Adjust point X";
                TPoint pntTest(kCIDLib::i4MaxInt);
                pntTest.AdjustX(1);
            }
             else if (c4Test == 13)
            {
                pszTest = L"Adjust point Y";
                TPoint pntTest(kCIDLib::i4MaxInt);
                pntTest.AdjustY(1);
            }
             else if (c4Test == 14)
            {
                pszTest = L"Adjust point fields";
                TPoint pntTest(kCIDLib::i4MinInt);
                pntTest.Adjust(-1, -1);
            }
             else if (c4Test == 15)
            {
                pszTest = L"adjusting area origin";
                TArea areaTest(kCIDLib::i4MinInt, kCIDLib::i4MinInt, 10, 10);
                areaTest.AdjustOrg(-1, -1);
            }
             else
            {
                break;
            }
        }

        catch(...)
        {
            bCaughtIt = kCIDLib::True;
        }

        if (!bCaughtIt)
        {
            strmOut << _CurLn_ << L"Failed to catch over/under flow when "
                    << pszTest << NewLn;
        }

        c4Test++;
    }
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestCoordinates
//
// DESCRIPTION:
//
//  This method calls a couple local functions that test various coordinate
//  oriented classes that don't warrant a specific test method of their own.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestCoordinates()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"Ctor tests";
        __ConstructorTests(strmOut());

        pszCurTest = L"Data access tests";
        __DataAccessTests(strmOut());

        pszCurTest = L"Over/Underflow";
        __UnderOverFlow(strmOut());

        pszCurTest = L"operator";
        __OperatorTests(strmOut());

        pszCurTest = L"range class";
        __TestRange(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
