//
// NAME: TestCIDLib_CommonCollection.Hpp
//
// DESCRIPTION:
//
//  This header provides some templatized tests that can be instantiated for
//  any kind of collection. This allows us to apply these tests to all
//  collection types while minimizing redundancy of code. All of the collection
//  testing modules also have some collection specific code, but this common
//  stuff helps minimize the work a lot.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// ----------------------------------------------------------------------------
//  This macro provides dummy implementations for some constructors and
//  operators that are not implemented in the constructors. When we force an
//  explicit instantiation, to test for errors, we get warnings if these are
//  not implemented. So, this is used in the test modules to provide dummy
//  implementations for them. This insures that any really missing methods
//  still get reported, but that these don't.
// ----------------------------------------------------------------------------
#define DummyTmplImpls(type) \
tCIDLib::TVoid type::operator=(const type&) {}

#define DummyTmplImpls2(type) \
type::type() {} \
tCIDLib::TVoid type::operator=(const type&) {}



// ----------------------------------------------------------------------------
//  This template does some very basic collection tests that pretty much any
//  collection must support.
// ----------------------------------------------------------------------------
template <class TCol> tCIDLib::TVoid
TestColBasics(          TTextStream&    strmOut
                ,       TCol&           colToTest
                , const tCIDLib::TCard4 c4InitialCount)
{
    // Make sure we didn't get called with 0 initial count
    if (!c4InitialCount)
    {
        strmOut << _CurLn_ << L"Cannot have initial count of 0 for test!" << NewLn;
        return;
    }

    // Get the reported element count
    tCIDLib::TCard4 c4ElemCount = colToTest.c4ElemCount();

    if (c4ElemCount != c4InitialCount)
    {
        strmOut << _CurLn_ << L"Collection reports " << c4ElemCount
                << L" elements, but initial count said " << c4InitialCount
                << NewLn;
        return;
    }

    // Get a cursor and check the elements
    TCol::TCursor cursTest(&colToTest);

    if (cursTest.c4ElemCount() != c4InitialCount)
        strmOut << _CurLn_ << L"Element count not correct via cursor" << NewLn;

    // If elements are in the collection, then internal iterator reset must work
    if (!colToTest.bResetIter())
    {
        strmOut << _CurLn_ << L"Element count was != 0 but bResetIter failed!" << NewLn;
        return;
    }

    // Now internal iterate and count the elements
    tCIDLib::TCard4 c4Count = 0;
    do
    {
        c4Count++;
    }   while (colToTest.bNext());

    if (c4Count != c4ElemCount)
    {
        strmOut << _CurLn_ << L"Internal iterated count != element count" << NewLn;
        return;
    }

    // Create another cursor to test invalidation
    TCol::TCursor cursTest2(&colToTest);

    //
    //  Delete the first element. We have to reset the cursor, which leaves it
    //  at the first element, then tell the collection to delete that value.
    //
    cursTest.bReset();
    colToTest.FlushAt(cursTest);

    //
    //  The passed cursor should have stayed valid because it was adjusted
    //  But the other one should not be now.
    //
    if (!cursTest.bIsValid() && !colToTest.bIsEmpty())
        strmOut << _CurLn_ << L"Cursor was incorrectly invalidated" << NewLn;

    if (cursTest2.bIsValid())
        strmOut << _CurLn_ << L"Cursor should have been invalidated" << NewLn;

    // There should be one less items in the bag now
    if (colToTest.c4ElemCount() != c4InitialCount-1)
        strmOut << _CurLn_ << L"Element count not correct after flush of element" << NewLn;

    // There should be one less object via the cursor also
    if (cursTest.c4ElemCount() != c4InitialCount-1)
        strmOut << _CurLn_ << L"Element count not correct via cursor after flush of element" << NewLn;

    // Flush the objects all out of the collection
    colToTest.Flush();

    // Make sure the element count is still right
    if (colToTest.c4ElemCount() != 0)
        strmOut << _CurLn_ << L"Flush did not zero element count" << NewLn;

    // Check it the other way
    if (!colToTest.bIsEmpty())
        strmOut << _CurLn_ << L"bIsEmpty is false but there are no elements" << NewLn;
}


// ----------------------------------------------------------------------------
//  This template does some basic testing that things stay correct while
//  removing elements while iterating via the internal iterator.
// ----------------------------------------------------------------------------
template <class TCol> tCIDLib::TVoid
TestColBasics2(         TTextStream&    strmOut
                ,       TCol&           colToTest
                , const tCIDLib::TCard4 c4InitialCount)
{
    //
    //  Loop and flush the current element until we are empty. On each one
    //  test that the count is correct and that the bIsEmpty() is correct
    //  and that the flush returns the correct status.
    //
    if (!colToTest.bResetIter())
    {
        strmOut << _CurLn_ << L"The collection was empty" << NewLn;
        return;
    }

    tCIDLib::TCard4 c4CurCount = colToTest.c4ElemCount();
    while (1)
    {
        if (!colToTest.bFlushCur())
        {
            if (c4CurCount != 1)
            {
                strmOut << _CurLn_
                        << L"bFlushCur() returned false before count was 1"
                        << NewLn;
            }
            return;
        }

        // Make sure that the count went down by one
        if (colToTest.c4ElemCount() != c4CurCount-1)
        {
            strmOut << _CurLn_ << L"bFlushCur() did not bump the count correctly"
                    << NewLn;
            return;
        }

        // Make sure it does not show empty
        if (colToTest.bIsEmpty())
        {
            strmOut << _CurLn_ << L"Collection shows empty but is should not"
                    << NewLn;
            return;
        }

        c4CurCount = colToTest.c4ElemCount();
    }

    // bNext() should return false here
    if (colToTest.bNext())
    {
        strmOut << _CurLn_ << L"The supposedly empty collection returned "
                              L" true for bNext()" << NewLn;
    }
}


// ----------------------------------------------------------------------------
//  This template does some collection copying and uplication tests that all
//  the collections support via global template methods.
// ----------------------------------------------------------------------------
template <class TCol, class TEq> tCIDLib::TVoid
TestColCopy(        TTextStream&    strmOut
            , const TCol&           colToTest
            , const TEq&            compToUse)
{
    // Duplicate the collection and put a janitor on it
    TCol*           pcolNew = ::pDupCollection(colToTest);
    TJanitor<TCol>  janCol(pcolNew);

    // Make sure that they are equal
    if (!bCompCollections(colToTest, *pcolNew, compToUse))
    {
        strmOut << _CurLn_
                << L"The duplicated collection is not equal to original"
                << NewLn;
    }

    // Flush the new collection
    pcolNew->Flush();

    // Now copy the nodes of the original collection
    CopyCollectionNodes(*pcolNew, colToTest);

    // Make sure that they are equal
    if (!bCompCollections(colToTest, *pcolNew, compToUse))
    {
        strmOut << _CurLn_
                << L"The copied collection is not equal to original"
                << NewLn;
    }
}
