//
// NAME: TestCIDLib_BinaryStreams.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  binary stream classes.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/29/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
//  This method provides some common tests for binary streams. Streams are
//  created with various implementation objects and passed here for testing.
//  These should all create the same results.
//
static tCIDLib::TVoid
__CommonBinTests(           TTextStream&        strmOut
                    ,       TBinaryStream&      strmToTest
                    , const tCIDLib::TFilePos   fposCurLogical
                    , const tCIDLib::TFilePos   fposPhysical)
{
    // See if the stream shows the correct logical and physical ends
    if (strmToTest.fposLogicalEnd() != fposCurLogical)
    {
        strmOut << _CurLn_ << L"Initial logical end was "
                << strmToTest.fposLogicalEnd() << L" but should have been "
                << fposCurLogical << NewLn;
        return;
    }

    if (strmToTest.fposPhysicalEnd() != fposPhysical)
    {
        strmOut << _CurLn_ << L"Physical end was "
                << strmToTest.fposPhysicalEnd() << L" but should have been "
                << fposPhysical << NewLn;
        return;
    }

    // Reset the stream and make sure the position goes to zero
    strmToTest.Reset();
    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Reset did not result in position of zero"
                << NewLn;
        return;
    }

    //
    //  The caller insures that there is no initial data in the stream
    //  so we should have an 'end of stream' state now.
    //
    if (!strmToTest.bEndOfStream())
    {
        strmOut << _CurLn_ << L"Initial stream is not at end of data" << NewLn;
        return;
    }

    // Start tracking how much data we've written
    tCIDLib::TFilePos fposCount = 0;

    //
    //  Ok, so lets write out some data and test that the stream position
    //  moves up correctly.
    //
    strmToTest << tCIDLib::TCard1(1);
    fposCount += sizeof(tCIDLib::TCard1);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard1 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TCard2(2);
    fposCount += sizeof(tCIDLib::TCard2);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard2 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TCard4(4);
    fposCount += sizeof(tCIDLib::TCard4);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard4 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TInt1(-1);
    fposCount += sizeof(tCIDLib::TInt1);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt1 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TInt2(-2);
    fposCount += sizeof(tCIDLib::TInt2);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt2 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TInt4(-4);
    fposCount += sizeof(tCIDLib::TInt4);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt4 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    //
    //  Since we are writing new data, we should still be at the end of
    //  stream anywhere during any of these writes.
    //
    if (!strmToTest.bEndOfStream())
    {
        strmOut << _CurLn_ << L"Stream is not at end of data after writes"
                << NewLn;
        return;
    }

    strmToTest << tCIDLib::TFloat4(4.0);
    fposCount += sizeof(tCIDLib::TFloat4);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TFloat4 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest << tCIDLib::TFloat8(8.0);
    fposCount += sizeof(tCIDLib::TFloat8);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TFloat8 value did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    //
    //  Now lets write out some arrays of values and make sure that they
    //  all do right. This will be enough data to force any incrementally
    //  committing type of stream sink/source, which is important to test.
    //
    tCIDLib::TCard1     ac1Test[256];
    tCIDLib::TCard2     ac2Test[128];
    tCIDLib::TCard4     ac4Test[128];

    tCIDLib::TInt1      ai1Test[256];
    tCIDLib::TInt2      ai2Test[128];
    tCIDLib::TInt4      ai4Test[128];

    tCIDLib::TFloat4    af4Test[256];
    tCIDLib::TFloat8    af8Test[256];

    for (tCIDLib::TCard4 c4Index = 0; c4Index < 256; c4Index++)
        ac1Test[c4Index] = tCIDLib::TCard1(c4Index);

    for (c4Index = 0; c4Index < 128; c4Index++)
        ac2Test[c4Index] = tCIDLib::TCard2(c4Index);

    for (c4Index = 0; c4Index < 128; c4Index++)
        ac4Test[c4Index] = c4Index;

    for (c4Index = 0; c4Index < 256; c4Index++)
        ai1Test[c4Index] = tCIDLib::TInt1(c4Index);

    for (c4Index = 0; c4Index < 128; c4Index++)
        ai2Test[c4Index] = tCIDLib::TInt2(c4Index);

    for (c4Index = 0; c4Index < 128; c4Index++)
        ai4Test[c4Index] = c4Index;

    for (c4Index = 0; c4Index < 256; c4Index++)
        af4Test[c4Index] = tCIDLib::TFloat4(c4Index);

    for (c4Index = 0; c4Index < 256; c4Index++)
        af8Test[c4Index] = tCIDLib::TFloat8(c4Index);

    strmToTest.WriteCard1Array(ac1Test, 256);
    fposCount += sizeof(ac1Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard1 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteCard2Array(ac2Test, 128);
    fposCount += sizeof(ac2Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard2 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteCard4Array(ac4Test, 128);
    fposCount += sizeof(ac4Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TCard4 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteInt1Array(ai1Test, 256);
    fposCount += sizeof(ai1Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt1 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteInt2Array(ai2Test, 128);
    fposCount += sizeof(ai2Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt2 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteInt4Array(ai4Test, 128);
    fposCount += sizeof(ai4Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TInt4 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteFloat4Array(af4Test, 256);
    fposCount += sizeof(af4Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TFloat4 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    strmToTest.WriteFloat8Array(af8Test, 256);
    fposCount += sizeof(af8Test);
    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Write of TFloat8 array did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    //
    //  Now reset the stream and read the values back that we just wrote
    //  and make sure that they are valid. Modify the arrays to make sure
    //  that we don't just see the original values.
    //
    strmToTest.Reset();

    tCIDLib::TCard1     c1Read;
    tCIDLib::TCard2     c2Read;
    tCIDLib::TCard4     c4Read;
    tCIDLib::TInt1      i1Read;
    tCIDLib::TInt2      i2Read;
    tCIDLib::TInt4      i4Read;
    tCIDLib::TFloat4    f4Read;
    tCIDLib::TFloat8    f8Read;

    TRawMem::SetMemBuf(ac1Test, tCIDLib::TCard1(0), 256);
    TRawMem::SetMemBuf(ac2Test, tCIDLib::TCard2(0), 128);
    TRawMem::SetMemBuf(ac4Test, tCIDLib::TCard4(0), 128);
    TRawMem::SetMemBuf(ai1Test, tCIDLib::TInt1(0), 256);
    TRawMem::SetMemBuf(ai2Test, tCIDLib::TInt2(0), 128);
    TRawMem::SetMemBuf(ai4Test, tCIDLib::TInt4(0), 128);
    TRawMem::SetMemBuf(af4Test, tCIDLib::TFloat4(0), 256);
    TRawMem::SetMemBuf(af8Test, tCIDLib::TFloat8(0), 256);

    strmToTest >> c1Read;
    strmToTest >> c2Read;
    strmToTest >> c4Read;
    strmToTest >> i1Read;
    strmToTest >> i2Read;
    strmToTest >> i4Read;
    strmToTest >> f4Read;
    strmToTest >> f8Read;

    strmToTest.ReadCard1Array(ac1Test, 256);
    strmToTest.ReadCard2Array(ac2Test, 128);
    strmToTest.ReadCard4Array(ac4Test, 128);
    strmToTest.ReadInt1Array(ai1Test, 256);
    strmToTest.ReadInt2Array(ai2Test, 128);
    strmToTest.ReadInt4Array(ai4Test, 128);
    strmToTest.ReadFloat4Array(af4Test, 256);
    strmToTest.ReadFloat8Array(af8Test, 256);

    if (strmToTest.fposCurPos() != fposCount)
    {
        strmOut << _CurLn_ << L"Read back of data did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    // And, once again, we should be at the end of data
    if (!strmToTest.bEndOfStream())
    {
        strmOut << _CurLn_ << L"Stream is not at end of data after reads"
                << NewLn;
        return;
    }

    if ((c1Read != 1)
    ||  (c2Read != 2)
    ||  (c4Read != 4)
    ||  (i1Read != -1)
    ||  (i2Read != -2)
    ||  (i4Read != -4)
    ||  (f4Read != 4.0)
    ||  (f8Read != 8.0))
    {
        strmOut << _CurLn_ << L"Data read back in was not correct" << NewLn;
        return;
    }

    //
    //  Compare all of the read in arrays with the values that we know
    //  we put there.
    //
    for (c4Index = 0; c4Index < 256; c4Index++)
    {
        if (ac1Test[c4Index] != tCIDLib::TCard1(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 128; c4Index++)
    {
        if (ac2Test[c4Index] != tCIDLib::TCard2(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 128; c4Index++)
    {
        if (ac4Test[c4Index] != c4Index)
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 256; c4Index++)
    {
        if (ai1Test[c4Index] != tCIDLib::TInt1(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 128; c4Index++)
    {
        if (ai2Test[c4Index] != tCIDLib::TInt2(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 128; c4Index++)
    {
        if (ai4Test[c4Index] != tCIDLib::TInt4(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 256; c4Index++)
    {
        if (af4Test[c4Index] != tCIDLib::TFloat4(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    for (c4Index = 0; c4Index < 256; c4Index++)
    {
        if (af8Test[c4Index] != tCIDLib::TFloat8(c4Index))
        {
            strmOut << _CurLn_ << L"Array data read back was not correct"
                    << NewLn;
            break;
        }
    }

    //
    //  Reset the stream again and test that it goes to zero. Then do a
    //  seek to end and make sure it goes back to the position we left it
    //  at during writing.
    //
    strmToTest.Reset();
    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Reset did not result in position of zero"
                << NewLn;
        return;
    }

    if (strmToTest.fposSeekToEnd() != fposCount)
    {
        strmOut << _CurLn_ << L"Seek to end did not affect stream"
                              L" position correctly" << NewLn;
        return;
    }

    //
    //  Lets make sure that we get an appropriate exception for trying to
    //  read while at the end of the stream.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        strmToTest >> c1Read;
    }

    catch(const TError& errToCatch)
    {
        if (errToCatch.bCheckError(facCIDLib, kCIDErrs::errcStrm_EndOfStream))
            bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Did not catch attempt to read at EOS" << NewLn;
        return;
    }

    //
    //  Reset the stream and skip forward so that we are 2 bytes from the
    //  end. Then try to read something that would go over that. This
    //  should also cause an exception.
    //
    strmToTest.Reset();
    if (strmToTest.fposSkipForwardBy(tCIDLib::TCard4(fposCount-2)) != fposCount - 2)
    {
        strmOut << _CurLn_ << L"Skip forward did not get to correct position"
                << NewLn;
        return;
    }

    bCaughtIt = kCIDLib::False;
    try
    {
        strmToTest >> c4Read;
    }

    catch(const TError& errToCatch)
    {
        if (errToCatch.bCheckError(facCIDLib, kCIDErrs::errcStrm_NotAllData))
            bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
    {
        strmOut << _CurLn_ << L"Did not catch attempt to read past EOS" << NewLn;
        return;
    }

    //
    //  Now write out class info for a class and read it back in to make
    //  sure that it comes back ok. We reset the stream first.
    //
    strmToTest.Reset();
    strmToTest.WriteClassInfo(TString::clsThis);
    strmToTest.Reset();
    TClass clsTest(strmToTest.clsReadClassInfo());

    if (clsTest != TString::clsThis)
    {
        strmOut << _CurLn_ << L"Read in class info was not correct" << NewLn;
        return;
    }

    // Truncate the stream
    strmToTest.TruncateAtZero();

    //
    //  Now both the logical end and current position should be zero.
    //
    if (strmToTest.fposLogicalEnd() != 0)
    {
        strmOut << _CurLn_ << L"Truncate did not result in logical end of zero"
                << NewLn;
        return;
    }

    if (strmToTest.fposCurPos() != 0)
    {
        strmOut << _CurLn_ << L"Truncate did not result in current pos of zero"
                << NewLn;
        return;
    }
}

static tCIDLib::TVoid __BasicStreamTests(TTextStream& strmOut)
{
    //
    //  We need to create multiple binary streams, each with different
    //  implementation objects, and pass them to a common tester. 
    //
    {
        strmOut << L"Testing memory buffer stream" << NewLn;
        TBinaryMBufStream strmToTest(1024, 2 * kCIDLib::c4MemPageSize);
        __CommonBinTests
        (
            strmOut
            , strmToTest
            , 0
            , kCIDLib::c4MemPageSize * 2
        );
    }

    // Now test a file stream
    {
        strmOut << L"Testing file buffer stream" << NewLn;
        TBinaryFileStream strmToTest
        (
            L"TestBinFileStream.Dat"
            , tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );
        __CommonBinTests(strmOut, strmToTest, 0, kCIDLib::i8MaxInt);
    }
}



// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestBinStreams
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  aspects of the binary stream classes.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestBinStreams()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        c4CurCount = facCIDLib.c4ObjectCount();

        pszCurTest = L"basic";
        __BasicStreamTests(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Binary stream test leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" binary streams test" << NewLn;
        throw;
    }
}
