//
// NAME: TestCIDLib_BaseArray.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the classes
//  created from the TBaseArray template. This template class is used to
//  create arrays of fundamental types.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/25/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


static tCIDLib::TVoid TestBaseArrayClasses(TTextStream& strmOut)
{
    TBaseArray<tCIDLib::TCard1> ac1Test(256);

    // Set some elements and read them back
    ac1Test[255] = 0xAF;
    ac1Test[0]   = 0xAF;

    if (ac1Test[255] != ac1Test[0])
        strmOut << _CurLn_ << L"TCard1 element set/get test failed" << NewLn;

    //
    //  Set the all entries to an arbitrary value and then check them all
    //  to make sure they were set.
    //
    ac1Test.SetAll(156);

    for (tCIDLib::TCard4 c4Ind = 0; c4Ind <= kCIDLib::c1MaxCard; c4Ind++)
    {
        if (ac1Test[c4Ind] != 156)
            strmOut << _CurLn_ << L"TCard1Array value was not set to 156" << NewLn;
    }

    //
    //  Zero back out all of the elements and then check them all to make
    //  sure they were cleared.
    //
    ac1Test.ZeroAll();

    for (c4Ind = 0; c4Ind <= kCIDLib::c1MaxCard; c4Ind++)
    {
        if (ac1Test[c4Ind] != 0)
            strmOut << _CurLn_ << L"TCard1Array value was not set to 0" << NewLn;
    }

    //
    //  Fill the array with randomly generated values and then sort it
    //  and make sure it sorted ok. Seed it with the current thread id.
    //  Clip them to valid TCard1 values by using a mod divide.
    //
    TMathLib::SeedRandom(TSysInfo::tidCurrent());
    for (c4Ind = 0; c4Ind <= kCIDLib::c4MaxCard; c4Ind++)
    {
        ac1Test[c4Ind] = tCIDLib::TCard1
        (
            TMathLib::c4RandomNum() % kCIDLib::c1MaxCard
        );
    }

    ac1Test.Sort();

    // Now confirm the sort
    for (c4Ind = 0; c4Ind < kCIDLib::c1MaxCard; c4Ind++)
    {
        if (ac1Test[c4Ind] > ac1Test[c4Ind+1])
        {
            strmOut << _CurLn_ << L"Array was not in sorted order at element: "
                    << c4Ind << NewLn;
        }
    }
}



//
// FUNCTION/METHOD NAME: __TestConstructors
//
// DESCRIPTION:
//
//  This method will try out various constructor scenarios to make sure its
//  all getting initialized correctly.
// ---------------------------------------
//   INPUT: strmOut is the stream to dump messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestConstructors(TTextStream& strmOut)
{
    TBaseArray<tCIDLib::TCard1> ac1Test(256);

    //
    //  Make sure that all of the elements got zeroed out to start with.
    //
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 256; c4Ind++)
    {
        if (ac1Test[c4Ind] != 0)
        {
            strmOut << _CurLn_
                    << L"Initial TCard1Array value is not 0 at index "
                    << c4Ind << NewLn;
            break;
        }
    }

    TBaseArray<tCIDLib::TCard4> ac4Test(64, 16);

    //
    //  Make sure that all of the elements got zeroed out to start with.
    //
    for (c4Ind = 0; c4Ind < 64; c4Ind++)
    {
        if (ac4Test[c4Ind] != 16)
        {
            strmOut << _CurLn_
                    << L"Initial TCard4Array value is not 16 at index "
                    << c4Ind << NewLn;
            break;
        }
    }
}


//
// FUNCTION/METHOD NAME: __TestNumbering
//
// DESCRIPTION:
//
//  This method will test the numbering system, which initializes arrays
//  with contiguous values.
// ---------------------------------------
//   INPUT: strmOut is the stream to dump messages to
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestNumbering(TTextStream& strmOut)
{
    // Create an array of 4 byte ints
    TBaseArray<tCIDLib::TInt4> ai4Test(8);

    // Test the numbering method
    ai4Test.Number();
    if ((ai4Test[0] != 0) || (ai4Test[7] != 7))
        strmOut << _CurLn_ << L"Number() method did not work for TInt4" << NewLn;

    // Test the numbering method
    ai4Test.Number(8);
    if ((ai4Test[0] != 8) || (ai4Test[7] != 15))
        strmOut << _CurLn_
                << L"Offset Number() method did not work for TInt4" << NewLn;
}


//
// FUNCTION/METHOD NAME: __TestRotation
//
// DESCRIPTION:
//
//  This method test the up and down rotation systems. It numbers the
//  elements, then rotates them and checks the results.
// ---------------------------------------
//   INPUT: strmOut is the text stream to send messages to.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __TestRotation(TTextStream& strmOut)
{
    // Create an array to rotate, and number it
    TBaseArray<tCIDLib::TInt4> ai4Test(8);
    ai4Test.Number();

    // Test the rotation methods
    ai4Test.RotateUp(2);
    if ((ai4Test[0] != 6) || (ai4Test[7] != 5))
        strmOut << _CurLn_ << L"RotateUp() method did not work" << NewLn;

    ai4Test.RotateDown(4);
    if ((ai4Test[0] != 2) || (ai4Test[7] != 1))
        strmOut << _CurLn_ << L"RotateDown() method did not work" << NewLn;
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestBaseArrays
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various
//  instantiations of the base array class.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestBaseArrays()
{
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        pszCurTest = L"constructor";
        __TestConstructors(strmOut());

        pszCurTest = L"numbering";
        __TestNumbering(strmOut());

        pszCurTest = L"rotation";
        __TestRotation(strmOut());
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" test" << NewLn;
        throw;
    }
}
