//
// NAME: TestCIDLib_Bags.Cpp
//
// DESCRIPTION:
//
//  This module is part of the TestCIDLib.Exe program and is called from the
//  program's main() function. The functions in this module test the
//  bag collection classes.
//
//  This module brings in a common set of collection tests that are implemented
//  as templates themselves so that they can be invoked for all types of
//  collections without redundancy. But some of the testing is always specific
//  to the particular collection type.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 04/06/93
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"
#include    "TestCIDLib_CommonCollect.Hpp"


// -----------------------------------------------------------------------------
//  Typedef our collections used here
// -----------------------------------------------------------------------------
typedef TBag<TCardinal>     TBagOfTCardinal;
typedef TCntPtr<TString>    TCntStr;
typedef TBag<TCntStr>       TBagOfTString;
typedef TSortedBag<TFloat>  TSortedBagOfTFloat;


// -----------------------------------------------------------------------------
//  Force full instantiations of at least one specialization of each class so
//  that we can catch any errors
// -----------------------------------------------------------------------------
template class TBag<TCardinal>;
template class TSortedBag<TFloat>;


// -----------------------------------------------------------------------------
//  In order to avoid warnings about unimplemented methods, provide dummy
//  implementations here.
// -----------------------------------------------------------------------------
DummyTmplImpls(TBag<TCardinal>)
DummyTmplImpls(TSortedBag<TFloat>)


// -----------------------------------------------------------------------------
//  Local static functions
// -----------------------------------------------------------------------------
static tCIDLib::TVoid __TestBagCommon(TTextStream& strmOut)
{
    // Create a bag of cardinal objects. It can hold up to 16 objects
    TBagOfTCardinal colTest(16);

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"TBagOfTCardinal")))
    {
        strmOut << _CurLn_ << L"Bag class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    // Throw in some cardinal objects
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Add(TCardinal(c4Ind+1));

    //
    //  Do the copy and duplication tests. We provide a standard object
    //  equality object for the element comparison.
    //
    TestColCopy(strmOut, colTest, TStdObjEq<TCardinal>());

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colTest, 16);

    //
    //  Do the basic test again but this time with only 1 element, to check
    //  for any boundary conditions.
    //
    colTest.Add(TCardinal(1));
    TestColBasics(strmOut, colTest, 1);

    // Throw back some cardinal objects
    for (c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Add(TCardinal(c4Ind+1));

    // And do the second basic collection test
    TestColBasics2(strmOut, colTest, 16);
}


static tCIDLib::TVoid __TestBagBasics(TTextStream& strmOut)
{
    //
    //  Create a bag of cardinal objects. It can hold up to 16 objects
    //  and load it up with consecutive numeric values.
    //
    TBagOfTCardinal colTest(16);
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Add(TCardinal(c4Ind+1));

    //
    //  We want to make sure that all of the values are present. We cannot
    //  assume any order for bags, so we use a check off array.
    //
    tCIDLib::TCard1 ac1Check[16];
    for (c4Ind = 0; c4Ind < 16; c4Ind++)
        ac1Check[c4Ind] = 0;

    // Create a cursor
    TBagOfTCardinal::TCursor cursTest(&colTest);

    // Reset the cursor
    cursTest.bReset();
    do
    {
        tCIDLib::TCard4 c4Val = cursTest.objCur();

        if (c4Val > 16)
        {
            strmOut << _CurLn_ << L"Bag element value > 15" << NewLn;
            break;
        }

        if (c4Val < 1)
        {
            strmOut << _CurLn_ << L"Bag element value < 1" << NewLn;
            break;
        }

        if (ac1Check[c4Val-1])
        {
            strmOut << _CurLn_ << L"Bag element value "
                    << c4Val << L" occured more than once" << NewLn;
        }

        // Mark this one
        ac1Check[c4Val-1]++;

    }   while (cursTest.bNext());

    // Make sure we hit them all
    for (c4Ind = 0; c4Ind < 16; c4Ind++)
    {
        if (!ac1Check[c4Ind])
        {
            strmOut << _CurLn_ << L"Bag element value "
                    << c4Ind+1 << L" was not found" << NewLn;
        }
    }
}


static tCIDLib::TVoid __TestBagSearch(TTextStream& strmOut)
{
    TBagOfTCardinal colTest(4);

    colTest.Add(TCardinal(1));
    colTest.Add(TCardinal(1));
    colTest.Add(TCardinal(1));
    colTest.Add(TCardinal(2));

    TColSearch<TBagOfTCardinal,TCardinal> colsTest
    (
        &colTest
        , new TStdObjEq<TCardinal>
    );

    // Look for a 1 value
    if (!colsTest.bFind(TCardinal(1)))
        strmOut << _CurLn_ << L"Bag search failed to find TCardinal(1)" << NewLn;

    // We should find two more
    if (!colsTest.bFindNext(TCardinal(1)))
        strmOut << _CurLn_ << L"Bag search failed to find 2nd TCardinal(1)" << NewLn;

    if (!colsTest.bFindNext(TCardinal(1)))
        strmOut << _CurLn_ << L"Bag search failed to find 3rd TCardinal(1)" << NewLn;

    // We should not find another one
    if (colsTest.bFindNext(TCardinal(1)))
        strmOut << _CurLn_ << L"Bag search found a 4th TCardinal(1)" << NewLn;

    // Should find one 2 value
    if (!colsTest.bFind(TCardinal(2)))
        strmOut << _CurLn_ << L"Bag search failed to find TCardinal(2)" << NewLn;

    // Get a cursor copy out and it should pointer to this element
    TBagOfTCardinal::TCursor cursFind(colsTest.cursCur());

    if (cursFind.objCur() != TCardinal(2))
        strmOut << _CurLn_ << L"Cursor from search not pointing at found node" << NewLn;
}


static tCIDLib::TVoid __TestByRefBags(TTextStream& strmOut)
{
    //
    //  Create a bag that is in terms of the counted pointer template class,
    //  which is in terms of TString. This will let us put in anything
    //  derived from TString.
    //
    TBagOfTString   colToTest(64);

    //
    //  Fill in the array with some string objects in their counter objects.
    //  We created a typedef for the counted string pointer above, called
    //  TCntStr.
    //
    colToTest.Add(TCntStr(new TString(L"This is string #1")));
    colToTest.Add(TCntStr(new TString(L"This is string #2")));
    colToTest.Add(TCntStr(new TString(L"This is string #3")));
    colToTest.Add(TCntStr(new TString(L"This is string #4")));
    colToTest.Add(TCntStr(new TString(L"This is string #5")));
    colToTest.Add(TCntStr(new TString(L"This is string #6")));
    colToTest.Add(TCntStr(new TString(L"This is string #7")));

    // Reset the internal iterator
    colToTest.bResetIter();

    // And create a cursor
    TBagOfTString::TCursor cursTest(&colToTest);

    // Check that they agree as to the first element
    if (colToTest.objCur().objData() != cursTest.objCur().objData())
        strmOut << _CurLn_ << L"Cursor and iterator don't agree" << NewLn;

    //
    //  Invoke the basic collection testing template function. It will do
    //  some generic things that all collections should do the same.
    //
    TestColBasics(strmOut, colToTest, 7);
}


static tCIDLib::TVoid __TestSortedBags(TTextStream& strmOut)
{
    // Create a bag of cardinal objects. It can hold up to 16 objects
    TSortedBagOfTFloat colTest(16, new TStdObjComp<TFloat>);

    // Check the automatic class name generation
    if (!colTest.bIsA(TClass(L"TSortedBagOfTFloat")))
    {
        strmOut << _CurLn_ << L"Bag class name was bad. It was: "
                << colTest.clsIsA() << NewLn;
    }

    //
    //  Throw in some cardinal objects, in reverse order. It should sort
    //  them as they are inserted.
    //
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < 16; c4Ind++)
        colTest.Add(TFloat(16UL - c4Ind));

    // Get a cursor and check the elements
    TSortedBagOfTFloat::TCursor cursTest(&colTest);

    // Iterate the bag and make sure that they are in sorted order.
    if (!cursTest.bReset())
    {
        strmOut << _CurLn_ << L"Cursor reset but there are elements available"
                << NewLn;
        return;
    }

    tCIDLib::TFloat8 f8Test = 1.0;
    do
    {
        if (cursTest.objCur() != f8Test)
        {
            strmOut << _CurLn_ << L"Elements were not found in sorted order"
                    << NewLn;
            break;
        }
        f8Test += 1.0;
    }   while (cursTest.bNext());
}



// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TestBags
//
// DESCRIPTION:
//
//  This method calls a number of local functions that test various aspects of
//  the bag collections.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacTestCIDLib::TestBags()
{
    tCIDLib::TCard4     c4CurCount;
    const tCIDLib::Tch* pszCurTest = L"None";
    try
    {
        TKrnlMemCheck kmchkTest;

        // Get the current object count
        c4CurCount = facCIDLib.c4ObjectCount();
        
        // Invoke the bag tests
        pszCurTest = L"common";
        __TestBagCommon(strmOut());

        pszCurTest = L"basic";
        __TestBagBasics(strmOut());

        pszCurTest = L"by ref";
        __TestByRefBags(strmOut());

        pszCurTest = L"search";
        __TestBagSearch(strmOut());

        pszCurTest = L"sorted";
        __TestSortedBags(strmOut());

        if (c4CurCount != facCIDLib.c4ObjectCount())
            strmOut() << _CurLn_ << L"Bag tests leaked objects" << NewLn;
    }

    catch(...)
    {
        strmOut()   << L"Exception occured during the " << pszCurTest
                    << L" bag test" << NewLn;
        throw;
    }
}
