//
//  FILE NAME: TestCIDLib.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the test program. It does some setup then
//  calls all of the testing modules.
//
//
//  CAVEATS/GOTCHAS:
//



// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestCIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Global data
// ----------------------------------------------------------------------------
TFacTestCIDLib  facTestCIDLib;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"TestCIDLibMainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local data
// ----------------------------------------------------------------------------
static tCIDLib::TBoolean    __bRunAll;
static TTestFuncRecord      __aTestFunctions[] =
{
        { TFacTestCIDLib::ShowSystemInfo    , L"SysInfo"        , kCIDLib::False }
    ,   { TFacTestCIDLib::TestEnvironment   , L"Environment"    , kCIDLib::False }
    ,   { TFacTestCIDLib::TestPointerClasses, L"Pointers"       , kCIDLib::False }
    ,   { TFacTestCIDLib::TestStrings       , L"Strings"        , kCIDLib::False }
    ,   { TFacTestCIDLib::TestRTTI          , L"RTTI"           , kCIDLib::False }
    ,   { TFacTestCIDLib::TestNumericClasses, L"Numerics"       , kCIDLib::False }
    ,   { TFacTestCIDLib::TestProcess       , L"Process"        , kCIDLib::False }
    ,   { TFacTestCIDLib::TestMemoryClasses , L"Memory"         , kCIDLib::False }
    ,   { TFacTestCIDLib::TestCoordinates   , L"Coordinates"    , kCIDLib::False }
    ,   { TFacTestCIDLib::TestBaseArrays    , L"BaseArrays"     , kCIDLib::False }
    ,   { TFacTestCIDLib::TestFiles         , L"Files"          , kCIDLib::False }
    ,   { TFacTestCIDLib::TestTime          , L"Time"           , kCIDLib::False }
    ,   { TFacTestCIDLib::TestBinStreams    , L"BinStreams"     , kCIDLib::False }
    ,   { TFacTestCIDLib::TestTextStreams   , L"TextStreams"    , kCIDLib::False }
    ,   { TFacTestCIDLib::TestBags          , L"Bags"           , kCIDLib::False }
    ,   { TFacTestCIDLib::TestDeques        , L"Deques"         , kCIDLib::False }
    ,   { TFacTestCIDLib::TestQueues        , L"Queues"         , kCIDLib::False }
    ,   { TFacTestCIDLib::TestObjArrays     , L"ObjArrays"      , kCIDLib::False }
    ,   { TFacTestCIDLib::TestHashMap       , L"HashMap"        , kCIDLib::False }
    ,   { TFacTestCIDLib::TestHashSet       , L"HashSet"        , kCIDLib::False }
    ,   { TFacTestCIDLib::TestStacks        , L"Stacks"         , kCIDLib::False }
};
static const tCIDLib::TCard4 __c4TestFuncCount = c4ArrayElems(__aTestFunctions);


// ----------------------------------------------------------------------------
//   CLASS: TTestSysExcept
//  PREFIX: sexc
//
//  This is a derivative of the TKrnlSysExcept class that allows us to
//  deal with system exceptions in a portable way. This lets us both test
//  this class and provide some information on any exception that occurs.
//  So it wraps all of the calls to the testing functions
// ----------------------------------------------------------------------------

class TTestSysExcept : public TKrnlSysExcept
{
    public :
        // --------------------------------------------------------------------
        //  Constructors and Destructor
        // --------------------------------------------------------------------
        TTestSysExcept() {}

        ~TTestSysExcept() {}


    protected :
        // --------------------------------------------------------------------
        //  Protected, inherited methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _DoWork();

        tCIDLib::ESysExceptRes _eRecoverTest();

        tCIDLib::TVoid _Handler();
};


tCIDLib::TVoid TTestSysExcept::_DoWork()
{
    //
    //  Call all of the testing methods. These are all methods of
    //  the facility object, but they are spread out in the modules
    //  that are relevant to that particular subsystem.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TestFuncCount; c4Index++)
    {
        if (__aTestFunctions[c4Index].bRun || __bRunAll)
        {
            facTestCIDLib.strmOut() << L"Testing "
                                    << __aTestFunctions[c4Index].pszName
                                    << L"..." << NewLn;

            // Call the test function.
            __aTestFunctions[c4Index].pfnTester();

            // And validate the heap
            TKrnlMemCheck::ValidateHeap();

            facTestCIDLib.strmOut()
                << __aTestFunctions[c4Index].pszName
                << L" tests complete" << DNewLn;
        }
    }
}

tCIDLib::ESysExceptRes TTestSysExcept::_eRecoverTest()
{
    // Report the exception
    facTestCIDLib.strmOut() << L"A system exception occured. "
                               L"Exception type: " << _eExceptCode()
                            << NewLn;

    // We let them all propogate
    return tCIDLib::ESysExceptRes_Propogate;
}

tCIDLib::TVoid TTestSysExcept::_Handler()
{
    // Nothing to do. _eRecoverTest just reports them
}



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the EExitCodes values to indicate why we exited.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    {
        // Set our logging threshold down low for testing purposes
        TSysInfo::bVerboseMode(kCIDLib::True);

        // Log our arrival
        facTestCIDLib.LogMsg
        (
            __FILE__
            , __LINE__
            , L"TestCIDLib starting up..."
            , tCIDLib::ESev_Status
        );

        // Announce ourself
        facTestCIDLib.Announce();

        //
        //  See what tests we need to run. The user indicates which
        //  tests by naming them on the command line. If none were named
        //  then we run them all.
        //
        tCIDLib::TCard4     c4Index;
        tCIDLib::TCard4     c4FnInd;
        TString             strTmp;

        //
        //  If there are 2 params, and teh second one is /?, then display
        //  all of the tests we can do.
        //
        if (TSysInfo::c4CmdLineArgCount() == 2)
        {
            TSysInfo::CmdLineArg(1, strTmp);
            if (strTmp == L"/?")
            {
                facTestCIDLib.ShowTests();
                return tCIDLib::EExit_Normal;
            }
        }

        //
        //  Otherwise, if there are any user params, find the ones that match
        //  tests we know about and run them. If no users parms, then run
        //  them all.
        //  
        if (TSysInfo::c4CmdLineArgCount() > 1)
        {
            tCIDLib::TBoolean bAnyMatches = kCIDLib::False;
            for (c4Index = 1; c4Index < TSysInfo::c4CmdLineArgCount(); c4Index++)
            {
                TSysInfo::CmdLineArg(c4Index, strTmp);
                if (strTmp[0] == L'/')
                    continue;

                tCIDLib::TBoolean bMatched = kCIDLib::False;
                for (c4FnInd = 0; c4FnInd < __c4TestFuncCount; c4FnInd++)
                {
                    if (!strTmp.eICompare(__aTestFunctions[c4FnInd].pszName))
                    {
                        bAnyMatches = kCIDLib::True;
                        bMatched = kCIDLib::True;
                        __aTestFunctions[c4FnInd].bRun = kCIDLib::True;
                    }
                }

                if (!bMatched)
                {
                    facTestCIDLib.strmOut()
                                << strTmp
                                << L" is not a valid test name" << NewLn;
                }
            }

            if (!bAnyMatches)
            {
                facTestCIDLib.ShowTests();
                return tCIDLib::EExit_Normal;
            }
        }
         else
        {
            __bRunAll = kCIDLib::True;
        }

        facTestCIDLib.strmOut() << DNewLn;

        try
        {
            TTestSysExcept sexcTest;
            sexcTest.DoWork();
        }

        // Catch and report any error that occured
        catch(const TError& errToCatch)
        {
            facTestCIDLib.strmOut()
                << L"Unhandled exception occured!" << DNewLn
                << L"   File: " << errToCatch.strFileName() << NewLn
                << L"   Line: " << errToCatch.c4LineNum() << NewLn
                << L"  Error: " << errToCatch.strErrText() << NewLn
                << L"Aux Txt: " << errToCatch.strAuxText() << NewLn
                << L"ErrCode: " << errToCatch.errcId() << NewLn
                << L" Thread: " << errToCatch.strThread() << NewLn;
            return tCIDLib::EExit_FatalError;
        }
    }

    // Log our exit
    facTestCIDLib.LogMsg
    (
        __FILE__
        , __LINE__
        , L"TestCIDLib terminating..."
        , tCIDLib::ESev_Status
    );

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}



// -----------------------------------------------------------------------------
//  CLASS: TFacTestCIDLib
// PREFIX: fac
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Static data
// -----------------------------------------------------------------------------
TTextStream*    TFacTestCIDLib::__pstrmOutput = 0;

// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Constructors and destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TFacTestCIDLib
//                       ~TFacTestCIDLib
//
// DESCRIPTION:
//
//  The default constructor just inits the output stream pointer to 0. We
//  call our parent first and provide the needed information.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
TFacTestCIDLib::TFacTestCIDLib() :

    TFacility
    (
        L"TestCIDLib"
        , tCIDLib::EModType_Exe
        , kCIDLib::c4MajVersion
        , kCIDLib::c4MinVersion
    )
{
    // Set our static stream member
    __pstrmOutput = new TConsole
    (
        kCIDLib::True
        , 0
        , tCIDLib::ERedir_Allow
    );
}

TFacTestCIDLib::~TFacTestCIDLib()
{
}


// -----------------------------------------------------------------------------
//  TFacTestCIDLib: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFacTestCIDLib::Announce()
{
    // Announce ourselves
    strmOut()   << L"\r\nTestCIDLib.Exe" << NewLn
                << L"Version " << facCIDLib.strVersion()
                << L", Compiled: " << TString(__DATE__) << NewLn
                << L"Test facility for the CIDLib libraries"
                << DNewLn;
}


tCIDLib::TVoid TFacTestCIDLib::ShowSystemInfo()
{
    // Display the command line parameters
    strmOut()   << L"Command line parm count: "
                << TSysInfo::c4CmdLineArgCount()
                << NewLn;

    tCIDLib::TCard4 c4Index;
    TString         strTmp(kCIDLib::pszEmptyZStr, 128);
    for (c4Index = 0; c4Index < TSysInfo::c4CmdLineArgCount(); c4Index++)
    {
        TSysInfo::CmdLineArg(c4Index, strTmp);
        strmOut() << L"Arg" << c4Index << L"=" << strTmp << NewLn;
    }
    strmOut() << DNewLn;

    //
    //  We use two alternating formatting schemes here, so we just create
    //  two stream format objects with those schemes. Then we can just
    //  dump each one in to set the formatting.
    //
    //  The legends are right justified in a field large enough that they
    //  are all indented over a little. The data is not justified and the
    //  field width is set back to 0, which means just whatever space that
    //  they naturally take.
    //
    TStreamFmt  strmfLegend(22, 2, tCIDLib::EHJustify_Right, L' ');
    TStreamFmt  strmfData(0, 2, tCIDLib::EHJustify_Left, L' ');

    //
    //  Get the host OS version information for later display.
    //
    tCIDLib::TCard4 c4MajVer, c4MinVer, c4BuildNum;
    TSysInfo::QueryOSInfo(c4MajVer, c4MinVer, c4BuildNum);

    //
    //  Do a display using the info methods provided by CIDLib.Dll
    //  and Kernel.Dll.
    //
    strmOut()   << L"Test CIDLib System Info methods" << NewLn
                << L"-------------------------------"  << NewLn
                << strmfLegend << L"Process Name: "
                << strmfData << TSysInfo::strProcessName() << NewLn

                << strmfLegend << L"User Name: "
                << strmfData << TSysInfo::strUserName() << NewLn

                << strmfLegend << L"CPU Type: "
                << strmfData << TSysInfo::eCPUType() << NewLn

                << strmfLegend << L"OS Version: "
                << strmfData << c4MajVer << L"." << c4MinVer
                << L", Build: " << c4BuildNum << NewLn

                << strmfLegend << L"Node Name: "
                << strmfData << TSysInfo::strNodeName() << NewLn

                << strmfLegend << L"Physical Memory: "
                << strmfData << TLocCardinal(TSysInfo::c4TotalPhysicalMem())
                << NewLn

                << strmfLegend << L"Max Path Length: "
                << strmfData << kCIDLib::c4MaxPathLen << NewLn

                << strmfLegend << L"Verbose: "
                << strmfData << TSysInfo::bVerboseMode() << NewLn

                << strmfLegend << L"Object Metrics: "
                << strmfData << TSysInfo::bObjectMetrics() << NewLn

                << strmfLegend << L"Log Threshold: "
                << strmfData << TFacility::eSevThreshold() << NewLn

                << DNewLn;
}


tCIDLib::TVoid TFacTestCIDLib::ShowTests()
{
    strmOut() << L"\r\nThis program provides the following tests:\r\n\n";

    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TestFuncCount; c4Index++)
        strmOut() << L"    " << __aTestFunctions[c4Index].pszName << NewLn;

    strmOut() << NewLn << L"Indicate the tests to run as separate parameters"
              << NewLn << L"No parameters will run all tests." << DNewLn;
}
