//
//  FILE NAME: TestCIDCrypto.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 10/22/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the test program. It does some setup then
//  calls all of the testing modules.
//
//
//  CAVEATS/GOTCHAS:
//



// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "TestCIDCrypto.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Global data
// ----------------------------------------------------------------------------
TFacTestCIDCrypto   facTestCIDCrypto;


// ----------------------------------------------------------------------------
//  Local data
// ----------------------------------------------------------------------------
static tCIDLib::TBoolean    __bRunAll;
static TTestFuncRecord      __aTestFunctions[] =
{
        { TFacTestCIDCrypto::TestXor        , L"Xor"        , kCIDLib::False }
    ,   { TFacTestCIDCrypto::TestBlowfish   , L"Blowfish"   , kCIDLib::False }
};
static const tCIDLib::TCard4 __c4TestFuncCount = c4ArrayElems(__aTestFunctions);



// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"TestCIDCryptoMainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the EExitCodes values to indicate why we exited.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    {
        // Set our logging threshold down low for testing purposes
        TSysInfo::bVerboseMode(kCIDLib::True);

        // Log our arrival
        facTestCIDCrypto.LogMsg
        (
            __FILE__
            , __LINE__
            , L"TestCIDCrypto starting up..."
            , tCIDLib::ESev_Status
        );

        // Announce ourself
        facTestCIDCrypto.Announce();

        //
        //  See what tests we need to run. The user indicates which
        //  tests by naming them on the command line. If none were named
        //  then we run them all.
        //
        tCIDLib::TCard4     c4Index;
        tCIDLib::TCard4     c4FnInd;
        TString             strTmp;

        //
        //  If there are 2 params, and teh second one is /?, then display
        //  all of the tests we can do.
        //
        if (TSysInfo::c4CmdLineArgCount() == 2)
        {
            TSysInfo::CmdLineArg(1, strTmp);
            if (strTmp == L"/?")
            {
                facTestCIDCrypto.ShowTests();
                return tCIDLib::EExit_Normal;
            }
        }

        //
        //  Otherwise, if there are any user params, find the ones that match
        //  tests we know about and run them. If no users parms, then run
        //  them all.
        //  
        if (TSysInfo::c4CmdLineArgCount() > 1)
        {
            tCIDLib::TBoolean bAnyMatches = kCIDLib::False;
            for (c4Index = 1; c4Index < TSysInfo::c4CmdLineArgCount(); c4Index++)
            {
                TSysInfo::CmdLineArg(c4Index, strTmp);
                if (strTmp[0] == L'/')
                    continue;

                tCIDLib::TBoolean bMatched = kCIDLib::False;
                for (c4FnInd = 0; c4FnInd < __c4TestFuncCount; c4FnInd++)
                {
                    if (!strTmp.eICompare(__aTestFunctions[c4FnInd].pszName))
                    {
                        bAnyMatches = kCIDLib::True;
                        bMatched = kCIDLib::True;
                        __aTestFunctions[c4FnInd].bRun = kCIDLib::True;
                    }
                }

                if (!bMatched)
                {
                    facTestCIDCrypto.strmOut()
                                << strTmp
                                << L" is not a valid test name" << NewLn;
                }
            }

            if (!bAnyMatches)
            {
                facTestCIDCrypto.ShowTests();
                return tCIDLib::EExit_Normal;
            }
        }
         else
        {
            __bRunAll = kCIDLib::True;
        }

        facTestCIDCrypto.strmOut() << DNewLn;

        try
        {
            //
            //  Call all of the testing methods. These are all methods of
            //  the facility object, but they are spread out in the modules
            //  that are relevant to that particular subsystem.
            //
            for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TestFuncCount; c4Index++)
            {
                if (__aTestFunctions[c4Index].bRun || __bRunAll)
                {
                    facTestCIDCrypto.strmOut()
                            << L"Testing "
                            << __aTestFunctions[c4Index].pszName
                            << L"..." << NewLn;

                    // Call the test function.
                    __aTestFunctions[c4Index].pfnTester();

                    // And validate the heap
                    TKrnlMemCheck::ValidateHeap();

                    facTestCIDCrypto.strmOut()
                        << __aTestFunctions[c4Index].pszName
                        << L" tests complete" << DNewLn;
                }
            }
        }

        // Catch and report any error that occured
        catch(const TError& errToCatch)
        {
            facTestCIDCrypto.strmOut()
                << L"Unhandled exception occured!" << DNewLn
                << L"   File: " << errToCatch.strFileName() << NewLn
                << L"   Line: " << errToCatch.c4LineNum() << NewLn
                << L"  Error: " << errToCatch.strErrText() << NewLn
                << L"Aux Txt: " << errToCatch.strAuxText() << NewLn
                << L"ErrCode: " << errToCatch.errcId() << NewLn
                << L" Thread: " << errToCatch.strThread() << NewLn;
            return tCIDLib::EExit_FatalError;
        }
    }

    // Log our exit
    facTestCIDCrypto.LogMsg
    (
        __FILE__
        , __LINE__
        , L"TestCIDLib terminating..."
        , tCIDLib::ESev_Status
    );

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}



// -----------------------------------------------------------------------------
//  CLASS: TFacTestCIDCrypto
// PREFIX: fac
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TFacTestCIDCrypto: Static data
// -----------------------------------------------------------------------------
TTextStream*    TFacTestCIDCrypto::__pstrmOutput = 0;

// -----------------------------------------------------------------------------
//  TFacTestCIDCrypto: Constructors and destructors
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TFacTestCIDCrypto
//                       ~TFacTestCIDCrypto
//
// DESCRIPTION:
//
//  The default constructor just inits the output stream pointer to 0. We
//  call our parent first and provide the needed information.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
TFacTestCIDCrypto::TFacTestCIDCrypto() :

    TFacility
    (
        L"TestCIDCrypto"
        , tCIDLib::EModType_Exe
        , kCIDLib::c4MajVersion
        , kCIDLib::c4MinVersion
    )
{
    // Set our static stream member
    __pstrmOutput = new TConsole
    (
        kCIDLib::True
        , 0
        , tCIDLib::ERedir_Allow
    );
}

TFacTestCIDCrypto::~TFacTestCIDCrypto()
{
}


// -----------------------------------------------------------------------------
//  TFacTestCIDCrypto: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TFacTestCIDCrypto::Announce()
{
    // Announce ourselves
    strmOut()   << L"\r\nTestCIDCrypto.Exe" << NewLn
                << L"Version " << facTestCIDCrypto.strVersion()
                << L", Compiled: " << TString(__DATE__) << NewLn
                << L"Test facility for the CIDLib Cryptography libraries"
                << DNewLn;
}

tCIDLib::TVoid TFacTestCIDCrypto::ShowTests()
{
    strmOut() << L"\r\nThis program provides the following tests:\r\n\n";

    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TestFuncCount; c4Index++)
        strmOut() << L"    " << __aTestFunctions[c4Index].pszName << NewLn;

    strmOut() << NewLn << L"Indicate the tests to run as separate parameters"
              << NewLn << L"No parameters will run all tests." << DNewLn;
}
