//
//  FILE NAME: Streams5.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the fifth of the streaming oriented demo
//  programs. This one just demonstrates some basic text stream input
//  mechanisms. It just outputs some prompts to ask the user to enter some
//  values and then gets the input and displays whether the input was
//  correct or not.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. We
//      enable the command line recall and editing capabilities for this
//      program since its interactive, and don't allow redirection.
// ----------------------------------------------------------------------------
static TConsole     __conOut
                    (
                        kCIDLib::True
                        , 16
                        , tCIDLib::ERedir_Disallow
                    );


// ----------------------------------------------------------------------------
//  Do the magic main module code
//
//  This tells CIDLib what the main thread of the program is. This is the
//  only thread object that is run automatically. All others are started
//  manually when required or desired.
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Streams5MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __GetValue
//
// DESCRIPTION:
//
//  A template function that is used to get all the various types of values.
//  It does the standard work of prompting and checking and displaying results
// ---------------------------------------
//   INPUT: strLegal is a string with the legal values of the type being
//              gotten.
//          strType is a string with the name of the type being gotten
//
//  OUTPUT: None
//
//  RETURN: A dummy return value to get around an error in VC++. Get rid
//              of this and every invocation will use the type of the last
//              invocation!!
//
template <class T> T __GetAValue(const TString& strLegal, const TString& strType)
{
    T Val;

    // Ask the user to enter a value value
    try
    {
        __conOut << L"Enter a " << strType << L" value (" << strLegal << L"): ";
        __conOut >> Val;
        __conOut << L"You entered: " << Val << DNewLn;
    }

    catch(const TError& errToCatch)
    {
        __conOut << L"Value was invalid. Reason: "
                 << errToCatch.strErrText()
                 << DNewLn;
    }
    return Val;
}


//
// FUNCTION/METHOD NAME: __DoWork
//
// DESCRIPTION:
//
//  This function is called from the main thread to do the work of the
//  program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __DoWork()
{
    // Ask the user to enter a boolean value
    __GetAValue<tCIDLib::TBoolean>(L"True/False", L"TBoolean");

    // Ask the user to enter a TCard1 value
    __GetAValue<tCIDLib::TCard1>(L"0..255", L"TCard1");

    // Ask the user to enter a TCard2 value
    __GetAValue<tCIDLib::TCard2>(L"0..65535", L"TCard2");

    // Ask the user to enter a TCard4 value
    __GetAValue<tCIDLib::TCard4>(L"0..4294967295", L"TCard4");

    // Ask the user to enter a TInt1 value
    __GetAValue<tCIDLib::TInt1>(L"-128..127", L"TInt1");

    // Ask the user to enter a TInt2 value
    __GetAValue<tCIDLib::TInt2>(L"-32768..32767", L"TInt2");

    // Ask the user to enter a TInt4 value
    __GetAValue<tCIDLib::TInt4>(L"-2147483648..2147483647", L"TInt4");

    // Ask the user to enter a TInt8 value
    __GetAValue<tCIDLib::TInt8>(L"-9223372036854775808..9223372036854775807", L"TInt8");

    // Ask the user to enter a TFloat4 value
    __GetAValue<tCIDLib::TFloat8>(L"1.175494351e-38F..3.402823466e+38F", L"TFloat4");

    // Ask the user to enter a TFloat8 value
    __GetAValue<tCIDLib::TFloat8>(L"2.22507385e-308 .. 1.7976931e+308", L"TFloat8");
}


//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and partly a testing program, we'd like to
    //  catch all unexpected exceptions cleanly and report them. So put the
    //  whole thing in a try.
    //
    try
    {
        __DoWork();
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib.Dll, but I test
    //  for them in my test programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
