//
//  FILE NAME: Streams3_Widgets.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/01/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header file for the Streams3_Widgets.Cpp module. This module
//  implements a set of simple classes that derive from a common class. These
//  classes implement the streamable interface, the polymorphic duplicability
//  interface, and the required RTTI functionality to be polymorphically
//  streamed.
//
//  These classes emulate a set of 'graphic widgets' such as circles, boxes,
//  lines, etc... They all must implement a Draw() method, which is a
//  pure virtual in the TBaseWidget class. But, since this is just a TTY
//  output program, the 'drawing' is just a matter of outputting the type
//  of the class plus its data information. The type is output using the
//  clsIsA() RTTI method.
//
//  Since TBaseWidget also mixes in MStreamable and MDuplicable, but does not
//  implement the interface, each derivative must implement them. MStreamable
//  allows the objects to be streamed to/from a binary stream. MDuplicable
//  supports polymorphic duplication. They all use the DefPolyDup() macro
//  to implement the default duplication code, which is just a call to their
//  own copy constructor.
//
//  TFilledBoxWidget is a second level derivative, so note how it calls its
//  parent's version of everything. The standard RTTI macro create a typedef
//  called TParent, which stands for the parent class. So it always calls
//  through this magic TParent class instead of hard coding its parent class.
//  And all of the Draw() methods call down to the base class first
//  because it outputs the type info, which is common to all of their output.
//
//  CAVEATS/GOTCHAS:
//
//  1)  Because these are trivial examples, they do not follow the formal
//      guidelines for style. They just implement all of their functionality
//      inline, within the class definition. And they are more 'scrunched'
//      up that I would normally do so that they can all fit easily within
//      this one file.
//


// ----------------------------------------------------------------------------
//   CLASS: TBaseWidget
//  PREFIX: wid
// ----------------------------------------------------------------------------
class TBaseWidget : public TObject, public MStreamable, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Public constructors and destructors
        // --------------------------------------------------------------------
        ~TBaseWidget() {}

        // --------------------------------------------------------------------
        //  Public, virtual methods
        // --------------------------------------------------------------------
        virtual TTextStream& Draw(TTextStream& strmToWriteTo) const = 0
        {
            strmToWriteTo << L"Type: " << clsIsA() << L" - ";
            return strmToWriteTo;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors and operators
        // --------------------------------------------------------------------
        TBaseWidget() {}

        TBaseWidget(const TBaseWidget& toCopy) {}

        TBaseWidget& operator=(const TBaseWidget& toCopy) {return *this;}

        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TBaseWidget,TObject)
};



// ----------------------------------------------------------------------------
//   CLASS: TLineWidget
//  PREFIX: wid
// ----------------------------------------------------------------------------
class TLineWidget : public TBaseWidget
{
    public  :
        // --------------------------------------------------------------------
        //  Public constructors and destructors
        // --------------------------------------------------------------------
        TLineWidget(const TPoint& pntFrom, const TPoint& pntTo) :
            __pntFrom(pntFrom)
            , __pntTo(pntTo)
        {}

        TLineWidget(const TLineWidget& widToCopy) :
            __pntFrom(widToCopy.__pntFrom)
            , __pntTo(widToCopy.__pntTo)
        {}

        ~TLineWidget() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TLineWidget& operator=(const TLineWidget& widToAssign)
        {
            __pntFrom = widToAssign.__pntFrom;
            __pntTo = widToAssign.__pntTo;
            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        TTextStream& Draw(TTextStream& strmToWriteTo) const
        {
            strmToWriteTo   << TParent::Draw(strmToWriteTo)
                            << L"{" << __pntFrom
                            << L"," << __pntTo << L"}";
            return strmToWriteTo;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TLineWidget() :

            __pntFrom(0, 0)
            , __pntTo(1, 1)
        {}


        // --------------------------------------------------------------------
        //  Protected, implemented methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom(TBinaryStream& strmToReadFrom)
        {
            strmToReadFrom >> __pntFrom >> __pntTo;
        }

        tCIDLib::TVoid _StreamTo(TBinaryStream& strmToWriteTo) const
        {
            strmToWriteTo << __pntFrom << __pntTo;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pntFrom
        //  __pntTo
        //      The points that the line draws from and to.
        // --------------------------------------------------------------------
        TPoint  __pntFrom;
        TPoint  __pntTo;

        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TLineWidget,TBaseWidget)
        DefPolyDup(TLineWidget)
        BefriendFactory(TLineWidget)
};



// ----------------------------------------------------------------------------
//   CLASS: TCircleWidget
//  PREFIX: wid
// ----------------------------------------------------------------------------
class TCircleWidget : public TBaseWidget
{
    public  :
        // --------------------------------------------------------------------
        //  Public constructors and destructors
        // --------------------------------------------------------------------
        TCircleWidget(const TPoint& pntCenter, const tCIDLib::TCard4 c4Radius) :
            __c4Radius(c4Radius)
            , __pntCenter(pntCenter)
        {}

        TCircleWidget(const TCircleWidget& widToCopy) :
            __c4Radius(widToCopy.__c4Radius)
            , __pntCenter(widToCopy.__pntCenter)
        {}

        ~TCircleWidget() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TCircleWidget& operator=(const TCircleWidget& widToAssign)
        {
            __c4Radius = widToAssign.__c4Radius;
            __pntCenter = widToAssign.__pntCenter;
            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        TTextStream& Draw(TTextStream& strmToWriteTo) const
        {
            strmToWriteTo   << TParent::Draw(strmToWriteTo)
                            << L"Center/Radius: {" << __pntCenter
                            << L"," << __c4Radius << L"}";
            return strmToWriteTo;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TCircleWidget() :

            __c4Radius(1)
            , __pntCenter(0, 0)
        {}


        // --------------------------------------------------------------------
        //  Protected, implemented methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom(TBinaryStream& strmToReadFrom)
        {
            strmToReadFrom >> __pntCenter >> __c4Radius;
        }

        tCIDLib::TVoid _StreamTo(TBinaryStream& strmToWriteTo) const
        {
            strmToWriteTo << __pntCenter << __c4Radius;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4Radius
        //  __pntCenter
        //      The center point and radius of the circle
        // --------------------------------------------------------------------
        tCIDLib::TCard4 __c4Radius;
        TPoint          __pntCenter;

        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TCircleWidget,TBaseWidget)
        DefPolyDup(TCircleWidget)
        BefriendFactory(TCircleWidget)
};



// ----------------------------------------------------------------------------
//   CLASS: TBoxWidget
//  PREFIX: wid
// ----------------------------------------------------------------------------
class TBoxWidget : public TBaseWidget
{
    public  :
        // --------------------------------------------------------------------
        //  Public constructors and destructors
        // --------------------------------------------------------------------
        TBoxWidget(const TPoint& pntUL, const TPoint& pntLR) :
            __pntUL(pntUL)
            , __pntLR(pntLR)
        {}

        TBoxWidget(const TBoxWidget& widToCopy) :
            __pntUL(widToCopy.__pntUL)
            , __pntLR(widToCopy.__pntLR)
        {}

        ~TBoxWidget() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TBoxWidget& operator=(const TBoxWidget& widToAssign)
        {
            __pntUL = widToAssign.__pntUL;
            __pntLR = widToAssign.__pntLR;
            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        TTextStream& Draw(TTextStream& strmToWriteTo) const
        {
            strmToWriteTo   << TParent::Draw(strmToWriteTo)
                            << L"Corners: {" << __pntUL
                            << L"," << __pntLR << L"}";
            return strmToWriteTo;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TBoxWidget() :

            __pntUL(0, 0)
            , __pntLR(1, 1)
        {}


        // --------------------------------------------------------------------
        //  Protected, implemented methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom(TBinaryStream& strmToReadFrom)
        {
            strmToReadFrom >> __pntUL >> __pntLR;
        }

        tCIDLib::TVoid _StreamTo(TBinaryStream& strmToWriteTo) const
        {
            strmToWriteTo << __pntUL << __pntLR;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __pntUL
        //  __pntLR
        //      The points that the line draws from and to.
        // --------------------------------------------------------------------
        TPoint  __pntUL;
        TPoint  __pntLR;

        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TBoxWidget,TBaseWidget)
        DefPolyDup(TBoxWidget)
        BefriendFactory(TBoxWidget)
};




// ----------------------------------------------------------------------------
//   CLASS: TFilledBoxWidget
//  PREFIX: wid
// ----------------------------------------------------------------------------
class TFilledBoxWidget : public TBoxWidget
{
    public  :
        // --------------------------------------------------------------------
        //  Public constructors and destructors
        // --------------------------------------------------------------------
        TFilledBoxWidget(   const   TPoint&     pntUL
                            , const TPoint&     pntLL
                            , const TRGBClr&    rgbFill) :
            TBoxWidget(pntUL, pntLL)
            , __rgbFill(rgbFill)
        {}

        TFilledBoxWidget(const TFilledBoxWidget& widToCopy) :
            TBoxWidget(widToCopy)
            , __rgbFill(widToCopy.__rgbFill)
        {}

        ~TFilledBoxWidget() {}


        // --------------------------------------------------------------------
        //  Public operators
        // --------------------------------------------------------------------
        TFilledBoxWidget& operator=(const TFilledBoxWidget& widToAssign)
        {
            TParent::operator=(widToAssign);
            __rgbFill = widToAssign.__rgbFill;
            return *this;
        }

        // --------------------------------------------------------------------
        //  Public, inherited methods
        // --------------------------------------------------------------------
        TTextStream& Draw(TTextStream& strmToWriteTo) const
        {
            TParent::Draw(strmToWriteTo);
            strmToWriteTo << L", Fill Color: " << __rgbFill;
            return strmToWriteTo;
        }


    protected   :
        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TFilledBoxWidget() :

            __rgbFill(0xFF, 0xFF, 0xFF)
        {}


        // --------------------------------------------------------------------
        //  Protected, implemented methods
        // --------------------------------------------------------------------
        tCIDLib::TVoid _StreamFrom(TBinaryStream& strmToReadFrom)
        {
            TParent::_StreamFrom(strmToReadFrom);
            strmToReadFrom >> __rgbFill;
        }

        tCIDLib::TVoid _StreamTo(TBinaryStream& strmToWriteTo) const
        {
            TParent::_StreamTo(strmToWriteTo);
            strmToWriteTo << __rgbFill;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __rgbFill
        //      The fill color to use
        // --------------------------------------------------------------------
        TRGBClr     __rgbFill;

        // --------------------------------------------------------------------
        //  Do any needed macros
        // --------------------------------------------------------------------
        RTTIMacros(TFilledBoxWidget,TBoxWidget)
        DefPolyDup(TFilledBoxWidget)
        BefriendFactory(TFilledBoxWidget)
};
