//
//  FILE NAME: Streams3.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/01/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the third of the streaming oriented demo
//  programs. This one demonstrates the magic of polymorphic stream.
//
//  In order to demonstrate this, we need a set of classes that are all
//  derived from a base class. In order to demonstrate the concept fully,
//  both from the streaming side and from the perspective of writing classes
//  that are streamable and support the required RTTI stuff, we create a
//  small family of classes for testing.
//
//  Streams3_Widgets.Hpp implements the family of classes that we use for
//  testing. They are very simple and are just done inline. They are not
//  full fledged classes so they don't deal with all the issues of unimplemented
//  constructors and operators and such.
//
//  This module creates a file stream and streams the objects out to the
//  stream. It then streams them back in, demonstrating the ability to figure
//  out from the stream itself what is in the data stream, create the objects
//  to stream them back in using dynamic typing, and then to stream them in.
//
//  This all happens behind the scenes, so its a very convenient and powerful
//  way to make a heterogenous collection of objects persistent. Managing a
//  heterogenous group of objects via a base class happens all the time from
//  a language perspective. Polymorphic streaming extends that level of
//  abstraction to persistent storage.
//
//  We also demonstrate here a 'by reference' collection which is used to
//  manage the objects. In order to put pointers into a collection we have
//  to use a 'pointer wrapper' class, TCntPtr in this case, which will
//  reference count and mange the data pointer for us.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Include our main header, which includes anything else we need
// ----------------------------------------------------------------------------
#include    "Streams3.Hpp"


// ----------------------------------------------------------------------------
//  Since all of our test widget classes are inline, they don't have their
//  own Cpp file. So we do their RTTI macros for them here. Note that we
//  use RTTIData2, which is the version that enables dynamic type creation.
//  TBaseWidget does not enable this because it is an abstract base class and
//  cannot be directly created.
// ----------------------------------------------------------------------------
RTTIData(TBaseWidget,TObject)
RTTIData2(TLineWidget,TBaseWidget)
RTTIData2(TCircleWidget,TBaseWidget)
RTTIData2(TBoxWidget,TBaseWidget)
RTTIData2(TFilledBoxWidget,TBoxWidget)


// ----------------------------------------------------------------------------
//  Local types
//
//  TWidgetPtr
//  TBagOWidgets
//      This is a convenience typedef for our collection templates. We have
//      to create a counted pointer for a TBaseWidget pointer and a bag of
//      those counted pointers.
// ----------------------------------------------------------------------------
typedef TCntPtr<TBaseWidget>    TWidgetPtr;
typedef TBag<TWidgetPtr>        TBagOWidgets;


// ----------------------------------------------------------------------------
//  Local static data
//
//  __rgbBlack
//  __rgbWhite
//      Some convenience colors to use in our test.
//
//  __strTestFile
//      This is the name of the file that we use as the backing for the file
//      stream that we test.
// ----------------------------------------------------------------------------
static const TRGBClr    __rgbBlack(0,0,0);
static const TRGBClr    __rgbWhite(0xFF,0xFF,0xFF);
static const TString    __strTestFile(L"TestStreamFile.Dat");


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Streams3MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __FillBag
//
// DESCRIPTION:
//
//  Since this program has no user interaction, we need to have some widgets
//  to use, as though they were entered by a user in a real drawing program.
//  So this guy just puts some widgets into the passed collection.
// ---------------------------------------
//   INPUT: colWidgets is the bag of widgets that we fill in.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __FillBag(TBagOWidgets& colWidgets)
{
    // A line from 10,10 to 100,100
    colWidgets.Add
    (
        TWidgetPtr(new TLineWidget(TPoint(10, 10), TPoint(100, 100)))
    );

    // An unfilled box from 50,50 to 64,92
    colWidgets.Add
    (
        TWidgetPtr(new TBoxWidget(TPoint(50,50), TPoint(64,92)))
    );

    // A line from 200,150 to 210,155
    colWidgets.Add
    (
        TWidgetPtr(new TLineWidget(TPoint(200, 150), TPoint(210, 155)))
    );


    // A filled box from 500,50 to 525,100, white filled
    colWidgets.Add
    (
        TWidgetPtr
        (
            new TFilledBoxWidget(TPoint(500,500), TPoint(525,100), __rgbWhite)
        )
    );

    // A filled box from 50,100 to 55,110, black filled
    colWidgets.Add
    (
        TWidgetPtr
        (
            new TFilledBoxWidget(TPoint(50,100), TPoint(55,110), __rgbBlack)
        )
    );

    // A line from 320,480 to 20, 15
    colWidgets.Add
    (
        TWidgetPtr(new TLineWidget(TPoint(320, 480), TPoint(20, 15)))
    );
}


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. It creates the file
//  stream, streams out the objects 'drawing' them as it goes, then streams
//  them back in and 'draws' them again. This should cause the exact same
//  output as the originals.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and partly a testing program, we'd like to
    //  catch all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        //
        //  Create a file based binary stream to use as the test stream
        //  in this program.
        //
        TBinaryFileStream strmFileTest
        (
            __strTestFile
            , tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        // Create our bag that we will use to hold the widgets
        TBagOWidgets colWidgets;

        //
        //  Since we have no user interface, just call a function that will
        //  fill in our bag of widgets with some test objects.
        //
        __FillBag(colWidgets);

        //
        //  Now we can create a cursor and iterate the bag. We polymorphically
        //  stream each one out. This uses a template method that understands
        //  now to write out the correct type information with the object.
        //
        TBagOWidgets::TCursor cursWidgets(&colWidgets);
        __conOut << DNewLn << L"Writing and 'drawing' objects" << NewLn;
        do
        {
            //
            //  Note the syntax. The current object is the counted pointer
            //  object, not the actual object pointer. So from the current
            //  object we get the actual data object pointer.
            //
            ::PolymorphicWrite(cursWidgets.objCur().pobjData(), strmFileTest);

            // Tell the one we just wrote to 'draw' itself
            cursWidgets.objCur().objData().Draw(__conOut);
            __conOut << NewLn;

        }   while (cursWidgets.bNext());

        // Remember how many we wrote
        const tCIDLib::TCard4 c4CountWritten = colWidgets.c4ElemCount();

        // Flush the collection now so that the original objects are destroyed
        colWidgets.Flush();

        //
        //  Reset the stream to its start and start streaming in. We just
        //  stream until we hit the end of the stream.
        //
        strmFileTest.Reset();
        while (!strmFileTest.bEndOfStream())
        {
            // A base widget pointer to read objects back into
            TBaseWidget* pwidRead;

            //
            //  Now we use the polymorphic reading template function that
            //  understands now to find out what the next object is, create
            //  an object of that type using dynamic type creation, then
            //  to stream into that object.
            //
            //  We get back a dynamically allocated object that can be put
            //  into a counted pointer object and into the collection.
            //
            ::PolymorphicRead(pwidRead, strmFileTest);

            // Put it into the collection
            colWidgets.Add(TWidgetPtr(pwidRead));
        }

        //
        //  Just to show you how PolymorphicRead can do what it does, here
        //  is small example of dynamic typing. Because we used the
        //  RTTIData2() macros, above, for our widgets classes, they support
        //  dynamic creation. So we can create a new object by name. It will
        //  be default constructed (though it can be arranged to have the
        //  factory creation not be via the default constructor.)
        //
        //  Trace into this code to see how it works.
        //
        TLineWidget* pwidByName;
        MakeNewOfType(pwidByName, L"TLineWidget");

        // Just to prove its there, do an assignment to it
        *pwidByName = TLineWidget(TPoint(1,1), TPoint(2,2));

        // Now delete it
        delete pwidByName;

        //
        //  Anyway, back to our regularly scheduled program. Make sure that
        //  we streamed back in the number of elements we wrote out before,
        //  just as a basic sanity check.
        //
        if (colWidgets.c4ElemCount() != c4CountWritten)
        {
            __conOut << L"Read in element count != to written out count"
                     << NewLn;
            return tCIDLib::EExit_FatalError;
        }

        //
        //  Now reset the cursor so we can iterate the collection again
        //  and 'draw' the objects once more. The output should be exactly
        //  the same as before.
        //
        //  Here again, sanity check and make sure that the reset call returns
        //  true, which indicates that there are elements to be cursored.
        //
        if (!cursWidgets.bReset())
        {
            __conOut << L"Widget collection cursor reset returned false!"
                     << NewLn;
            return tCIDLib::EExit_FatalError;
        }

        //
        //  We have the same number of elements so iterate them and 'draw'
        //  each one. Note that, here again, the element is the counted
        //  pointer, so we go the next level and get the object out of it
        //  because that's what we want to call the Draw() method on.
        //
        __conOut << DNewLn << L"'Drawing' read in objects" << NewLn;
        do
        {
            cursWidgets.objCur().objData().Draw(__conOut);
            __conOut << NewLn;
        }   while (cursWidgets.bNext());
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
