//
//  FILE NAME: Streams2.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the second of the streaming oriented demo
//  programs. This one just demonstrates the basics of binary streaming.
//
//  This guy has a local function that will stream a number of objects to
//  a passed binary stream. It prints out the number of byte used after
//  each object is streamed. Then it resets the stream and streams the
//  object back into different objects and compares them to the originals
//  to make sure that they came back in correctly.
//
//  The main program creates a memory based binary stream, and passes
//  it to the function for testing. Then it creates a file based binary
//  stream and passes it in also. These shoold have the exact same results
//  except one goes to a file and another goes to a memory buffer,
//  demonstrating the flexibility of streams.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This program is very simple so it does not attempt to be language
//      independent and it does not provide its own facility object since
//      it does not need one.
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Local static data
//
//  __strTestFile
//      This is the name of the file that we use as the backing for the file
//      stream that we test.
// ----------------------------------------------------------------------------
static const TString    __strTestFile(L"TestStreamFile.Dat");


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
//
//  This tells CIDLib which thread is the primary program thread. This is the
//  only thread started automatically. All others are started manully when
//  desired or required. One, and only one, module of the program must do this
//  in order to have a meaningful program.
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Streams2MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __StreamTo
//
// DESCRIPTION:
//
//  This is a testing function. It takes a stream (any derivative of the
//  binary stream class) and does some basic binary streaming operations
//  on it. It stream out some objects, then streams them back into another
//  set of objects. It then compares the two sets to make sure that they
//  came back in correctly.
// ---------------------------------------
//   INPUT: strmToTest is the binary stream derivative to test.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __StreamTo(TBinaryStream& strmToTest)
{
    //
    //  Ok, So lets create a set of objects that we can stream to this
    //  new stream. All of these classes implement the MStreamable mixin
    //  interface. The << and >> operators are written in terms of this
    //  mixin, so there is just one set of these operators that work for
    //  all streamable classes.
    //
    TPoint      pntTest1(1,1);
    TPoint      pntTest2(100, 100);
    TArea       areaTest1(10, 10, 64, 92);
    T3DVector   vecTest1(1.0, 2.0, 3.0);
    TRGBClr     rgbTest1(0xFF, 0xAC, 0xCC);

    //
    //  Ok, stream them puppies into the stream buffer. After each one,
    //  just for fun, print out the size of the memory stream data, which
    //  is just the current stream position.
    //
    strmToTest << pntTest1;
    __conOut    << L"Streamed pntTest1. Stream data bytes="
                << strmToTest.fposCurPos() << NewLn;

    strmToTest << pntTest2;
    __conOut    << L"Streamed pntTest2. Stream data bytes="
                << strmToTest.fposCurPos() << NewLn;

    strmToTest << areaTest1;
    __conOut    << L"Streamed areaTest1. Stream data bytes="
                << strmToTest.fposCurPos() << NewLn;

    strmToTest << vecTest1;
    __conOut    << L"Streamed vecTest1. Stream data bytes="
                << strmToTest.fposCurPos() << NewLn;

    strmToTest << rgbTest1;
    __conOut    << L"Streamed rgbTest1. Stream data bytes="
                << strmToTest.fposCurPos() << NewLn;

    //
    //  Now, lets stream them back in and compare them. First we have
    //  to reset the stream back to the start.
    //
    strmToTest.Reset();

    //
    //  Create a new set of objects with different names so that we
    //  compare them to the originals.
    //
    TPoint      pntRead1;
    TPoint      pntRead2;
    TArea       areaRead1;
    T3DVector   vecRead1;
    TRGBClr     rgbRead1;

    // And stream them puppies back in
    strmToTest  >> pntRead1
                >> pntRead2
                >> areaRead1
                >> vecRead1
                >> rgbRead1;

    //
    //  Compare each object with its original. Any differences is an
    //  error in the streaming system (or more likely in the _StreamTo()
    //  and _StreamFrom() methods of the class that fails.)
    //
    if (pntRead1 != pntTest1)
        __conOut << L"First TPoint object failed comparison\r\n";

    if (pntRead2 != pntTest2)
        __conOut << L"Second TPoint object failed comparison\r\n";

    if (areaRead1 != areaTest1)
        __conOut << L"First TArea object failed comparison\r\n";

    if (vecRead1 != vecTest1)
        __conOut << L"First T3DVector object failed comparison\r\n";

    if (rgbRead1 != rgbTest1)
        __conOut << L"First TRGBClr object failed comparison\r\n";
}

// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. This guy just
//  creates two different types of binary streams and passes them both to
//  the __StreamTo() testing method above.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and partly a testing program, we'd like to
    //  catch all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        //
        //  Create a memory buffer based stream. This class is just a simple
        //  derivative of TBinaryStream that handles the mechanics of setting
        //  it up with a binary memory stream implementation object. You can
        //  easily do this manually, but this is simpler and safe.
        //
        //  The parameters are the initial size, and maximum growth size, and
        //  the access rights. We let the access rights default, which is to
        //  read/write access.
        //
        //  It uses a TSysBuf buffer because memory streams can sometimes
        //  be allowed to grow pretty large. Using a system buffer is more
        //  efficient in this case because it can just commit pages as the
        //  stream grows. But it means the minimum buffer size is a host
        //  memory page. We set the initial allocation to 4096 and the maximum
        //  growth size to 8192.
        //
        TBinaryMBufStream strmMemTest(4096, 8192);

        //
        //  Call the test function with this stream and let it do the streaming
        //  tests on it.
        //
        __conOut << L"Streaming to the memory stream..." << NewLn;
        __StreamTo(strmMemTest);
        __conOut << DNewLn;


        //
        //  Now lets create a file based binary stream. As with the memory
        //  stream above, there is a convenience class that sets up a binary
        //  stream with a file based binary stream implementation object.
        //
        TBinaryFileStream strmFileTest
        (
            __strTestFile
            , tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        //
        //  Now pass this one to the same test function. The test function
        //  does not know or care what kind of stream it is. So this one
        //  should have the same results.
        //
        __conOut << L"Streaming to the file stream..." << NewLn;
        __StreamTo(strmFileTest);
        __conOut << DNewLn;
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
