//
//  FILE NAME: Streams1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first of the streaming oriented demo
//  programs. This one just demonstrates the basics of text streaming.
//
//  This guy opens a test file, SourceTestFile.Text, which consists of a
//  list of words and their definitions. They are in this form:
//
//      Word - Definition text
//
//  The file is read a line at a time and then the word part and the
//  definition text part are pulled out into separate strings. Then this
//  information is output in a simple columnar report format.
//
//  The output is sent to the standard output, via the console object that
//  is also used for all the other output.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This simple program does not attempt to be language independent.
//
//  2)  This program also does not create a facility object of its own because
//      its so simple that it has no need for one.
//
//  3)  It is assumed that this program is run from the project directory,
//      where the test source file is. If not, it displays an error that it
//      cannot find the file.
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local constant data
//
//  __strTestFile
//      This is the name of the test file that we read in our test text
//      from.
// ----------------------------------------------------------------------------
static const TString    __strTestFile(L"SourceTestFile.Text");


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
//
//  This tells CIDLib what the main thread of the program is. This is the
//  only thread object that is run automatically. All others are started
//  manually when required or desired.
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Streams1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __DoWork
//
// DESCRIPTION:
//
//  This method is called to do the work, from the main thread function
//  below.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __DoWork()
{
    // See if the file exists
    if (!TFileSys::bExists(__strTestFile))
    {
        __conOut    << L"The test source file " << __strTestFile
                    << L" was not found" << NewLn;
        return;
    }

    //
    //  First of all lets see if we can find and open our test file. We
    //  use the TTextFileStream class which is a convenience class. It
    //  is a text stream which understand how to use a file as its data
    //  sink/source.
    //
    TTextFileStream strmSource
    (
        __strTestFile
        , tCIDLib::ETextFmt_ASCII
        , tCIDLib::EAccess_Read
        , tCIDLib::ECreateAct_OpenIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    //
    //  Create two stream format objects. One is for the descriptive
    //  word column and one is for the descriptive text column.
    //
    TStreamFmt  strmfWord(24, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);
    TStreamFmt  strmfDescription(0, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);

    //
    //  Now output the header lines. We putout a title and then underline
    //  them.
    //
    __conOut    << DNewLn
                << strmfWord << L"Term"
                << strmfDescription << L"Description of Term" << NewLn
                << strmfWord << L"------------"
                << strmfDescription << L"-------------------" << NewLn;

    //
    //  Ok, so lets read in each line. If its not empty, then we parse it
    //  to get out the term and its description. We then output the term
    //  and its text in the same way that we just did the titles.
    //
    TString strInput;
    TString strTerm;
    TString strText;
    while (!strmSource.bEndOfStream())
    {
        // Get a line of text
        strmSource.c4GetLine(strInput);

        //
        //  Now break out the two parts, they are separated by a dash.
        //
        strInput.bStripToken(strTerm, 0, L'-');
        strInput.bStripToken(strText, 1, L'-');

        // Strip any whitespace off them
        strTerm.StripWhitespace();
        strText.StripWhitespace();

        // And output them
        __conOut    << strmfWord << strTerm
                    << strmfDescription << strText << NewLn;
    }
}


//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and partly a testing program, we'd like to
    //  catch all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        __DoWork();
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my test programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
