//
//  FILE NAME: RTTI1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/08/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first of the RTTI oriented demo
//  programs. This one does not do anything realistic but just shows all of
//  of the things you can do with the RTTI system.
//
//  In order to make it obvious what is happening, a small class is defined
//  here that implements all of the standard RTTI stuff. The class is trivial
//  so that it's implementation does not cloud the picture. The class is
//  very trivial so its just done all inline.
//
//  RTTI enabled classes must:
//
//  1)  Derive from TObject.
//  2)  Use the RTTIMacros() macro in the class definition.
//  3)  Use the RTTIData() or RTTIData2() macros in their Cpp file.
//  4)  Support a default constructor. The constructor can be protected, but
//          the class then has to us the BefriendFactory() macro so that the
//          factory method can access the default constructor.
//  5)  Although its not necessary, supporting polymorphic duplication is
//          a plus by supporting the MDuplicable interface.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"RTTI1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//   CLASS: TTestClass
//  PREFIX: test
// ----------------------------------------------------------------------------
class TTestClass : public TObject, public MDuplicable
{
    public  :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TTestClass(const tCIDLib::TInt4 i4Value) :

            __i4Value(i4Value)
        {
        }

        TTestClass(const TTestClass& testToCopy) :

            __i4Value(testToCopy.__i4Value)
        {
        }

        ~TTestClass() {}


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        // --------------------------------------------------------------------
        tCIDLib::TInt4 i4Value() const
        {
            return __i4Value;
        }


    protected  :
        // --------------------------------------------------------------------
        //  Hidden constructors
        // --------------------------------------------------------------------
        TTestClass() :

            __i4Value(kCIDLib::i4MaxInt)
        {
        }

    private :
        // --------------------------------------------------------------------
        //  Private data
        //
        //  __i4Value
        //      Our data member
        // --------------------------------------------------------------------
        tCIDLib::TInt4      __i4Value;


        // --------------------------------------------------------------------
        //  Do needed macros. Always just put them at the end of the class
        //  definition because they modify the public/protected/private
        //  scope at will.
        //
        //  BefriendFactory
        //      This method lets us support factory creation (which requires
        //      a default constructor) without exposing our default
        //      constructor.
        //
        //  DefPolyDup
        //      Does the default polymorphic duplication, which is just a
        //      call to our own copy constructor to new up a copy and return
        //      it. This implements the pobjDuplicate() method which we get
        //      from MDuplicable.
        //
        //  RTTIMacros()
        //      Indicates our class and our parent class. It also implements
        //      the standard RTTI methods: clsIsA(), clsParent(), and
        //      bIsDescendantOf().
        // --------------------------------------------------------------------
        BefriendFactory(TTestClass)
        DefPolyDup(TTestClass)
        RTTIMacros(TTestClass,TObject)
};

// ----------------------------------------------------------------------------
//  Do our out of line macros. We want to support factory creation (or dynamic
//  creation) so we use RTTIData2().
// ----------------------------------------------------------------------------
RTTIData2(TTestClass,TObject)


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. This guy just does
//  a lot of RTTI stuff.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Ok, lets get started. First all all, any RTTI enabled class has
    //  a const static member of TClass that is the 'meta class' for that
    //  class. So lets dump it out to the output console.
    //
    __conOut << L"Our type is: " << TTestClass::clsThis << NewLn;

    //
    //  Another way to get to the same place, and to do it polymorphically,
    //  is the clsIsA() method. So lets create a TTestClass object and then
    //  get another pointer to it via the base TObject class.
    //
    TTestClass  testObj(1);
    TObject*    pobjTest = &testObj;

    //
    //  So we can do a dump of the class via the object or via the pointer
    //  and get the same result. clsIsA() always returns the REAL type of
    //  class, not of the type its accessed through.
    //
    __conOut << L"Through testObj: " << testObj.clsIsA() << NewLn;
    __conOut << L"Through pobjTest: " << pobjTest->clsIsA() << NewLn;

    //
    //  We can also get the class type of our parent class, which returns
    //  its class object.
    //
    __conOut << L"testObj's parent class is: " << testObj.clsParent() << NewLn;

    //
    //  And of course we can access the static object via the object but
    //  accessing it via a base class will get the static object of the
    //  base class.
    //
    __conOut << L"Through testObj.clsThis: " << testObj.clsThis << NewLn;
    __conOut << L"Through pobjTest->clsThis: " << pobjTest->clsThis << NewLn;

    //
    //  We can compare the types of an object to see if its of a particular
    //  type.
    //
    if (testObj.clsIsA() == TTestClass::clsThis)
    {
        __conOut << L"Yes, testObject is a " << TTestClass::clsThis << NewLn;
    }

    //
    //  We can test for descendence from a particular class. So lets do one
    //  that's true and one that is not.
    //
    if (testObj.bIsDescendantOf(TObject::clsThis))
    {
        __conOut << L"Yes, testObj is descended from " << TObject::clsThis
                 << NewLn;
    }

    if (!testObj.bIsDescendantOf(TArea::clsThis))
    {
        __conOut << L"No, testObj is not descended from " << TArea::clsThis
                 << NewLn;
    }

    //
    //  We can also do somethings using just the type name at runtime.
    //  For instance we can get the class object for a class. This stuff
    //  is done via static methods of the TClass class.
    //
    TClass clsTmp = TClass::clsForType(L"TTestClass");
    __conOut << L"The dynamically gotten class is: " << clsTmp << NewLn;

    //
    //  We can create a new object of a type that supports dynamic creation.
    //  First we will use the rawest mechanism which returns it as a TObject
    //  object.
    //
    TObject* pobjDynamic = TClass::pobjMakeNewOfType(L"TTestClass");
    __conOut    << L"Dynamically created object is of type: "
                << pobjDynamic->clsIsA() << NewLn;
    delete pobjDynamic;

    //
    //  There is a template method that does a lot of work for you, taking
    //  a pointer, making sure that the pointer can legally point to an
    //  object of the type to be created, and creating it, and casting the
    //  returned object to the desired type.
    //
    TTestClass* ptestDynamic;
    MakeNewOfType(ptestDynamic, L"TTestClass");
    __conOut    << L"Dynamically created object is of type: "
                << ptestDynamic->clsIsA() << NewLn;

    //
    //  You can do a dynamic cast test that will return 0 if the cast is
    //  not valid, or the object cast as the new type if the cast is legal.
    //  This is in the style used by the standard C++ RTTI.
    //
    pobjDynamic = pDynamicCast<TObject>(ptestDynamic);
    if (!pobjDynamic)
        __conOut << L"Could not cast ptestDynamic to pobjDynamic" << NewLn;

    // Clean up the temp object
    delete ptestDynamic;

    //
    //  We can test whether it would be valid to cast an and object to a
    //  particular type. If it fails it will throw a kCIDErrs::errcRTTI_BadCast
    //  exception.
    //
    tCIDLib::TBoolean bCaughtIt = kCIDLib::False;
    try
    {
        TArea areaTmp;
        TClass::TestCanCast(areaTmp, L"TTestClass");
    }

    catch(const TError& errToCatch)
    {
        if (errToCatch.bCheckError(facCIDLib, kCIDErrs::errcRTTI_BadCast))
            bCaughtIt = kCIDLib::True;
    }

    if (!bCaughtIt)
        __conOut << L"Failed to catch the bad cast error" << NewLn;
    else
        __conOut << L"Caught the expected bad cast error" << NewLn;

    return tCIDLib::EExit_Normal;
};
