//
//  FILE NAME: Process3.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 09/13/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the third process control oriented demo
//  programs. This program demonstrates the execution of another process
//  and waiting for it to die.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes
//
//  We don't have a main header of our own for this simple program, so just
//  include the CIDLib facility header, which is the only one we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class (derived from TTextStream.)
// ----------------------------------------------------------------------------
static TConsole         __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
//
//  This must be done in one (and only one) module of the client program.
//  It tells CIDLib which is the main thread of the program.) GUI apps use
//  a similar, but differently named, mechanism.
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Process3MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. It just starts the
//  external process and waits for it to die, printing out a little info
//  on it in the meantime.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: EExit_Normal if things go ok. If the worker thread returns a
//              non-normal return, we return that.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Start up the worker thread and just wait for it to die. Indicate
    //  what we are doing to the user.
    //
    __conOut << NewLn << L"Starting Process2 program..." << NewLn;

    //
    //  Create an external process object. We just let it inherit our
    //  environment, so all we have to provide is the program name.
    //
    TExternalProcess extpToRun(L"Process2.Exe");

    // And now start it up
    __conOut << L"Waiting for Process2 to die..." << NewLn;
    extpToRun.Start();

    tCIDLib::EExitCodes eExit = extpToRun.eWaitForDeath();

    // Display the exit code
    __conOut << L"Process2 returned exit code: " << eExit << DNewLn;

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
