//
//  FILE NAME: Process1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first process/thread oriented demo
//  programs. This one just dumps the process registry information. The
//  process registry is the list of currently running CIDLib processes. Each
//  one of them is magically entered into a registry, which can be accessed
//  by other CIDLib apps.
//
//  This demo program is set up to be language independent so all of its
//  message and error text is in the Process1.MsgText file.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "Process1.Hpp"



// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class (derived from TTextStream)
//      that supports fancy user input (command editing and recall.) We don't
//      use that capability here but use this TConsole anyway since its
//      designed specifically to deal with the console without any effort on
//      our part.
//
//  __facProcess1
//      Our facility object, so we can get to our facility's messages (and
//      other resources, though messages are all we need here.) Since we have
//      no specific needs, we just create a TFacility object. Like all of the
//      CIDLib code, we set its version to the current CIDLib release version.
// ----------------------------------------------------------------------------
TConsole        __conOut;
TFacility       __facProcess1
                (
                    L"Process1"
                    , tCIDLib::EModType_Exe
                    , kCIDLib::c4MajVersion
                    , kCIDLib::c4MinVersion
                );



// ----------------------------------------------------------------------------
//  Do the magic main module code
//
//  This must be done in one (and only one) module of the client program.
//  It tells CIDLib which is the main thread of the program.) A GUI program
//  has a similar mechanism, but uses a different macro. Any value thread
//  constructor can be provided here. This one just indicates the thread name
//  and provides a thread function to run.
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Process1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. It just gets a
//  copy of the registry and dumps out the information.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: EExit_Normal, just because we are supposed to return something.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    //
    //  We have to let our calling thread go first. In many cases, when one
    //  thread starts another, it needs to provide information to that
    //  thread and be sure that the started thread has gotten it before
    //  continuing. So new threads are start up such that they are 'synced'
    //  with the 'calling' thread. So the calling thread is blocked until
    //  the new thread calls Sync() to let it go. This way, if the new
    //  thread needs to do any work, it can do that first then let the
    //  calling thread go. But, be sure to catch any exceptions that might
    //  happen during init and be sure the calling thread gets let go or
    //  it will timeout and throw an exception.
    //
    thrThis.Sync();

    //
    //  Set the processes state to up and running. Each program has an field
    //  in its registry entry that indicates its basic state. The first two
    //  (initializing, and entering the main thread) are done for you, as the
    //  last one (dead.) The others you can set to indicate the major milestones
    //  of your program.
    //
    //  This one should be set once the main thread finishes any setup and
    //  starts doing its real work.
    //
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Output a little blurb. Note the use of the L prefix! All raw strings
    //  must be UNICode strings when dumped to a text stream.
    //
    __conOut    << NewLn << __facProcess1.strMsg(kProc1Msgs::midBlurb)
                << DNewLn;

    //
    //  Set up some stream formats. Our output has these columns
    //
    //      name, handle, id, info card, info line
    //
    //  The handle, info card, and id can all use the same one since they
    //  are all just numeric values. The name and needs one and the info
    //  card does not since its just trailing.
    //
    //  Stream format objects allow us to create specific groupings of
    //  text stream formats that can be manipulated as a whole and set on the
    //  text stream as a whole by just streaming it into the stream.
    //
    TStreamFmt  strmfName(32, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);
    TStreamFmt  strmfNumbers(8, 0, tCIDLib::EHJustify_Right, kCIDLib::chSpace);

    //
    //  Dump out the header lines for each of the colums of info that we
    //  dump for each process. We use ESpaces(2) to put two spaces between
    //  each column. If use just dumped "  ", it would be formatted and
    //  justified and turn into either 32 or 8 spaces according to which of
    //  the formatting schemes was on the stream at the time.
    //
    __conOut    << strmfName
                << __facProcess1.strMsg(kProc1Msgs::midCol1Title)
                << TTextStream::ESpaces(2)
                << strmfNumbers
                << __facProcess1.strMsg(kProc1Msgs::midCol2Title)
                << TTextStream::ESpaces(2)
                << __facProcess1.strMsg(kProc1Msgs::midCol3Title)
                << TTextStream::ESpaces(2)
                << TTextStream::strmfDefault
                << __facProcess1.strMsg(kProc1Msgs::midCol4Title)
                << TTextStream::ENewLine;

    __conOut    << strmfName
                << L"----------------" << TTextStream::ESpaces(2)
                << strmfNumbers
                << L"--------"  << TTextStream::ESpaces(2)
                << L"--------" << TTextStream::ESpaces(2)
                << TTextStream::strmfDefault
                << L"--------------------"
                << TTextStream::ENewLine;

    //
    //  Now we can get a collection of the available process registry
    //  entries. This is the best way to get the whole list because it
    //  minimizes the time that the registry is locked, and insures that
    //  the registry state does not change while we work our way through
    //  it.
    //
    //  However, if you are willing to work at the CIDKernel facility level,
    //  you can lock the registry directly and putz with it.
    //
    TBag<TRegProcessInfo>   colProcs;
    if (!TProcessRegistry::c4GetRegisteredProcs(colProcs))
    {
        //
        //  This should never happen since this program itself should be
        //  in the registry at a minimum, but check for it anyway.
        //
        __conOut << __facProcess1.strMsg(kProc1Msgs::midEmptyRegistry)
                 << DNewLn;

        // Return a standard error exit code
        return tCIDLib::EExit_FatalError;
    }

    //
    //  Now we can dump the info. We create a cursor for the collection
    //  and iterate it.
    //
    TBag<TRegProcessInfo>::TCursor  cursProcs(&colProcs);

    //
    //  Reset the cursor so that we can iterate the process info records.
    //  We don't really need to check the return (which is kCIDLib::True)
    //  if the collection has any elements), since we checked above. But,
    //  here again, for testing purposes, do it anyway.
    //
    if (cursProcs.bReset())
    {
        do
        {
            // Dump out the information
            __conOut    << strmfName
                        << cursProcs.objCur().strName()
                        << TTextStream::ESpaces(2)
                        << strmfNumbers
                        << cursProcs.objCur().pidThis()
                        << TTextStream::ESpaces(2)
                        << TCardinal
                           (
                            cursProcs.objCur().c4InfoCard()
                            , tCIDLib::ERadix_Hex
                           )
                        << TTextStream::ESpaces(2)
                        << TTextStream::strmfDefault
                        << cursProcs.objCur().strInfoLine()
                        << TTextStream::ENewLine;

        }   while (cursProcs.bNext());
    }
     else
    {
        // Report the error to the user
        __conOut << __facProcess1.strMsg(kProc1Msgs::midEmptyBag)
                 << DNewLn;

        // Return a standard error exit code
        return tCIDLib::EExit_FatalError;
    }

    //
    //  Set the processes state to terminating. This indicates that the
    //  program is done with its application specific work and is exiting.
    //  Any other states are set for you by the CIDLib code as things
    //  unwind, ending up with a status of dead.
    //
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    // Just return normal since things went ok
    return tCIDLib::EExit_Normal;
}
