//
//  FILE NAME: MiniMetrics.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 03/23/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the program. It contains all of the code
//  since this is such a simple app. This program is a raw Win32 app that
//  is used to view CIDLib metrics. There is a need for a such a simple
//  metrics viewer so that there is a way to view metrics even if the CIDLib
//  system itself is not in a particularly healthy state. Other, more
//  elaborate, metrics programs can be built using CIDLib itself and made
//  available for end users.
//
//  Note that we can use CIDKernel services here, so we can stay somewhat
//  more portable and avoid replicating a lot of stuff that is already
//  done in the kernel, but it still avoids dependence on CIDLib and other
//  high level DLLs.
//
//
//  CAVEATS/GOTCHAS:
//
//  1)  This is a VERY BAD example of anything stylistic. By necessity it
//      makes use of CIDLib structures and types in some cases, but other
//      wise it is a straight Win32 app. So its ugly, but its kind of a
//      unique example.
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    <windows.h>
#include    "CIDKernel.Hpp"
#include    "MiniMetrics_ResourceIds.H"


// -----------------------------------------------------------------------------
//  Local data types
//
//  TMetricSel
//      This is a small structure used to return info from the new session
//      selection dialog.
// -----------------------------------------------------------------------------
struct  TMetricSel
{
    tCIDLib::TCard4 c4ProcInd;
    tCIDLib::TCard4 c4GroupInd;
};


// ----------------------------------------------------------------------------
//  Local, constant data
// ----------------------------------------------------------------------------
const COLORREF              __rgbWindowBgn = RGB(128,128,128);
const tCIDLib::Tch* const   __pszAppName = L"MiniMetrics.Exe";
const tCIDLib::Tch* const   __pszAppTitle = L"CIDLib Basic Metrics Viewer";


// ----------------------------------------------------------------------------
//  Local data
//
//  __bSessionOpen
//      Indicates whether a session is currently open. If it is True, then
//      __preCurrent, __pkmtrCurrent, and __c4GroupIndex are all valid.
//
//  __c4GroupIndex
//      This is the index of the group that we are currently monitoring. Its
//      only valid if __bSessionOpen is True.
//
//  __hmodThis
//      The program instance passed by the system to our main
//
//  __hprocCurrent
//      The process handle of the current process. When none, its set to
//      kCIDLib::hprocInvalid.
//
//  __hwndMain
//      The handle of the main window.
//
//  __pkmtrCurrent
//      A pointer to the raw metrics directory structure that we are
//      monitoring.
//
//  __pkmtxProcReg
//      This is a kernel mutex object used to lock the process registry.
//      It is named so we just open our own copy.
//
//  __preCurrent
//      This is a copy of the process registry entry for the process we
//      monitoring, if any. Only valid if __bSesssionOpen is True.
// ----------------------------------------------------------------------------
static tCIDLib::TBoolean            __bSessionOpen;
static tCIDLib::TCard4              __c4GroupIndex;
static tCIDLib::TModHandle          __hmodThis;
static tCIDLib::TProcessHandle      __hprocCurrent;
static HWND                         __hwndMain;
static TKrnlMetricDirectory*        __pkmtrCurrent;
static TKrnlMutex*                  __pkmtxProcReg;
static tCIDLib::TRawProcessEntry    __preCurrent;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static BOOL CALLBACK __bMainWndProc
(
            HWND                hwndThis
    ,       UINT                uMsg
    ,       WPARAM              wParam
    ,       LPARAM              lParam
);

static BOOL CALLBACK __bNewSessionProc
(
            HWND                hwndThis
    ,       UINT                uMsg
    ,       WPARAM              wParam
    ,       LPARAM              lParam
);

static VOID __CloseSession
(
            HWND                hwndDlg
);

static VOID __InitControls
(
            HWND                hwndDlg
);

static VOID __LoadGroups
(
    const   HWND                hwndDlg
    , const HWND                hwndListBox
    , const tCIDLib::TCard4     c4ProcInd
);

static VOID __LoadMetrics
(
    const   HWND                hwndDlg
    , const HWND                hwndListBox
);

static VOID __LoadProcesses
(
    const   HWND                hwndDlg
    , const HWND                hwndListBox
);

static VOID __ShowMessage
(
            HWND                hwndOwner
    , const tCIDLib::Tch* const pszText
);

static VOID __UpdateMetricInfo
(
            HWND                hwndDlg
    , const tCIDLib::TCard4     c4MetricIndex
);

static VOID __UpdateSessionInfo
(
            HWND                hwndThis
);


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bInitApplication
//
// DESCRIPTION:
//
//  Initializes the application by doing the grunt work setup of
//  registering the window class and creating the main window.
// ---------------------------------------
//   INPUT: hmodThis if the program instance
//
//  OUTPUT: None
//
//  RETURN: TRUE if successful, else FALSE
//
static BOOL __bInitApplication(tCIDLib::TModHandle hmodThis)
{
    // Store instance handle in a global variable
	__hmodThis = hmodThis;

    //
    //  Create our copy of the process registry mutex so that we can
    //  lock it when we need to. First we have to create the name object
    //  so we can build the full name.
    //
    TKrnlRscName krsnProcReg
    (
        L"CIDCorp"
        , L"ProcRegistry"
        , L"MainRegistry"
    );

    // Build the full name to the mutex
    tCIDLib::Tch szTmpName[1024];
    krsnProcReg.BuildFullName
    (
        szTmpName
        , c4MaxBufChars(szTmpName)
        , tCIDLib::ENamedRsc_Mutex
    );

    // And now create the mutex
    __pkmtxProcReg = new TKrnlMutex(szTmpName);
    __pkmtxProcReg->CreateOrOpen(tCIDLib::ELockState_Unlocked);

    // Create the main window, which is a modeless dialog
    __hwndMain = CreateDialog
    (
        hmodThis
        , MAKEINTRESOURCE(DID_MAIN)
        , 0
        , __bMainWndProc
    );

	if (!__hwndMain)
		return (FALSE);

    // It loaded, so show it
    ShowWindow(__hwndMain, TRUE);

    return TRUE;
}


//
// FUNCTION/METHOD NAME: __bMainWndProc
//
// DESCRIPTION:
//
//  This is the window proc for the main program window. Our main window
//  is a modeless dialog box, so we don't pass things on to the default
//  window proc.
// ---------------------------------------
//   INPUT: hwndThis the program's frame window handle
//          uMsg is the message being recieved
//          wParam, lParam are the window parameters.
//
//  OUTPUT: None
//
//  RETURN: 1 if we handle it, else 0.
//
static BOOL CALLBACK __bMainWndProc(HWND        hwndThis
                                    , UINT      uMsg
                                    , WPARAM    wParam
                                    , LPARAM    lParam)
{
    if (uMsg == WM_COMMAND)
    {
        if (LOWORD(wParam) == MID_SESSION_CLOSE)
        {
            __CloseSession(hwndThis);
        }
        else if (LOWORD(wParam) == MID_SESSION_NEW)
        {
            //
            //  See first if there are even any processes registered.
            //  If not, then just show a message and don't both popping
            //  up the dialog.
            //
            if (!TKrnlProcRegistry::c4ProcessCount())
            {
                __ShowMessage
                (
                    hwndThis
                    , L"There are not CIDLib processes currently registred"
                );
            }

            //
            //  Pop up the process/group selection dialog. This guy will,
            //  if the user selects a group, leave the global data
            //  values set up for the selected process and group.
            //
            TMetricSel mselInfo;
            if (DialogBoxParam
            (
                __hmodThis
                , MAKEINTRESOURCE(DID_NEWSESSION)
                , hwndThis
                , __bNewSessionProc
                , (LPARAM)&mselInfo) != DID_NEWSESS_ACCEPT)
            {
                return 1;
            }

            //
            //  If the user selected a group, then let's start monitoring
            //  that metric group. We create a new metric directory object
            //  through which this group will be accessed. And we store
            //  the index of the group to monitor.
            //
            if (__pkmtrCurrent)
            {
                // Clean up any current metric directory object
                __CloseSession(hwndThis);
            }

            try
            {
                // Lock the process registry and get out our process entry
                TKrnlMutexLocker   kmtxlReg(__pkmtxProcReg);
                __preCurrent = TKrnlProcRegistry::preEntryAt(mselInfo.c4ProcInd);

                // Get a handle for this entry
                __hprocCurrent = TKrnlProcRegistry::hprocFromId(__preCurrent.pidThis);

                // Create the new metric directory object
                __pkmtrCurrent = new TKrnlMetricDirectory(__preCurrent.pidThis);

                // And store away the group index we are monitoring
                __c4GroupIndex = mselInfo.c4GroupInd;

                // Load up the available metrics
                __LoadMetrics(hwndThis, GetDlgItem(hwndThis, DID_MAIN_MTRLIST));

                //
                //  Initialize the controls that show metric info that stays
                //  the same for the whole session.
                //
                __UpdateSessionInfo(hwndThis);
            }

            catch(const tCIDLib::Tch* const pszError)
            {
                __CloseSession(hwndThis);
                __ShowMessage(hwndThis, pszError);
                return 1;
            }

            // Indicate that we have an open session
            __bSessionOpen = kCIDLib::True;

            // And enable the close session menu item and refresh buttons
            EnableMenuItem(GetMenu(hwndThis), MID_SESSION_CLOSE, MF_ENABLED);

            EnableWindow(GetDlgItem(hwndThis, DID_MAIN_REFVALUE), TRUE);
            EnableWindow(GetDlgItem(hwndThis, DID_MAIN_REFINFO), TRUE);
        }
         else if ((LOWORD(wParam) == MID_EXIT_EXIT)
              ||  (LOWORD(wParam) == IDCANCEL))
        {
            if (MessageBox
            (
                hwndThis
                , L"Exit the MiniMetrics Viewer?"
                , L"CIDLib Metrics Viewer"
                , MB_YESNO | MB_ICONQUESTION | MB_APPLMODAL) == IDYES)
            {
                PostQuitMessage(0);
            }
        }
         else if (((LOWORD(wParam) == DID_MAIN_MTRLIST)
              &&  (HIWORD(wParam) == LBN_SELCHANGE))
              || (LOWORD(wParam) == DID_MAIN_REFVALUE))
        {
            int iSelected = SendDlgItemMessage
            (
                hwndThis
                , DID_MAIN_MTRLIST
                , LB_GETCURSEL
                , 0
                , 0
            );

            if (iSelected == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected metric"
                );
                return 1;
            }

            // Update the metrics info
            try
            {
                __UpdateMetricInfo(hwndThis, iSelected);
            }

            catch(const tCIDLib::Tch* pszError)
            {
                __ShowMessage(hwndThis, pszError);
                return 1;
            }
        }
         else if (LOWORD(wParam) == DID_MAIN_REFINFO)
        {
            // The user wants to refresh the process info
            __UpdateSessionInfo(hwndThis);
        }
        return 1;
    }
     else if (uMsg == WM_DESTROY)
    {
        PostQuitMessage(0);
    }
     else if (uMsg == WM_INITDIALOG)
    {
        // Do any needed initialization of controls
        __InitControls(hwndThis);
        return 1;
    }
    return 0;
}


//
// FUNCTION/METHOD NAME: __bNewSessionProc
//
// DESCRIPTION:
//
//  This is the window proc for the process/group selection dialog. This
//  dialog allows the user to select a process and to select a metrics
//  group within that process.
// ---------------------------------------
//   INPUT: hwndThis is the dialog window handle.
//          uMsg is the message being recieved.
//          wParam, lParam are the window parameters.
//
//  OUTPUT: None
//
//  RETURN: 1 if we handle it, else 0.
//
static BOOL CALLBACK __bNewSessionProc( HWND        hwndThis
                                        , UINT      uMsg
                                        , WPARAM    wParam
                                        , LPARAM    lParam)
{
    if (uMsg == WM_COMMAND)
    {
        if (LOWORD(wParam) == DID_NEWSESS_ACCEPT)
        {
            //
            //  Set up the global data so that we are monitoring the
            //  selected group. So lets get the selected group and its
            //  item value, which is the group index we want.
            //
            int iSelected = SendDlgItemMessage
            (
                hwndThis
                , DID_NEWSESS_GROUPLIST
                , LB_GETCURSEL
                , 0
                , 0
            );

            if (iSelected == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected group"
                );
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }

            ULONG c4GroupIndex = SendDlgItemMessage
            (
                hwndThis
                , DID_NEWSESS_GROUPLIST
                , LB_GETITEMDATA
                , (WPARAM)iSelected
                , 0
            );

            if (c4GroupIndex == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected group's item data"
                );
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }

            iSelected = SendDlgItemMessage
            (
                hwndThis
                , DID_NEWSESS_PROGLIST
                , LB_GETCURSEL
                , 0
                , 0
            );

            if (iSelected == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected process"
                );
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }

            ULONG ulProcessIndex = SendDlgItemMessage
            (
                hwndThis
                , DID_NEWSESS_PROGLIST
                , LB_GETITEMDATA
                , (WPARAM)iSelected
                , 0
            );

            if (c4GroupIndex == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected process' item data"
                );
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }

            //
            //  Ok, this is the process and group that we are going to
            //  use. So get the use provided structure out and put these
            //  values into them.
            //
            TMetricSel* pmselInfo
                            = (TMetricSel*)GetWindowLong(hwndThis, DWL_USER);
            pmselInfo->c4ProcInd = ulProcessIndex;
            pmselInfo->c4GroupInd = c4GroupIndex;

            // Dismiss the dialog
            EndDialog(hwndThis, LOWORD(wParam));
            return 1;
        }
         else if ((LOWORD(wParam) == DID_NEWSESS_CANCEL)
              ||  (LOWORD(wParam) == IDCANCEL))
        {
            // Dismiss the dialog
            EndDialog(hwndThis, LOWORD(DID_NEWSESS_CANCEL));
            return 1;
        }
         else if ((LOWORD(wParam) == DID_NEWSESS_PROGLIST)
              &&  (HIWORD(wParam) == LBN_SELCHANGE))
        {
            //
            //  A new process was selected in the program list box. We
            //  need to load up the available metrics for this new
            //  process. Start by getting the process registry index
            //  from the list box item selected.
            //
            int iSelected = SendMessage((HWND)lParam, LB_GETCURSEL, 0, 0);
            if (iSelected == LB_ERR)
            {
                __ShowMessage
                (
                    hwndThis
                    , L"An error occured while getting the selected process"
                );
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }

            ULONG ulIndex = SendMessage
            (
                (HWND)lParam
                , LB_GETITEMDATA
                , (WPARAM)iSelected
                , 0
            );

            //
            //  Load up the groups available. If it returns FALSE, then
            //  we need to dismiss. Assume the function told the user
            //  why.
            //
            HWND hwndGroupList = GetDlgItem(hwndThis, DID_NEWSESS_GROUPLIST);
            try
            {
                __LoadGroups(hwndThis, hwndGroupList, ulIndex);
            }

            catch(const tCIDLib::Tch* pszError)
            {
                __ShowMessage(hwndThis, pszError);
                EndDialog(hwndThis, DID_NEWSESS_CANCEL);
                return 1;
            }
        }
        return 0;
    }
     else if (uMsg == WM_INITDIALOG)
    {
        // Get out the passed parameter and store it for later
        TMetricSel* pmselInfo = (TMetricSel*)lParam;
        SetWindowLong(hwndThis, DWL_USER, lParam);

        //
        //  Load up the available processes. Of course they might be
        //  gone by the time they are actually referenced, but we have
        //  to deal with that when it happens.
        //
        HWND hwndProcList = GetDlgItem(hwndThis, DID_NEWSESS_PROGLIST);

        try
        {
            __LoadProcesses(hwndThis, hwndProcList);
        }

        catch(const tCIDLib::Tch* pszError)
        {
            __ShowMessage(hwndThis, pszError);
            EndDialog(hwndThis, DID_NEWSESS_CANCEL);
            return 1;
        }

        return 1;
    }
    return 0;
}


//
// FUNCTION/METHOD NAME: __CloseSession
//
// DESCRIPTION:
//
//  This method is called to close down the current session, if any. It
//  will disable the refresh buttons, and clear out the controls, and
//  delete any global data that needs it.
// ---------------------------------------
//   INPUT: hwndDlg is the handle to the dialog box
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __CloseSession(HWND hwndDlg)
{
    // Clean up the current metric directory if any
    if (__pkmtrCurrent)
    {
        delete __pkmtrCurrent;
        __pkmtrCurrent = 0;
        __bSessionOpen = kCIDLib::False;
    }

    if (__hprocCurrent == kCIDLib::hprocInvalid)
    {
        TKrnlProcRegistry::CloseHandle(__hprocCurrent);
        __hprocCurrent = kCIDLib::hprocInvalid;
    }

    // Clean out all of the entry field controls
    SetDlgItemText(hwndDlg, DID_MAIN_MTRNAME, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_MTRHELP, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_MTRTYPE, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_MTRVALUE, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_PROCNAME, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_INFOSTRING, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_INFOCARD, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_GROUPDESC, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_PROCSTATE, kCIDLib::pszEmptyZStr);
    SetDlgItemText(hwndDlg, DID_MAIN_TERMCODE, kCIDLib::pszEmptyZStr);

    // Empty the list box of metrics
    SendDlgItemMessage(hwndDlg, DID_MAIN_MTRLIST, LB_RESETCONTENT, 0, 0);

    // Disable the refresh buttons
    EnableWindow(GetDlgItem(hwndDlg, DID_MAIN_REFVALUE), FALSE);
    EnableWindow(GetDlgItem(hwndDlg, DID_MAIN_REFINFO), FALSE);

    // Disable the session close menu item
    EnableMenuItem(GetMenu(hwndDlg), MID_SESSION_CLOSE, MF_GRAYED);
}


//
// FUNCTION/METHOD NAME: __FormatUnsignedValue
//
// DESCRIPTION:
//
//  Most unsigned values are displayed as decimal and hex format. So this
//  function does that grunt work.
// ---------------------------------------
//   INPUT: c4Value is the value to format
//          c4MaxChars are the maximum chars that the string can hold, not
//              counting the null.
//
//  OUTPUT: pszToFill is the string to format into.
//
//  RETURN: None
//
static tCIDLib::TVoid
__FormatUnsignedValue(  const   tCIDLib::TCard4     c4Value
                        , const tCIDLib::TCard4     c4MaxChars
                        ,       tCIDLib::Tch* const pszToFill)
{
    tCIDLib::TZStr128   szTmp1;
    tCIDLib::TZStr128   szTmp2;

    // Format the value as decimal and hex into temps
    TRawStr::FormatVal(c4Value, szTmp1, c4MaxBufChars(szTmp1), tCIDLib::ERadix_Dec);
    TRawStr::FormatVal(c4Value, szTmp2, c4MaxBufChars(szTmp2), tCIDLib::ERadix_Hex);

    // And copy to the target
    TRawStr::CopyCatStr
    (
        pszToFill
        , c4MaxChars
        , szTmp1
        , L"/0x"
        , szTmp2
    );
}


//
// FUNCTION/METHOD NAME: __InitControls
//
// DESCRIPTION:
//
//  This method is called once to initialize the controls in the main
//  window.
// ---------------------------------------
//   INPUT: hwndDlg is the handle to the dialog box
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __InitControls(HWND hwndDlg)
{
}


//
// FUNCTION/METHOD NAME: __LoadGroups
//
// DESCRIPTION:
//
//  This function will load the available metrics groups from the indicated
//  process into the passed list box.
// ---------------------------------------
//   INPUT: hwndDlg is the handle to the dialog box
//          hwndListBox is the handle to the list box to load
//          c4ProcInd is the index (into the process registry) of the process
//              to load the groups for.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __LoadGroups(   const   HWND    hwndDlg
                            , const HWND    hwndListBox
                            , const ULONG   c4ProcInd)
{
    // Clean any entries out of the current group list
    SendMessage(hwndListBox, LB_RESETCONTENT, 0, 0);

    // Lock the process registry
    TKrnlMutexLocker   kmtxlReg(__pkmtxProcReg);

    // Get a reference to the process entry
    const tCIDLib::TRawProcessEntry&
                        preCur = TKrnlProcRegistry::preEntryAt(c4ProcInd);

    //
    //  Create a metrics system object for this process. We don't use the
    //  global one here because this is called to load up the dialog
    //  from which the metrics are selected.
    //
    TKrnlMetricDirectory kmtrsTmp(preCur.pidThis);

    //
    //  Get a copy of the metrics directory of this process. We will use
    //  this copy to load the list box.
    //
    tCIDLib::TRawMetricDirectory mtrCopy;
    try
    {
        kmtrsTmp.CopyDirectory(mtrCopy);
    }

    catch(...)
    {
        throw L"An error occured while copying the directory contents";
    }

    for (ULONG ulIndex = 0; ulIndex < mtrCopy.c4GroupCount; ulIndex++)
    {
        int iIndex = SendMessage
        (
            hwndListBox
            , LB_ADDSTRING
            , 0
            , LPARAM(mtrCopy.amtrgList[ulIndex].szGroupName)
        );

        if ((iIndex == LB_ERR) || (iIndex == LB_ERRSPACE))
            throw L"An error occured inserting group name";

        //
        //  Store the group's actual index in the directory along with
        //  the list box element. This makes it easy later to get back to
        //  it without a name search.
        //
        SendMessage
        (
            hwndListBox
            , LB_SETITEMDATA
            , (WPARAM)iIndex
            , (LPARAM)ulIndex
        );
    }

    //
    //  Select the 0th element from the list if there were any. If there
    //  are not, then disable the Accept button.
    //
    if (mtrCopy.c4GroupCount)
        SendMessage(hwndListBox, LB_SETCURSEL, (WPARAM)0, 0);

    EnableWindow(GetDlgItem(hwndDlg, DID_NEWSESS_ACCEPT), mtrCopy.c4GroupCount);
}


//
// FUNCTION/METHOD NAME: __LoadMetrics
//
// DESCRIPTION:
//
//  This function will load up the names of all of the available metrics in
//  the group we are monitoring.
// ---------------------------------------
//   INPUT: hwndDlg is the handle of the owner dialog
//          hwndListBox is the handle of the list box to fill in.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __LoadMetrics(const HWND hwndDlg, const HWND hwndListBox)
{
    // Clean any entries out of the current group list
    SendMessage(hwndListBox, LB_RESETCONTENT, 0, 0);

    // Get a copy of the group we are monitoring
    tCIDLib::TRawMetricGroup mtrgTmp;
    try
    {
        __pkmtrCurrent->CopyGroup(mtrgTmp, __c4GroupIndex);
    }

    catch(...)
    {
        throw L"An error occured while copying the group contents";
    }

    for (ULONG ulIndex = 0; ulIndex < mtrgTmp.c4MetricCount; ulIndex++)
    {
        int iIndex = SendMessage
        (
            hwndListBox
            , LB_ADDSTRING
            , 0
            , LPARAM(mtrgTmp.amtrMetrics[ulIndex].szMetricName)
        );

        if ((iIndex == LB_ERR) || (iIndex == LB_ERRSPACE))
            throw L"An error occured inserting group name";
    }

    // Select the 0th element
    SendMessage(hwndListBox, LB_SETCURSEL, (WPARAM)0, 0);

    //
    //  Do an initial manual update of the controls that reflect the
    //  selected metric. The above selection setting does not cause the
    //  notification event that updates them when the user selects one.
    //
    //  Note that this one is not sorted, so we can just pass 0 as the
    //  metrics index.
    //
    __UpdateMetricInfo(hwndDlg, 0);
}


//
// FUNCTION/METHOD NAME: __LoadProcesses
//
// DESCRIPTION:
//
//  This function will load up the names of all of the available processes.
//  It looks at the process registry memory to find the process names.
// ---------------------------------------
//   INPUT: hwndDlg is the handle of the owner dialog
//          hwndListBox is the handle of the list box to fill in.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __LoadProcesses(HWND hwndDlg, HWND hwndListBox)
{
    //
    //  Get the list of processes loaded up to the list box. We just
    //  lock the registry and load up the list box.
    //
    {
        TKrnlMutexLocker   kmtxlReg(__pkmtxProcReg);

        for (ULONG ulIndex = 0; ulIndex < kCIDLib::c4MaxRegProc; ulIndex++)
        {
            const tCIDLib::TRawProcessEntry&
                            preCur = TKrnlProcRegistry::preEntryAt(ulIndex);

            if (preCur.bUsed)
            {
                int iIndex = SendMessage
                (
                    hwndListBox
                    , LB_ADDSTRING
                    , 0
                    , LPARAM(preCur.szProcName)
                );

                if ((iIndex == LB_ERR) || (iIndex == LB_ERRSPACE))
                    throw L"An error occured inserting process name";

                //
                //  Store the process' actual index in the registry
                //  with the list box element. This makes it easy later
                //  to get back to it without a name search.
                //
                SendMessage
                (
                    hwndListBox
                    , LB_SETITEMDATA
                    , (WPARAM)iIndex
                    , (LPARAM)ulIndex
                );
            }
        }
    }

    //
    //  Select the 0th element. This does not cause a list box selection
    //  change notification so we have to manually load the groups for
    //  this initial selection.
    //
    SendMessage(hwndListBox, LB_SETCURSEL, 0, 0);

    //
    //  Now we need to get the handle of the newly selected item. We
    //  need to do this because the list is sorted. So we don't know which
    //  process actually went to the 0th element.
    //
    int iSelected = SendMessage(hwndListBox, LB_GETCURSEL, 0, 0);
    if (iSelected == LB_ERR)
    {
        EnableWindow(GetDlgItem(hwndDlg, DID_NEWSESS_ACCEPT), FALSE);
        return;
    }

    ULONG ulIndex = SendMessage
    (
        hwndListBox
        , LB_GETITEMDATA
        , (WPARAM)iSelected
        , 0
    );

    HWND hwndGroupList = GetDlgItem(hwndDlg, DID_NEWSESS_GROUPLIST);
    __LoadGroups(hwndDlg, hwndGroupList, ulIndex);
}


//
// FUNCTION/METHOD NAME: __ShowMessage
//
// DESCRIPTION:
//
//  A quicky function to show a message in a popup message box. Most of
//  it is always the same, so this saves some effort.
// ---------------------------------------
//   INPUT: hwndOwner is the window to make the owner of the message box.
//          pszText is the message text to display.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __ShowMessage(HWND hwndOwner, const tCIDLib::Tch* const pszText)
{
    MessageBox
    (
        hwndOwner
        , pszText
        , L"CIDLib Metrics Viewer"
        , MB_OK | MB_ICONASTERISK | MB_APPLMODAL
    );
}


//
// FUNCTION/METHOD NAME: __UpdateMetricInfo
//
// DESCRIPTION:
//
//  This function will update the metric info fields that are driven by
//  the metric selected in the list box.
// ---------------------------------------
//   INPUT: hwndDlg is the handle of the main dialog
//          c4MetricIndex is the index of the metric within the current
//              group that we are update for.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID
__UpdateMetricInfo(HWND hwndDlg, const tCIDLib::TCard4 c4MetricIndex)
{
    // Get a copy of the metric
    tCIDLib::TRawMetric mtrTmp;

    try
    {
        __pkmtrCurrent->CopyMetric(mtrTmp, __c4GroupIndex, c4MetricIndex);
    }

    catch(...)
    {
        throw L"Could not get a copy of the metric entry";
    }

    // Update the metric name and description
    SetDlgItemText(hwndDlg, DID_MAIN_MTRNAME, mtrTmp.szMetricName);
    SetDlgItemText(hwndDlg, DID_MAIN_MTRHELP, mtrTmp.szMetricHelp);

    //
    //  Update the metric type and value. This is driven by the metric
    //  type, so we do them both in a switch statement.
    //
    tCIDLib::TCard4     c4Ind;
    tCIDLib::TCard4     c4Mask;
    tCIDLib::TCard4     c4Tmp;
    tCIDLib::TZStr64    szType;
    tCIDLib::TZStr128   szValue;
    switch(mtrTmp.eType)
    {
        case tCIDLib::EMetric_Accum :
            TRawStr::CopyStr(szType, L"Accumulator", c4MaxBufChars(szType));
            TRawStr::FormatVal(mtrTmp.f4FValue, szValue, 0, 128);
            break;

        case tCIDLib::EMetric_Cardinal :
            TRawStr::CopyStr(szType, L"Cardinal", c4MaxBufChars(szType));
            __FormatUnsignedValue(mtrTmp.c4CValue, c4MaxBufChars(szValue), szValue);
            break;

        case tCIDLib::EMetric_FlagSet :
            TRawStr::CopyStr(szType, L"Flags", c4MaxBufChars(szType));
            c4Mask  = 0x00000001;
            c4Tmp   = mtrTmp.c4CValue;
            for (c4Ind = 0; c4Ind < 32; c4Ind++)
            {
                if (c4Tmp & c4Mask)
                {
                    tCIDLib::TCard4 c4Mod = c4Ind % 16;
                    if (c4Mod < 10)
                        szValue[c4Ind] = L'0' + tCIDLib::Tch(c4Mod);
                    else
                        szValue[c4Ind] = L'a' + tCIDLib::Tch(c4Mod-10);
                }
                 else
                {
                    szValue[c4Ind] = L'-';
                }
                c4Mask <<= 1;
            }
            szValue[32] = 0;
            break;

        case tCIDLib::EMetric_Integer :
            TRawStr::CopyStr(szType, L"Integral", c4MaxBufChars(szType));
            TRawStr::FormatVal(mtrTmp.i4IValue, szValue, 128);
            break;

        case tCIDLib::EMetric_PeakMeter :
            TRawStr::CopyStr(szType, L"Peak Meter", c4MaxBufChars(szType));
            TRawStr::FormatVal(mtrTmp.c4CValue, szValue, 128);
            break;

        case tCIDLib::EMetric_Toggle :
            TRawStr::CopyStr(szType, L"Boolean", c4MaxBufChars(szType));
            if (mtrTmp.bToggle)
                TRawStr::CopyStr(szValue, L"True", c4MaxBufChars(szValue));
            else
                TRawStr::CopyStr(szValue, L"False", c4MaxBufChars(szValue));
            break;

        case tCIDLib::EMetric_TickCounter :
            TRawStr::CopyStr(szType, L"Tick Counter", c4MaxBufChars(szType));
            TRawStr::FormatVal(mtrTmp.f4FValue, szValue, 0, 128);
            break;

        default :
            throw L"Invalid metric type";
            break;
    }

    // Set the copied strings into their controls
    SetDlgItemText(hwndDlg, DID_MAIN_MTRTYPE, szType);
    SetDlgItemText(hwndDlg, DID_MAIN_MTRVALUE, szValue);
}


//
// FUNCTION/METHOD NAME: __UpdateSessionInfo
//
// DESCRIPTION:
//
//  This function will update all of the main window controls that are
//  related to the session, as apposed to the metrics of the selected group.
// ---------------------------------------
//   INPUT: hwndMain is the main window handle, i.e. the parent of all of
//              controls we will fill in.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __UpdateSessionInfo(HWND hwndThis)
{
    tCIDLib::TZStr128   szTmp;

    // Set the process name
    SetDlgItemText(hwndThis, DID_MAIN_PROCNAME, __preCurrent.szProcName);

    // Set the info string and card
    SetDlgItemText(hwndThis, DID_MAIN_INFOSTRING, __preCurrent.szInfoLine);

    __FormatUnsignedValue(__preCurrent.c4InfoCard, c4MaxBufChars(szTmp), szTmp);
    SetDlgItemText(hwndThis, DID_MAIN_INFOCARD, szTmp);

    // Get a copy of the group we are monitoring
    tCIDLib::TRawMetricGroup mtrgTmp;

    try
    {
        __pkmtrCurrent->CopyGroup(mtrgTmp, __c4GroupIndex);
    }

    catch(...)
    {
        throw L"Could nto get a copy of the metrics group info";
    }

    // Set the group description
    SetDlgItemText(hwndThis, DID_MAIN_GROUPDESC, mtrgTmp.szGroupHelp);

    //
    //  Set the process state. At this level we don't have access to the
    //  standard enum formatting stuff, so we have to do it manually.
    //
    tCIDLib::Tch szState[32];
    switch (__pkmtrCurrent->eState())
    {
        case tCIDLib::EProcState_Initializing   :
            TRawStr::CopyStr(szState, L"Initializing", c4MaxBufChars(szState));
            break;

        case tCIDLib::EProcState_Ready  :
            TRawStr::CopyStr(szState, L"Ready", c4MaxBufChars(szState));
            break;

        case tCIDLib::EProcState_Terminating    :
            TRawStr::CopyStr(szState, L"Terminating", c4MaxBufChars(szState));
            break;

        case tCIDLib::EProcState_Dead   :
            TRawStr::CopyStr(szState, L"Dead", c4MaxBufChars(szState));
            break;

        default                             :
            TRawStr::CopyStr(szState, L"????", c4MaxBufChars(szState));
            break;
    }
    SetDlgItemText(hwndThis, DID_MAIN_PROCSTATE, szState);

    //
    //  Get the exit code of the process. If we get an exception, we just
    //  assume it's not dead.
    //
    tCIDLib::EExitCodes eCode = tCIDLib::EExit_Normal;
    try
    {
        eCode = TKrnlProcRegistry::eWaitForDeath(__hprocCurrent, 0);
    }

    catch(...)
    {
        // Don't do anything. It just means its not dead yet
    }

    __FormatUnsignedValue(eCode, c4MaxBufChars(szTmp), szTmp);
    SetDlgItemText(hwndThis, DID_MAIN_TERMCODE, szTmp);
}
 

// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: WinMain
//
// DESCRIPTION:
//
//  This guy just does the program initialization, setting up the access
//  to the CIDLib process registry and metrics system and then getting
//  the main window prepared.
// ---------------------------------------
//   INPUT: hInstance is the process instance of this program
//          lpCmdLine is the command line string
//          nShowCmd indicates whether the window is to be initially
//              displayed.
//
//  OUTPUT: None
//
//  RETURN: 0 if successful, 1 if a failure
//
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE, LPSTR, int nShowCmd)
{
    // Init the app
    if (!__bInitApplication(hInstance))
        return FALSE;

	ShowWindow(__hwndMain, nShowCmd);
	UpdateWindow(__hwndMain);

    MSG qmsgGet;
	while (GetMessage(&qmsgGet, NULL, 0, 0) == TRUE)
        DispatchMessage(&qmsgGet);

    // Destroy the main winodw
    DestroyWindow(__hwndMain);

	return (qmsgGet.wParam);
}
