//
// NAME: MetricsDemo.Cpp
//
// DESCRIPTION:
//
//  This is a small demo program which demonstrates the metrics gathering
//  capabilities of the CIDLib system. It sets up a metrics group and then
//  just loops, manipulating the metrics, until the user terminates the
//  program by pressing enter.
//
//  Once a metrics group has been created and registered it cannot be
//  changed or deregistered because other apps may be referencing it. Other
//  (metrics monitor) apps that want to look at it do not need to know about
//  the app or its metrics ahead of time. They use the process registry
//  system to find the app and the metrics data is self describing.
//
//  This program also creates a process registry object and sets the cardinal
//  and string registry values that are available to each process. The metrics
//  program will show these values. They can be used by each process to set
//  some info for the metrics program to see, such as a shutdown code or
//  something of that nature.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/95
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
//
// MODIFICATION LOG:
//

// -----------------------------------------------------------------------------
//  CIDLib includes.
// -----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// -----------------------------------------------------------------------------
//  Local types
//
//  EDemoMetrics
//      This enumerates the metrics that we add to our metrics group. We just
//      set up one of each.
// -----------------------------------------------------------------------------
enum EDemoMetrics
{
    EDemoMetric_Accum
    , EDemoMetric_CardinalValue
    , EDemoMetric_FlagSet
    , EDemoMetric_IntegerValue
    , EDemoMetric_PeakMeter
    , EDemoMetric_TickCounter
    , EDemoMetric_Toggle

    // Must be the last entry
    , EDemoMetric_Count
};


// -----------------------------------------------------------------------------
//  Local function prototypes
//
//  __eMainThead
//      This is the thread function for the main thread, thrMain, that provides
//      the execution for this program.
//
//  __eMetricsThead
//      This is the thread function for the metrics thread, which just wakes
//      up once in a while and manipulates the test metrics group.
// -----------------------------------------------------------------------------
static tCIDLib::EExitCodes  __eMainThread(TThread&, tCIDLib::TVoid*);
static tCIDLib::EExitCodes  __eMetricsThread(TThread&, tCIDLib::TVoid*);


// -----------------------------------------------------------------------------
//  Local data declarations
//
//  __conOut
//      We create a standard console object for our input and output. The
//      TConsole class is a specialized text stream, which supports fancy
//      user input, with command line editing and recall.
//
//  __facMetricsDemo
//      This is our facility object. Since this program is very simple, we
//      just declare an instance of the basic TFacility class (instead of making
//      our own derivative.) There is no reason to derive because we add no
//      new functionality.
//
//  __pmtrgDemo
//      A pointer to our metrics group. It is created below. Metrics groups
//      cannot be created directly so we just have a pointer to one which
//      will be allocated later.
//
//  __thrMetrics
//      This is the thread object that manipulates the metrics information.
// -----------------------------------------------------------------------------
static TConsole         __conOut;
static TFacility        __facMetricsDemo
                        (
                            L"MetricsDemo"
                            , tCIDLib::EModType_Exe
                            , kCIDLib::c4MajVersion
                            , kCIDLib::c4MinVersion
                        );
static TMetricGroup*    __pmtrgDemo;
static TThread          __thrMetrics(L"MetricsThread", __eMetricsThread);


// -----------------------------------------------------------------------------
//  Include magic main module code
// -----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"MainThread", __eMainThread))


//
// FUNCTION/METHOD NAME: __eMainThread
//
// DESCRIPTION:
//
//  This is the main program thread. It sets up the metrics group and
//  registers it, then kicks off the metrics thread, and just waits for
//  the user to press Enter.
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThread(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    // Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    // Announce ourselves
    __conOut    << L"\r\nMetricsDemo.Exe\r\nCompiled: "
                << TString(__DATE__) << L"\r\n\n";

    // Log our startup
    __facMetricsDemo.LogMsg
    (
        __FILE__
        , __LINE__
        , L"Metrics Demo Program loading"
        , tCIDLib::ESev_Status
    );

    //
    //  Set up the metrics group and register it. We use a pointer for each
    //  one in order to keep from typing so much.
    //
    TMetricSystem::TMetricInfo  amtriList[EDemoMetric_Count];
    TMetricSystem::TMetricInfo* pmtriTmp;

    pmtriTmp = &amtriList[EDemoMetric_Accum];
    pmtriTmp->eType          = tCIDLib::EMetric_Accum;
    pmtriTmp->strMetricName  = L"An accumulator metric";
    pmtriTmp->strMetricHelp  = L"A 64 bit accumulator value";

    pmtriTmp = &amtriList[EDemoMetric_CardinalValue];
    pmtriTmp->eType          = tCIDLib::EMetric_Cardinal;
    pmtriTmp->strMetricName  = L"A cardinal value metric";
    pmtriTmp->strMetricHelp  = L"A 32 bit unsigned value";

    pmtriTmp = &amtriList[EDemoMetric_FlagSet];
    pmtriTmp->eType          = tCIDLib::EMetric_FlagSet;
    pmtriTmp->strMetricName  = L"A flag set metric";
    pmtriTmp->strMetricHelp  = L"A 32 bit flag set";

    pmtriTmp = &amtriList[EDemoMetric_IntegerValue];
    pmtriTmp->eType          = tCIDLib::EMetric_Integer;
    pmtriTmp->strMetricName  = L"An integer value metric";
    pmtriTmp->strMetricHelp  = L"A 32 bit signed value";

    pmtriTmp = &amtriList[EDemoMetric_PeakMeter];
    pmtriTmp->eType          = tCIDLib::EMetric_PeakMeter;
    pmtriTmp->strMetricName  = L"A peak meter metric";
    pmtriTmp->strMetricHelp  = L"Saves the highest 32 bit value set to it";

    pmtriTmp = &amtriList[EDemoMetric_TickCounter];
    pmtriTmp->eType          = tCIDLib::EMetric_TickCounter;
    pmtriTmp->strMetricName  = L"A tick counter metric";
    pmtriTmp->strMetricHelp  = L"This guy is a 64 bit tick counter";

    pmtriTmp = &amtriList[EDemoMetric_Toggle];
    pmtriTmp->eType          = tCIDLib::EMetric_Toggle;
    pmtriTmp->strMetricName  = L"A toggle metric";
    pmtriTmp->strMetricHelp  = L"This guy is a boolean toggle value";

    //
    //  Create a metric system object, which we need to registry a
    //  metrics group. Then register our group.
    //
    __pmtrgDemo = TMetricSystem::pmtrgRegisterGroup
    (
        L"DemoMetrics Metrics"
        , L"Test Metrics Group For Demo"
        , amtriList
        , EDemoMetric_Count
    );

    //
    //  Set our two process values in the process registry. These should
    //  show up in the metrics viewer.
    //
    TProcessRegistry::c4InfoCard(0xFACEECAF);
    TProcessRegistry::strInfoLine("This is a test");

    //
    //  Ok, we can now kick off the metric thread, which will just update
    //  the metrics periodically.
    //
    __thrMetrics.Start();

    // Wait for an Enter from the user
    __conOut << L"Press Enter To End the Program...";

    //
    //  Create an input string with a max size of 256 chars and fully
    //  committed up to the whole 256 chars. We could let it fault in
    //  the extra storage, but this is efficient and we can spare the
    //  bytes easily for this simple program.
    //
    TString strTmp(kCIDLib::pszEmptyZStr, 256, 256);
    __conOut.c4GetLine(strTmp);

    //
    //  Ask the metrics program to shutdown. He wakes up every 2 seconds
    //  so 10 seconds is way more than long enough to wait. The shutdown
    //  request method actually uses some underlying process sync
    //  functionality in order to do the shutdown request for us.
    //
    __conOut << L"Shutting down metrics thread..." << NewLn;
    try
    {
        __thrMetrics.RequestShutdown(10 Seconds);
    }

    catch(const TError& errToCatch)
    {
        __conOut << L"Error occured during metrics thread shutdown..."
                 << DNewLn << L"  Error: " << errToCatch << DNewLn;
    }
    __conOut << L"Shutdown complete" << NewLn;

    // Log our shutdown
    __facMetricsDemo.LogMsg
    (
        __FILE__
        , __LINE__
        , L"Metrics Demo Program shutdown"
        , tCIDLib::ESev_Status
    );

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    // Return a very obvious test code that should show up in the viewer
    return tCIDLib::EExitCodes(0xFACE);
}


//
// FUNCTION/METHOD NAME: __eMetricsThread
//
// DESCRIPTION:
//
//  This is a secondary thread that just periodically wakes up and
//  manipulates the metrics group, __pmtrgDemo;
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator.
//
//  OUTPUT: None
//
//  RETURN: Just returns tCIDLib::EExit_Normal.
//
tCIDLib::EExitCodes __eMetricsThread(TThread& thrThis, tCIDLib::TVoid*)
{
    // Do an initial sync so the caller can be let go
    thrThis.Sync();

    //
    //  These are some state variables that let use meaningfully change
    //  the metrics on each loop.
    //
    tCIDLib::TCard4  c4BitMask   = 0x01001001;

    // Seed the random number generator with some value
    TMathLib::SeedRandom(1009);

    // Loop until we are asked to shut down
    while (1)
    {
        // Sync up just in case the main thread is waiting for us
        thrThis.Sync();

        // Check for a shutdown request. If so, then fall out
        if (thrThis.bCheckShutdownRequest())
            break;

        //
        //  Adjust each of the metrics in some way.
        //
        __pmtrgDemo->AddToAccum(EDemoMetric_Accum, 10);

        //
        //  SetCard() will set it explitly, but we just want to keep
        //  bumping it up.
        //
        __pmtrgDemo->OffsetCard(EDemoMetric_CardinalValue, 1);

        //
        //  SetFlag will set/clear a single flag, but we just keep rotating
        //  a mask and setting the whole thing explitly. The target mask is
        //  set to 0xFFFFFFFF to tell him to take all of the bits in our
        //  mask as significant.
        //
        __pmtrgDemo->SetFlags(EDemoMetric_FlagSet, 0xFFFFFFFF, c4BitMask);
        c4BitMask = TRawBits::c4RotateLeft(c4BitMask, 1);

        //
        //  SetInt() will set it explicitly but we just keep bumping it down.
        //
        __pmtrgDemo->OffsetInt(EDemoMetric_IntegerValue, -1);

        //
        //  Generate a random number and set it to the peakmeter. It should
        //  store the highest number that we set into it.
        //
        __pmtrgDemo->SetPeak(EDemoMetric_PeakMeter, TMathLib::c4RandomNum());

        //
        //  For a tick counter, you can just increment it. It bumps up by
        //  1 for each tick.
        //
        __pmtrgDemo->IncCounter(EDemoMetric_TickCounter);

        //
        //  Flip the toggle. You can also call SetToggle() to set it to an
        //  explicit eTrue or eFalse value.
        //
        __pmtrgDemo->FlipToggle(EDemoMetric_Toggle);

        // Sleep for 2 seconds
        thrThis.Sleep(2 Seconds);
    }
    return tCIDLib::EExit_Normal;
}
