//
//  FILE NAME: MakeMsgs.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 06/26/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the MakeMsgs.Exe program. This program is a
//  low level utility that is used by CIDLib to create message files for
//  errors that are thrown.
//
//  The usage is this:
//
//      MakeMsgs facname errns msgns outpath
//
//      facname = The project name of the facility being built
//      errns   = The namespace for the errors Hpp file
//      msgsns  = The namespace for the messages Hpp file
//      outpath = The output path for the target binary file
//
//  The files created are based upon the name of the facilty.
//
//  For this invocation: "MakeMsgs Foo kFooErrs kFooMsgs $(OutDir)"
//
//          Foo_StringIds.Hpp       (In the project directory)
//          Foo_ErrorIds.Hpp        (In the project directory)
//          $(OutDir)\Foo.CIDMsg    (In the given directory)
//
//  The messages in Foo_StringIds.Hpp will be in the kFooMsgs namespace. The
//  messages in Foo_ErrorIds.Hpp will be in the kFooErrs namespace.
//
//  The other usage form is used to attach the built binary file to the
//  DLL or Exe file. The usage is:
//
//      MakeMsgs /Attach exepath msgpath
//
//      /Attach = Indicates that this is an attach operation
//      exepath = The path to the DLL/Exe to attach to
//      msgpath = The path to the CIDMsg binary message file to attach
//
//
//  CAVEATS/GOTCHAS:
//
//  1)  The source text file can be in UNICode or ASCII format! But the
//      actual output binary file is always in UNICode format so that it
//      can do what it is designed for (i.e. hold messages in any language.)
//      The output Hpp file is of course in ASCII since the compiler cannot
//      understand UNICode.
//


// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDKernel.Hpp"
#include    "CIDKernel_MsgFileSupport.Hpp"


// ----------------------------------------------------------------------------
//  Local constants
//
//  __c4MaxParmLen
//      The maximum lengths the command line parm storage strings are.
//
//  __c4BufSize
//      The size of the spooling buffer.
// ----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4MaxParmLen = 1024;
static const tCIDLib::TCard4    __c4BufSize = 1024;


// ----------------------------------------------------------------------------
//  Local types
//
//  EReturnVals
//      These are the program return values. This allows a command file
//      executing the program understand why it exited.
//
//  EMsgTypes
//      This enum indicates the type of message. Error and message type
//      messages are supported.
//
//  TIdInfo
//      This is a small structure that we used to create a linked list of
//      id information nodes. In order to create the binary message file, we
//      must have all of the information before we start. So they are saved
//      up during parsing and then written out after the whole source file
//      has been digested.
// ----------------------------------------------------------------------------
enum EReturnVals
{
    EReturn_Successful
    , EReturn_CantCreateErrHpp
    , EReturn_CantCreateMsgHpp
    , EReturn_CantCreateMsg
    , EReturn_CantOpenSource
    , EReturn_CantOpenTarget
    , EReturn_CantReadSource
    , EReturn_InvalidFormat
    , EReturn_Parameters
    , EReturn_CantUpdateResource
};

enum EMsgTypes
{
    EMsgType_None
    , EMsgType_EOF
    , EMsgType_Error
    , EMsgType_Message
};

struct TIdInfo
{
    tCIDLib::TErrCode   midThis;
    tCIDLib::Tch*       pszIdName;
    tCIDLib::Tch*       pszText;
    TIdInfo*            pidiNext;
};


// ----------------------------------------------------------------------------
//  Local data
//
//  __bUNICode
//      This is set if we find a UNICode marker in the source MsgText file,
//      indicating that the source file content is in UNICode. Else its
//      assumed to be ASCII and its converted.
//
//  __c4CurLine
//      This is maintained by the __chGetNext() function and represents the
//      current line within the source file.
//
//  __c4CurIndex
//  __c4CurSize
//      These are the current size and and index into the spooling buffer.
//
//  __c4MsgCount
//      This is the count of messages that was parsed out of the input
//      message text file. This means its the number of elements in the
//      __pidiHeader linked list.
//
//  __chUnget
//      A single character unget buffer.
//
//  __kflErrorHpp
//  __kflMessageHpp
//  __kflSource
//  __khflTargetMsg
//      These are the files handles for the files that we open. The source
//      is the only input file. The others are output files.
//
//  __kconOutput
//      This is a console object that we will use to output our user viewable
//      stuff. Since we don't use the default constructor we don't have to do
//      a separate Open() call.
//
//  __pidiHead
//      This is the head pointer to the linked list of id information
//      structure linked list.
//
//  __szBuffer
//      This is the spooling buffer, which is filled in blocks and then
//      spooled from by __chCharSpooler().
//
//  __szErrNameSpace
//  __szMsgNameSpace
//      These are filled with the namespaces for the two output Hpp files.
//      These come from the third and fourth parameters.
//
//  __szFacilityName
//      This is gotten from the second parameter and is used to build the
//      othe file names.
//
//  __szErrorHpp
//  __szMessageHpp
//  __szSourceFl
//  __szTargetFl
//      This is the path to the source file and the path to the output files.
// ----------------------------------------------------------------------------
static  tCIDLib::TBoolean   __bUNICode = kCIDLib::False;
static  tCIDLib::TCard4     __c4CurLine = 1;
static  tCIDLib::TCard4     __c4CurIndex;
static  tCIDLib::TCard4     __c4CurSize;
static  tCIDLib::TCard4     __c4MsgCount;
static  tCIDLib::Tch        __chUnget;
static  TKrnlFile           __kflErrorHpp;
static  TKrnlFile           __kflMessageHpp;
static  TKrnlConsole        __kconOutput
                            (
                                kCIDLib::False
                                , 0
                                , tCIDLib::ETextFmt_ASCII
                                , tCIDLib::ERedir_Allow
                            );
static  TKrnlFile           __kflSource;
static  TKrnlFile           __kflTargetMsg;
static  TIdInfo*            __pidiHead;
static  tCIDLib::Tch        __szFacilityName[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szBuffer[__c4BufSize];
static  tCIDLib::Tch        __szErrNameSpace[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szMsgNameSpace[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szErrorHpp[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szMessageHpp[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szSourceFl[__c4MaxParmLen+1];
static  tCIDLib::Tch        __szTargetFl[__c4MaxParmLen+1];


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::TVoid __ShowReturnVal
(
    const   EReturnVals             eRet
);

static tCIDLib::TVoid __WriteToHpp
(
            TKrnlFile&              kflTarget
    , const tCIDLib::Tch* const     pszToWrite
);

static tCIDLib::TVoid __WriteToHpp
(
            TKrnlFile&              kflTarget
    , const tCIDLib::Tsch* const    pszToWrite
);



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eAttach
//
// DESCRIPTION:
//
//  This is called when the /Attach option is seen. This is a separate mode
//  for the program that attaches a binary message file to a DLL/Exe file.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: One of the standard return values.
//
static EReturnVals __eAttach()
{
    //
    //  We need to open the source file and read it into memory. It has
    //  to be attached as a binary blob.
    //
    EReturnVals eRet = EReturn_Successful;
    try
    {
        eRet = EReturn_CantOpenSource;
        __kflSource.SetName(__szSourceFl);
        __kflSource.Open
        (
            tCIDLib::EAccess_Excl_Read
            , tCIDLib::ECreateAct_OpenIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        // Get the size of the file so we can allocate a buffer to read into
        tCIDLib::TCard4 c4Size = tCIDLib::TCard4(__kflSource.fposCurSize());

        // Allocate a buffer for it and put a janitor on it
        tCIDLib::TCard1* pc1Buf = (tCIDLib::TCard1*)TRawMem::pAllocSysMem
        (
            c4Size
            , tCIDLib::EMemAcc_ReadWrite
            , TRawMem::ECommit
        );

        try
        {
            // And read in the whole file
            eRet = EReturn_CantReadSource;
            __kflSource.ReadBuffer(pc1Buf, c4Size);

            // And try to update the message resources
            eRet = EReturn_CantUpdateResource;
            CIDMsgFile::UpdateMsgResource(__szTargetFl, pc1Buf, c4Size);
        }

        catch(...)
        {
            TRawMem::FreeSysMem(pc1Buf);
        }
    }

    catch(const TKrnlError& kerrToCatch)
    {
        __kconOutput.PutLine(L"Error occured while ");
        __ShowReturnVal(eRet);
        __kconOutput.PutLine(L"\r\n\n");

        __kconOutput.PutLine(L"   Error Code: ");
        tCIDLib::Tch szNum[16];
        TRawStr::FormatVal(kerrToCatch.errcId(), szNum, c4MaxBufChars(szNum));
        __kconOutput.PutLine(szNum);
        __kconOutput.PutLine(L"\r\n\n");
        return eRet;
    }

    catch(...)
    {
        __kconOutput.PutLine(L"System error occured while ");
        __ShowReturnVal(eRet);
        __kconOutput.PutLine(L"\r\n\n");
       return eRet;
    }

    return EReturn_Successful;
}


//
// FUNCTION/METHOD NAME: __chCharSpooler
//
// DESCRIPTION:
//
//  This function is a spooler for the input file. It reads in the input file
//  by chunks and spools out characters.
//
//  It is assumed that the message file is in UNICode format!
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The next character available, or kCIDLib::chNull if no more data.
//
static tCIDLib::Tch __chCharSpooler()
{
    //
    //  If we are out of data in the buffer, then read another blob. If
    //  we can't get any more data, return 0. Note that __c4CurSize is
    //  passed as the 'actual bytes read' parameter to be filled in.
    //
    if (__c4CurIndex == __c4CurSize)
    {
        if (__bUNICode)
        {
            __kflSource.ReadBuffer
            (
                __szBuffer
                , __c4BufSize * kCIDLib::c4CharBytes
                , __c4CurSize
            );

            // Turn the byte size into UNICode chararacter count
            __c4CurSize /= kCIDLib::c4CharBytes;
        }
         else
        {
            //
            //  Its not UNICode, so we need to read into a temp buffer
            //  and then convert it to UNICode into the spool buffer.
            //
            tCIDLib::Tsch szTmpBuf[__c4BufSize];
            __kflSource.ReadBuffer
            (
                szTmpBuf
                , __c4BufSize
                , __c4CurSize
            );

            //
            //  If we got any bytes, then we need to convert them. Otherwise
            //  we are done. Note that __c4CurSize in this scheme can be
            //  left as is because it will correctly represent the number of
            //  chars in this version of the read.
            //
            if (__c4CurSize)
            {
                for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4CurSize; c4Index++)
                    __szBuffer[c4Index] = TRawStr::chConvert(szTmpBuf[c4Index]);
            }
        }

        // No more data, return a nul char
        if (!__c4CurSize)
            return kCIDLib::chNull;

        // And reset the index into the buffer
        __c4CurIndex = 0;
    }
    return __szBuffer[__c4CurIndex++];
}


//
// FUNCTION/METHOD NAME: __chGetNext
//
// DESCRIPTION:
//
//  This function calls the spooler to spool out chars, and it handles
//  eating comment lines and carriage returns.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The next character available, or 0 if no more data.
//
static tCIDLib::Tch __chGetNext()
{
    tCIDLib::Tch chRet;

    // If there is an unget char, then return it first
    if (__chUnget)
    {
        chRet = __chUnget;
        __chUnget = 0;
        return chRet;
    }

    //
    //  No unget char, so we have to find a char to return. This loop handles
    //  scarfing out comment lines so that the main line logic does not
    //  have to deal with them.
    //
    while (1)
    {
        // Get the next char
        chRet = __chCharSpooler();

        //
        //  If this is a '/' character, then check the next character. If
        //  its also a '/', then this is a comment so we eat chars until the
        //  next new line. If its not, then unget it and return this char.
        //
        if (chRet == kCIDLib::chSlash)
        {
            tCIDLib::Tch chNext = __chCharSpooler();

            if (chNext == kCIDLib::chSlash)
            {
                while (1)
                {
                    chNext = __chCharSpooler();

                    if (!chNext)
                        return 0;

                    if (chNext == kCIDLib::chLF)
                        break;
                }
            }
             else
            {
                //
                //  The next char was not '/', so unget it and break out
                //  of inner loop.
                //
                __chUnget = chNext;
                break;
            }
        }
         else
        {
            break;
        }
    }

    return chRet;
}


//
// FUNCTION/METHOD NAME: __c4GetIdName
//
// DESCRIPTION:
//
//  This function will read in the next token from the file, which should
//  be the name of the id. It cannot have any whitespace, so we just read
//  until we hit a whitespace char.
// ---------------------------------------
//   INPUT: c4MaxChar is the maximum characters that the passed buffer
//              can hold.
//
//  OUTPUT: szBuf The buffer to put the rest of the line text into.
//
//  RETURN: The number of chars in the buffer, 0 if at the end of the file.
//
static tCIDLib::TCard4
__c4GetIdName(tCIDLib::Tch* szBuf, const tCIDLib::TCard4 c4MaxChars)
{
    // Init the buffer to empty
    szBuf[0] = kCIDLib::chNull;

    // Loop until we get a whitespace or end of data
    tCIDLib::TCard4 c4Index = 0;
    tCIDLib::Tch    chNext;
    while (1)
    {
        chNext = __chGetNext();

        // If its leading whitespace, then skip it
        if (!c4Index && TRawStr::bIsSpace(chNext))
            continue;

        if (TRawStr::bIsSpace(chNext) || (chNext == kCIDLib::chNull))
        {
            szBuf[c4Index] = 0;
            return c4Index;
        }

        if (c4Index+1 == c4MaxChars)
            throw L"Message id too long for provided buffer";

        szBuf[c4Index++] = chNext;
    }
}


//
// FUNCTION/METHOD NAME: __bGetIdValue
//
// DESCRIPTION:
//
//  This function will read in the next token from the file, which should
//  be the value of the id. It cannot have any whitespace, so we just read
//  until we hit a whitespace char.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: errcId is filled with the value read in.
//
//  RETURN: kCIDLib::True if successful, else kCIDLib::False.
//
static tCIDLib::TBoolean __bGetIdValue(tCIDLib::TErrCode& errcId)
{
    // Init a local buffer to empty
    const tCIDLib::TCard4 c4BufChars = 64;
    tCIDLib::Tch szBuf[c4BufChars+1];
    szBuf[0] = kCIDLib::chNull;

    // Loop until we get a whitespace or end of data
    tCIDLib::TCard4 c4Index = 0;
    tCIDLib::Tch    chNext;
    while (1)
    {
        chNext = __chGetNext();

        // If its leading whitespace, then skip it
        if (!c4Index && TRawStr::bIsSpace(chNext))
            continue;

        if (TRawStr::bIsSpace(chNext) || (chNext == kCIDLib::chNull))
        {
            // Cap it off and convert to a card4 value
            szBuf[c4Index] = 0;

            tCIDLib::TBoolean bValid;
            errcId = TRawStr::c4AsBinary
            (
                szBuf
                , tCIDLib::ERadix_Dec
                , bValid
            );
            return bValid;
        }

        if (c4Index+1 == c4BufChars)
            throw L"Message value too long for provided buffer";

        szBuf[c4Index++] = chNext;
    }
}


//
// FUNCTION/METHOD NAME: __c4GetRestOfLine
//
// DESCRIPTION:
//
//  This function will read in the rest of the current line from the
//  input file. If the line is quoted, then the quotes are tossed and
//  the leading and trailing spaces, if any, kept. If its not quoted,
//  then leading and trailing spaces are tossed.
// ---------------------------------------
//   INPUT: c4MaxChar is the maximum characters that the passed buffer
//              can hold.
//
//  OUTPUT: szBuf The buffer to put the rest of the line text into.
//
//  RETURN: The number of chars in the buffer, 0 if at the end of the file.
//
static tCIDLib::TCard4
__c4GetRestOfLine(tCIDLib::Tch* szBuf, const tCIDLib::TCard4 c4MaxChars)
{
    // Init the buffer to empty
    szBuf[0] = kCIDLib::chNull;

    //
    //  If the first character is a quote, then this is set and white
    //  space is not trimmed.
    //
    tCIDLib::TBoolean bKeepWS = kCIDLib::False;

    // Loop until we get a whitespace or end of data
    tCIDLib::TCard4 c4Index = 0;
    while (1)
    {
        tCIDLib::Tch chNext = __chGetNext();

        // If its leading whitespace, then skip it
        if (!c4Index && TRawStr::bIsSpace(chNext) && !bKeepWS)
            continue;

        //
        //  If this is the first non-whitespace and its a quote, then this
        //  is a quoted message, so we toss this char and set the bKeepWS
        //  flag.
        //
        if (!c4Index && (chNext == kCIDLib::chQuotation))
        {
            bKeepWS = kCIDLib::True;
            continue;
        }

        //
        //  If we've hit the end of the line, then lets wrap up the
        //  string and do any whitespace stripping and whatnot, then
        //  return.
        //
        if ((chNext == kCIDLib::chLF) || (chNext == kCIDLib::chNull))
        {
            szBuf[c4Index] = 0;

            // If the previous char is a CR, then eat it.
            if (c4Index > 0)
            {
                if (szBuf[c4Index-1] == kCIDLib::chCR)
                {
                    c4Index--;
                    szBuf[c4Index] = kCIDLib::chNull;
                }
            }

            //
            //  If bKeepWS is set, and the last char is a quote, then throw
            //  away the quote.
            //
            if (c4Index && bKeepWS)
            {
                if (szBuf[c4Index-1] == kCIDLib::chQuotation)
                {
                    c4Index--;
                    szBuf[c4Index] = kCIDLib::chNull;
                }
            }

            //
            //  If bKeepWS is not set, then strip leading and trailing
            //  whitespace.
            //
            if (!bKeepWS)
            {
                TRawStr::StripStr
                (
                    szBuf
                    , kCIDLib::szWhitespace
                    , tCIDLib::EStripMode_LeadTrail
                );
            }
            return TRawStr::c4StrLen(szBuf);
        }

        // Watch for an overflow of the caller's buffer
        if (c4Index+1 == c4MaxChars)
            throw L"Message text too long for provided buffer";

        //
        //  If this is a '\' character, then we need to see if its one of
        //  the standard escape sequences and translate it into a single,
        //  binary control character.
        //
        if (chNext == kCIDLib::chBackSlash)
        {
            tCIDLib::Tch chPeek = __chCharSpooler();

            tCIDLib::Tch chActual = 0;
            if (chPeek == kCIDLib::chLatin_r)
                chActual = kCIDLib::chCR;
            else if (chPeek == kCIDLib::chLatin_n)
                chActual = kCIDLib::chLF;
            else if (chPeek == kCIDLib::chLatin_t)
                chActual = kCIDLib::chTab;

            if (chActual)
            {
                szBuf[c4Index++] = chActual;
                continue;
            }
            __chUnget = chPeek;
        }

        // Nothing special, just put it in the string
        szBuf[c4Index++] = chNext;
    }
}


//
// FUNCTION/METHOD NAME: __eGetMsgType
//
// DESCRIPTION:
//
//  This method will get the message type indicator that is supposed to
//  prefix each message in the source file.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: One of the EMsgTypes values.
//
static EMsgTypes __eGetMsgType()
{
    // Init a local buffer to empty
    const tCIDLib::TCard4 c4BufChars = 64;
    tCIDLib::Tch szBuf[c4BufChars+1];
    szBuf[0] = kCIDLib::chNull;

    // Loop until we get a whitespace or end of data
    tCIDLib::TCard4 c4Index = 0;
    tCIDLib::Tch    chNext;
    while (1)
    {
        chNext = __chGetNext();

        // If its leading whitespace, then skip it
        if (!c4Index && TRawStr::bIsSpace(chNext))
            continue;

        if (TRawStr::bIsSpace(chNext) || (chNext == kCIDLib::chNull))
        {
            // Cap it off and convert to a card4 value
            szBuf[c4Index] = 0;

            if (!c4Index)
                return EMsgType_EOF;

            // See if its one of our values.
            if (!TRawStr::eCompareStr(szBuf, L"[M]"))
                return EMsgType_Message;

            if (!TRawStr::eCompareStr(szBuf, L"[E]"))
                return EMsgType_Error;

            return EMsgType_None;
        }

        if (c4Index+1 == c4BufChars)
            throw L"Message type value too long for provided buffer";

        szBuf[c4Index++] = chNext;
    }
}


//
// FUNCTION/METHOD NAME: __OutputBinMsgFile
//
// DESCRIPTION:
//
//  Once the source file has been fully parsed and the output Hpp file
//  created, this is called to complete the work by outputting the binary
//  message file. The information needed is in the global data.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __OutputBinMsgFile()
{
    //
    //  Calculate the offset where the text starts, which is after the
    //  index section. Note we add 4 bytes because the number of messages
    //  is written first.
    //
    tCIDLib::TCard4 c4StartText = (sizeof(CIDMsgFile::TMsgIndexEntry)
                                    * __c4MsgCount) + 4;

    // Write out the number of messages in the file
    __kflTargetMsg.WriteBuffer(&__c4MsgCount, sizeof(tCIDLib::TCard4));

    //
    //  Now that we know the offset of the text, we can write out the index
    //  information.
    //
    TIdInfo* pidiCur = __pidiHead;
    tCIDLib::TCard4 c4NextText = c4StartText;
    while (pidiCur)
    {
        CIDMsgFile::TMsgIndexEntry mieCur;

        // Fill in a message index entry for this element
        mieCur.midThis = pidiCur->midThis;
        mieCur.c2TextChars = tCIDLib::TCard2
        (
            TRawStr::c4StrLen(pidiCur->pszText)+1
        );
        mieCur.c4TextOffset = c4NextText;

        // Now write out the structure
        __kflTargetMsg.WriteBuffer(&mieCur, sizeof(mieCur));

        // Add the bytes to the next text offset
        c4NextText += mieCur.c2TextChars * kCIDLib::c4CharBytes;

        // Move up to the next node
        pidiCur = pidiCur->pidiNext;
    }

    //
    //  Query the current position now. It should be the position that
    //  we calculated as the starting text position above.
    //
    if (__kflTargetMsg.fposFilePtr() != c4StartText)
        throw L"The starting text position != the calculated position";

    //
    //  Now we must make one last pass to write out the text of each
    //  message.
    //
    pidiCur = __pidiHead;
    while (pidiCur)
    {
        // Write out the text of this entry
        __kflTargetMsg.WriteBuffer
        (
            pidiCur->pszText
            , (TRawStr::c4StrLen(pidiCur->pszText) + 1) * kCIDLib::c4CharBytes
        );

        // Move up to the next node
        pidiCur = pidiCur->pidiNext;
    }
}


//
// FUNCTION/METHOD NAME: __ParseAndOutput
//
// DESCRIPTION:
//
//  This function is called when all the files are opened or created and the
//  time has come to parse the input file and output the data.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ParseAndOutput()
{
    static tCIDLib::TErrCode    midCur;
    static tCIDLib::Tch         szIdBuffer[128];
    static tCIDLib::Tch         szTextBuffer[1024];

    //
    //  Get the first char and see if its a UNICode marker. If so, then
    //  set the __bUNICode flag. Else, seek back to start again. Note
    //  that we cannot get this first char via the spooling scheme because
    //  it has to know what kind of data is there in order to work.
    //
    tCIDLib::Tch chUNIMarker;
    __kflSource.ReadBuffer(&chUNIMarker, sizeof(chUNIMarker));
    if (chUNIMarker == kCIDLib::chMarker)
        __bUNICode = kCIDLib::True;
    else
        __kflSource.Seek(0, tCIDLib::ESeekFrom_Start);

    //
    //  Output a comment in the header files, indicating that they
    //  are generated file from this program.
    //
    __WriteToHpp
    (
        __kflErrorHpp
        , "//\n"
          "// This file was generated by the MakeMsgs.Exe utility\n"
          "//\n\n"
    );

    __WriteToHpp
    (
        __kflMessageHpp
        , "//\n"
          "// This file was generated by the MakeMsgs.Exe utility\n"
          "//\n\n"
    );

    //
    //  Write out the opening namespace information to the target Hpp
    //  file.
    //
    __WriteToHpp(__kflErrorHpp, "namespace ");
    __WriteToHpp(__kflMessageHpp, "namespace ");

    __WriteToHpp(__kflErrorHpp, __szErrNameSpace);
    __WriteToHpp(__kflMessageHpp, __szMsgNameSpace);
 
    __WriteToHpp(__kflErrorHpp, "\n{\n");
    __WriteToHpp(__kflMessageHpp, "\n{\n");


    //
    //  We enter a loop here and just read lines from the input file until
    //  we hit the end.
    //
    while (1)
    {
        //
        //  Get the Error/Message identifier. This controls the Hpp file to
        //  which we output. It should be either [E] or [M] meaning either
        //  an error or a message.
        //
        EMsgTypes eType = __eGetMsgType();

        if (eType == EMsgType_EOF)
            break;

        if (eType == EMsgType_None)
            throw L"Expected [E] or [M] message type indicator";

        // Get the next message id name
        if (!__c4GetIdName(szIdBuffer, c4MaxBufChars(szIdBuffer)))
            break;

        // Get the next field, which is the value
        if (!__bGetIdValue(midCur))
        {
            TRawStr::CopyCatStr
            (
                szTextBuffer
                , c4MaxBufChars(szTextBuffer)
                , L"Expected id value for id "
                , szIdBuffer
            );
            throw szTextBuffer;
        }

        // Get its text
        if (!__c4GetRestOfLine(szTextBuffer, c4MaxBufChars(szTextBuffer)))
        {
            TRawStr::CopyCatStr
            (
                szTextBuffer
                , c4MaxBufChars(szTextBuffer)
                , L"Expected text for id '"
                , szIdBuffer
            );
            throw szTextBuffer;
        }

        // Convert the message value to text
        tCIDLib::Tch szIdFmt[64];
        TRawStr::FormatVal(midCur, szIdFmt, c4MaxBufChars(szIdFmt));

        // Write out to the correct header
        if (eType == EMsgType_Error)
        {
            __WriteToHpp(__kflErrorHpp, "    const tCIDLib::TErrCode ");
            __WriteToHpp(__kflErrorHpp, szIdBuffer);
            __WriteToHpp(__kflErrorHpp, " = ");
            __WriteToHpp(__kflErrorHpp, szIdFmt);
            __WriteToHpp(__kflErrorHpp, ";\n");
        }
         else
        {
            __WriteToHpp(__kflMessageHpp, "    const tCIDLib::TMsgId ");
            __WriteToHpp(__kflMessageHpp, szIdBuffer);
            __WriteToHpp(__kflMessageHpp, " = ");
            __WriteToHpp(__kflMessageHpp, szIdFmt);
            __WriteToHpp(__kflMessageHpp, ";\n");
        }

        //
        //  Add a new element to the list of id information structures
        //
        TIdInfo* pidiNew = new TIdInfo;
        if (!__pidiHead)
        {
            __pidiHead = pidiNew;
            pidiNew->pidiNext = 0;
        }
         else
        {
            pidiNew->pidiNext = __pidiHead;
            __pidiHead = pidiNew;
        }
        pidiNew->pszIdName = TRawStr::pszReplicate(szIdBuffer);
        pidiNew->pszText = TRawStr::pszReplicate(szTextBuffer);
        pidiNew->midThis = midCur;

        // Bump up the count of ids
        __c4MsgCount++;
    }

    // And write out the ending header namespace stuff
    __WriteToHpp(__kflErrorHpp, "};\n\n");
    __WriteToHpp(__kflMessageHpp, "};\n\n");

    //
    //  Now we can create the output binary message text file. We call
    //  another method which will use the global data we've built to
    //  do the file.
    //
    __OutputBinMsgFile();
}


//
// FUNCTION/METHOD NAME: __ShowBlurb
//
// DESCRIPTION:
//
//  This function just shows a small opening program blurb.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowBlurb()
{
    __kconOutput.PutLine
    (
        L"MakeMsgs.Exe, CIDLib Message File Compiler\n"
        L"Compiled on: "
    );
    __kconOutput.PutLine(__DATE__);
    __kconOutput.PutLine(L"\r\n\n");
}


//
// FUNCTION/METHOD NAME: __ShowReturnVal
//
// DESCRIPTION:
//
//  This function shows information about a return value.
// ---------------------------------------
//   INPUT: eRet is the return value that is about to be returned.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowReturnVal(const EReturnVals eRet)
{
    if (eRet == EReturn_Successful)
    {
        __kconOutput.PutLine(L"No Error");
    }
    else if (eRet == EReturn_CantOpenSource)
    {
        __kconOutput.PutLine(L"opening input file: ");
        __kconOutput.PutLine(__szSourceFl);
    }
    else if (eRet == EReturn_CantCreateErrHpp)
    {
        __kconOutput.PutLine(L"creating output error Hpp file: ");
        __kconOutput.PutLine(__szErrorHpp);
    }
    else if (eRet == EReturn_CantCreateMsgHpp)
    {
        __kconOutput.PutLine(L"createing output message Hpp file: ");
        __kconOutput.PutLine(__szMessageHpp);
    }
    else if (eRet == EReturn_CantCreateMsg)
    {
        __kconOutput.PutLine(L"creating output Msg file: ");
        __kconOutput.PutLine(__szTargetFl);
    }
    else if (eRet == EReturn_InvalidFormat)
    {
        __kconOutput.PutLine(L"parsing the source file: ");
        __kconOutput.PutLine(__szSourceFl);
    }
    else if (eRet == EReturn_Parameters)
    {
        __kconOutput.PutLine(L"parsing command line parameters");
    }
}


//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  This method just shows the usage of the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowUsage()
{
    __kconOutput.PutLine
    (
        L"Usage:\n\n"
        L"   MakeMsgs fac errns msgns outpath\n"
        L"      fac is the facility name\n"
        L"      errns is the namespace to generate in the error Hpp file\n"
        L"      msgns is the namespace to generate in the messages Hpp file\n"
        L"      outpath is the path to output the binary file into.\n"
        L"or\n"
        L"   MakeMsgs /Attach exepath msgpath\n"
        L"      exepath is the path to the DLL/Exe file\n"
        L"      msgpath is the path to the binary msg file\n\n"
    );
}


//
// FUNCTION/METHOD NAME: __WriteToHpp
//
// DESCRIPTION:
//
//  This method is a convenience to write out text lines to the output
//  Hpp files. The raw kernel file class only supports binary output directly. 
//  Text to the Hpp file needs to be in short char format, because source
//  file cannot be UNICode, so it converts text before writing.
// ---------------------------------------
//   INPUT: kflTarget is the file to write to.
//          pszToWrite is the string to write out
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid
__WriteToHpp(TKrnlFile& kflTarget, const tCIDLib::Tch* const pszToWrite)
{
    // Convert the string to short char format
    tCIDLib::Tsch* pszTmp = TRawStr::pszConvert(pszToWrite);
    kflTarget.WriteBuffer(pszTmp, TRawStr::c4StrLen(pszTmp));
    delete pszTmp;
}

static tCIDLib::TVoid
__WriteToHpp(TKrnlFile& kflTarget, const tCIDLib::Tsch* const pszToWrite)
{
    kflTarget.WriteBuffer(pszToWrite, TRawStr::c4StrLen(pszToWrite));
}


// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: main
//
// DESCRIPTION:
//
//  This guy checks the input parms and gets all of the files prepared, then
//  calls a method to do the parsing and output.
// ---------------------------------------
//   INPUT: i4Args is the argument count.
//          apszArgs is the list of arguments.
//
//  OUTPUT: None
//
//  RETURN: One of the EReturnVals values.
//
extern "C" int wmain(tCIDLib::TInt4 i4Args, tCIDLib::Tch* apszArgs[])
{
    const tCIDLib::Tch* pszReason = L"Unknown";
    EReturnVals eRet = EReturn_Successful;

    // Display a little blurb
    __ShowBlurb();

    //
    //  See if this is an attach operation. If so, then there are 4
    //  params, and the 2nd one must be /Attach.
    //
    if (i4Args == 4)
    {
        if (TRawStr::eICompareStr(apszArgs[1], L"/ATTACH"))
        {
            __ShowUsage();
            return EReturn_Parameters;
        }
        TRawStr::CopyStr(__szSourceFl, apszArgs[3], __c4MaxParmLen);
        TRawStr::CopyStr(__szTargetFl, apszArgs[2], __c4MaxParmLen);
        return __eAttach();
    }

    // We have to have 5 parameters, our 4 plus the standard 1st one
    if (i4Args != 5)
    {
        __ShowUsage();
        return EReturn_Parameters;
    }

    //
    //  Get the command line parms out into our local buffers and add the
    //  appropriate extensions to files. The first one is the facility
    //  name and is used to build the other file names.
    //
    TRawStr::CopyStr(__szFacilityName, apszArgs[1], __c4MaxParmLen);
    TRawStr::CopyStr(__szErrNameSpace, apszArgs[2], __c4MaxParmLen);
    TRawStr::CopyStr(__szMsgNameSpace, apszArgs[3], __c4MaxParmLen);

    //
    //  Get the output path into the taret message buffer and then build
    //  it up to the full output file name.
    //
    tCIDLib::TCard4 c4Len = TRawStr::c4StrLen(apszArgs[4]);
    if (c4Len)
    {
        TRawStr::CopyStr(__szTargetFl, apszArgs[4], __c4MaxParmLen);
        if (__szTargetFl[c4Len-1] != L'\\')
        {
            TRawStr::CatStr
            (
                __szTargetFl
                , kCIDLib::szPathSeparator
                , __c4MaxParmLen
            );
        }
    }
     else
    {
        __szTargetFl[0] = kCIDLib::chNull;
    }
    TRawStr::CatStr(__szTargetFl, __szFacilityName, __c4MaxParmLen);
    TRawStr::CatStr(__szTargetFl, L".CIDMsg", __c4MaxParmLen);

    //
    //  Now build up the name of the source file Its assumed to be in the
    //  project directory (i.e. the current dir.) Its made up from the
    //  facility name.
    //
    TRawStr::CopyStr(__szSourceFl, __szFacilityName, __c4MaxParmLen);
    TRawStr::CatStr(__szSourceFl, L".MsgText", __c4MaxParmLen);

    //
    //  Build up the names of the two output Hpp files. They are built from
    //  the facility name.
    //
    TRawStr::CopyStr(__szErrorHpp, __szFacilityName, __c4MaxParmLen);
    TRawStr::CatStr(__szErrorHpp, L"_ErrorIds.Hpp", __c4MaxParmLen);

    TRawStr::CopyStr(__szMessageHpp, __szFacilityName, __c4MaxParmLen);
    TRawStr::CatStr(__szMessageHpp, L"_MessageIds.Hpp", __c4MaxParmLen);

    try
    {
        //
        //  Now lets try to open and/or create the files that we were told to
        //  use. Start with the input file, which we open in exclusive read
        //  and write mode.
        //
        eRet = EReturn_CantOpenSource;
        __kflSource.SetName(__szSourceFl);
        __kflSource.Open
        (
            tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_OpenIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        //
        //  Lets open the the three output files for writing, truncating any
        //  existing content.
        //
        eRet = EReturn_CantCreateErrHpp;
        __kflErrorHpp.SetName(__szErrorHpp);
        __kflErrorHpp.Open
        (
            tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        eRet = EReturn_CantCreateMsgHpp;
        __kflMessageHpp.SetName(__szMessageHpp);
        __kflMessageHpp.Open
        (
            tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );

        eRet = EReturn_CantCreateMsg;
        __kflTargetMsg.SetName(__szTargetFl);
        __kflTargetMsg.Open
        (
            tCIDLib::EAccess_Excl_ReadWrite
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_RandomAccess
        );
    }

    catch(const TKrnlError& kerrToCatch)
    {
        __kconOutput.PutLine(L"Error occured while ");
        __ShowReturnVal(eRet);
        __kconOutput.PutLine(L"\r\n\n");

        __kconOutput.PutLine(L"   Error Code: ");
        tCIDLib::Tch szNum[16];
        TRawStr::FormatVal(kerrToCatch.errcId(), szNum, c4MaxBufChars(szNum));
        __kconOutput.PutLine(szNum);
        __kconOutput.PutLine(L"\r\n\n");

        return eRet;
    }

    //
    //  Everyone seems happy, so lets call the function that does the
    //  parsing and creates the output files.
    //
    try
    {
        __ParseAndOutput();
    }

    catch(const tCIDLib::Tch* const pszError)
    {
        __kconOutput.PutLine
        (
            L"An error occured while parsing the input file.\r\n"
        );
        __kconOutput.PutLine(L"   Error: ");
        __kconOutput.PutLine(pszError);
        __kconOutput.PutLine(L", Line Number: ");

        tCIDLib::Tch szLineNum[16];
        TRawStr::FormatVal(__c4CurLine, szLineNum, c4MaxBufChars(szLineNum));
        __kconOutput.PutLine(szLineNum);
        __kconOutput.PutLine(L"\r\n\n");
        return EReturn_InvalidFormat;
    }

    catch(const TKrnlError& kerrToCatch)
    {
        __kconOutput.PutLine(L"Error occured while parsing the input file.");

        __kconOutput.PutLine(L"   Error Code: ");
        tCIDLib::Tch szNum[16];
        TRawStr::FormatVal(kerrToCatch.errcId(), szNum, c4MaxBufChars(szNum));
        __kconOutput.PutLine(szNum);

        __kconOutput.PutLine(L", Line Number: ");
        TRawStr::FormatVal(__c4CurLine, szNum, c4MaxBufChars(szNum));
        __kconOutput.PutLine(szNum);
        __kconOutput.PutLine(L"\r\n\n");

        return EReturn_InvalidFormat;
    }

    // Went ok
    return EReturn_Successful;
}
