//
// NAME: MakeDocs_Translator.Cpp
//
// DESCRIPTION: 
//
//  This module implements the couple of implementation methods of the
//  abstract base class TTranslator. Most of the class is pure virtual but
//  some of it can be written in terms of that abstract interface.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "MakeDocs.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TTranslator,TObject)


// ----------------------------------------------------------------------------
//  Local constant values
// ----------------------------------------------------------------------------
const tCIDLib::TCard4 __c4HashModulus = 17;


// ----------------------------------------------------------------------------
//  Local types
//
//  TTagItem
//      This is a small structure that is used to map tag text to their
//      enum values and vice versa.
// ----------------------------------------------------------------------------
struct TTagItem
{
    TTranslator::ETags      eTag;
    tCIDLib::Tch*           pszTag;
    tCIDLib::THashVal       hshTag;
};


// ----------------------------------------------------------------------------
//  Local data
//
//  __atagList
//      This is an array of TTagItem structures that map all off the tags
//      and enums.
// ----------------------------------------------------------------------------
static TTagItem __atagList[] =
{
        { TTranslator::ETag_Bold            , L"b"      , 0 }
    ,   { TTranslator::ETag_Break           , L"br"     , 0 }
    ,   { TTranslator::ETag_Divider         , L"hr"     , 0 }
    ,   { TTranslator::ETag_EndBold         , L"eb"     , 0 }
    ,   { TTranslator::ETag_EndHeading1     , L"eh1"    , 0 }
    ,   { TTranslator::ETag_EndHeading2     , L"eh2"    , 0 }
    ,   { TTranslator::ETag_EndHeading3     , L"eh3"    , 0 }
    ,   { TTranslator::ETag_EndParagraph    , L"ep"     , 0 }
    ,   { TTranslator::ETag_EndItalic       , L"ei"     , 0 }
    ,   { TTranslator::ETag_EndList         , L"els"    , 0 }
    ,   { TTranslator::ETag_EndSource       , L"epre"   , 0 }
    ,   { TTranslator::ETag_EndTable        , L"etb"    , 0 }
    ,   { TTranslator::ETag_EndTableCol1    , L"etb1"   , 0 }
    ,   { TTranslator::ETag_EndTableCol2    , L"etb2"   , 0 }
    ,   { TTranslator::ETag_Heading1        , L"h1"     , 0 }
    ,   { TTranslator::ETag_Heading2        , L"h2"     , 0 }
    ,   { TTranslator::ETag_Heading3        , L"h3"     , 0 }
    ,   { TTranslator::ETag_Indent          , L"id"     , 0 }
    ,   { TTranslator::ETag_Italic          , L"i"      , 0 }
    ,   { TTranslator::ETag_List            , L"ls"     , 0 }
    ,   { TTranslator::ETag_ListItem        , L"lsi"    , 0 }
    ,   { TTranslator::ETag_Outdent         , L"od"     , 0 }
    ,   { TTranslator::ETag_Paragraph       , L"p"      , 0 }
    ,   { TTranslator::ETag_Source          , L"pre"    , 0 }
    ,   { TTranslator::ETag_Table           , L"tb"     , 0 }
    ,   { TTranslator::ETag_EndTableCol1    , L"tb1"    , 0 }
    ,   { TTranslator::ETag_EndTableCol2    , L"tb2"    , 0 }
};
const tCIDLib::TCard4 __c4TagCount = c4ArrayElems(__atagList);



// ----------------------------------------------------------------------------
//  CLASS: TTranslator
// PREFIX: trans
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TTranslator: Constructors and destructors
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TTranslator
//
// DESCRIPTION:
//
//  On the first time through, this guy will initialize the hashes for
//  the tag text in the __atagList above.
// -------------------------------------
//   INPUT: strSpecialChars is the list of special characters that the
//              derived class wants to handle himself.
//
//  OUTPUT: None
//
//  RETURN: None
//
TTranslator::TTranslator(const TString& strSpecialChars) :

    __pstrmTarget(0)
    , __strSpecialChars(strSpecialChars)
{
    static tCIDLib::TBoolean bFirstTime = kCIDLib::True;
    if (bFirstTime)
    {
        bFirstTime = kCIDLib::False;

        // Do the hashes for all of the tokens.
        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TagCount; c4Ind++)
        {
            __atagList[c4Ind].hshTag = TRawStr::hshHashStr
            (
                __atagList[c4Ind].pszTag
                , __c4HashModulus
            );
        }
    }
}


// ----------------------------------------------------------------------------
//  TTranslator: Public, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TranslateTagText
//
// DESCRIPTION:
//
//  This method is called after a tag is hit that has associated free
//  form text. This method will copy text from the source parser to the
//  output stream until it hits a '@' character. It will unget this
//  character and return. If it hits any formatting tags along the way
//  it will call the derived class to have the format specific text output
//  to the output stream.
// -------------------------------------
//   INPUT: psrsrSource is the source parser object to get text from.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TTranslator::TranslateTagText(TDocParser& prsrSource)
{
    //
    //  Just start looping through, one character at a time. Because of the
    //  nature of this operation, there really is not any more efficient
    //  way to do it. We do try to be more efficient by builing up small
    //  arrays of chars then writing them all at once.
    //
    tCIDLib::TBoolean       bInPreBlock = kCIDLib::False;
    const tCIDLib::TCard4   c4BufSz = 128;
    tCIDLib::TCard4         c4BufInd = 0;
    tCIDLib::Tch            chCur;
    tCIDLib::Tch            szBuf[c4BufSz+1];
    while (1)
    {
        // Get the next character
        chCur = prsrSource.chGetNextChar();

        // If we get a nul, then we've hit the end
        if (chCur == kCIDLib::chNull)
        {
            // Flush out any chars in the temp buffer
            if (c4BufInd)
            {
                szBuf[c4BufInd] = kCIDLib::chNull;
                *__pstrmTarget << szBuf;
                c4BufInd = 0;
            }
            return;
        }

        // If its @, then its a token, so we are done
        if (chCur == L'@')
        {
            // Flush out any chars in the temp buffer
            if (c4BufInd)
            {
                szBuf[c4BufInd] = kCIDLib::chNull;
                *__pstrmTarget << szBuf;
                c4BufInd = 0;
            }

            // Unget the character
            prsrSource.UnGetChar(chCur);
            return;
        }

        //
        //  If its a /, then see if the next character is another one.
        //  If so, its a comment, so we flush the rest of the current
        //  line. If we are in a preformatted block, then let it through
        //  though.
        //
        if ((chCur == L'/') && !bInPreBlock)
        {
            tCIDLib::Tch chNext = prsrSource.chGetNextChar();
            if (chNext == L'/')
            {
                prsrSource.FlushLine();
                continue;
            }

            // It wasn't so unget the next char
            prsrSource.UnGetChar(chNext);
        }

        //
        //  If it is a '<' character then it could be one of our abstract
        //  formatting tags, so we have to look ahead and see if we can get
        //  a match.
        //
        if (chCur == L'<')
        {
            // Flush out any chars in the temp buffer
            if (c4BufInd)
            {
                szBuf[c4BufInd] = kCIDLib::chNull;
                *__pstrmTarget << szBuf;
                c4BufInd = 0;
            }

            //
            //  Ok, so lets look forward and see if we find a closing 
            //  bracket. If we hit a new line or any white space we know
            //  its not a token. If we fill the temp buffer we know its
            //  not because no tokesn are that big.
            //
            szBuf[0] = kCIDLib::chNull;

            tCIDLib::TBoolean bGotToken = kCIDLib::False;
            while (!bGotToken)
            {
                // Get the next char
                tCIDLib::Tch chNext = prsrSource.chGetNextChar();

                if (chNext == kCIDLib::chNull)
                    break;

                // If its the closing brace, lets test the text we got
                if (chNext == L'>')
                {
                    //
                    //  See if the text we found maps to one of our tags.
                    //  If so, then output the tag.
                    //
                    ETags eTag = __eMapToTag(szBuf);

                    if (eTag != ETag_None)
                    {
                        // Output this tag in the correct target format
                        OutputTag(eTag);

                        // Watch for pre and endpre blocks and toggle flag
                        if (eTag == TTranslator::ETag_Source)
                            bInPreBlock = kCIDLib::True;
                        else
                            bInPreBlock = kCIDLib::False;

                        // Continue back to the top
                        bGotToken = kCIDLib::True;
                        continue;
                    }
                     else
                    {
                        // No tag, so unget the next char and break out
                        prsrSource.UnGetChar(chNext);
                        break;
                    }
                }
                 else
                {
                    //
                    //  Put it into the temp buffer. If we fill the temp
                    //  buffer, then give up and unget the whole thing.
                    //
                    szBuf[c4BufInd++] = chNext;
                    szBuf[c4BufInd] = kCIDLib::chNull;

                    if (c4BufInd == c4BufSz)
                        break;
                }
            }

            // If we got a token, then skip back to the top
            if (bGotToken)
            {
                c4BufInd = 0;
                continue;
            }

            // We did not get the token, so unget the text and fall through
            prsrSource.UnGetToken(szBuf);
            c4BufInd = 0;
        }

        // If its a special char then let the derived class handle it
        tCIDLib::TCard4 c4Pos;
        if (__strSpecialChars.bFirstOccurrence(chCur, c4Pos))
        {
            // Flush out any chars in the temp buffer
            if (c4BufInd)
            {
                szBuf[c4BufInd] = kCIDLib::chNull;
                *__pstrmTarget << szBuf;
                c4BufInd = 0;
            }

            TranslateSpecialChar(chCur);
            continue;
        }
         else
        {
            //
            //  Its nothing special so just pass it on to the output stream.
            //  If the buffer is full, then flush it.
            //
            if (c4BufInd == c4BufSz)
            {
                szBuf[c4BufInd] = kCIDLib::chNull;
                *__pstrmTarget << szBuf;
                c4BufInd = 0;
            }

            szBuf[c4BufInd++] = chCur;
        }
    }
}


// ----------------------------------------------------------------------------
//  TTranslator: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMapToTag
//
// DESCRIPTION:
//
//  This 
// -------------------------------------
//   INPUT: strTag is the text of the tag to map
//
//  OUTPUT: None
//
//  RETURN: The ETags value that the text maps to
//
TTranslator::ETags TTranslator::__eMapToTag(const TString& strTag)
{
    // Get the hash of the passed tag text
    tCIDLib::THashVal hshTag = strTag.hshCalcHash(__c4HashModulus);

    //
    //  Now use that to check each tag for a possible match. If the
    //  hash matches, then do the actual comparision.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TagCount; c4Index++)
    {
        if (__atagList[c4Index].hshTag == hshTag)
        {
            if (strTag == __atagList[c4Index].pszTag)
                return __atagList[c4Index].eTag;
        }
    }
    return ETag_None;
}
