//
// NAME: MakeDocs_ThisFacility.Cpp
//
// DESCRIPTION: 
//
//  This module implements the facility class for this program. This program's
//  facility class provides the centralized access to program functionality.
//  Everything goes through it, making it easy and convenient to coordinate
//  the program's work.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//

// -----------------------------------------------------------------------------
//  Include underlying headers.
// -----------------------------------------------------------------------------
#include    "MakeDocs.Hpp"



// ----------------------------------------------------------------------------
//  CLASS: TFacMakeDocs
// PREFIX: fac
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TFacMakeDocs: Constructors and destructors
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TFacMakeDocs
//                       ~TFacMakeDocs
//
// DESCRIPTION:
//
//  The constructor just calls its parent and does some basic init of members
//  to defaults so that, if the user does not provide them on the command
//  line, they are reasonable.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
TFacMakeDocs::TFacMakeDocs() :

    TFacility
    (
        L"MakeDocs"
        , tCIDLib::EModType_Exe
        , kCIDLib::c4MajVersion
        , kCIDLib::c4MinVersion
    )
    , __bForceUpdate(kCIDLib::False)
    , __ptransOut(0)
{
    //
    //  Create a translator. Note that its not really useful until later
    //  when we can tell it the output stream its going to use.
    //
    __ptransOut = new THtmlTranslator();
}

TFacMakeDocs::~TFacMakeDocs()
{
}


// ----------------------------------------------------------------------------
//  TFacMakeDocs: Public, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: bParseCommandLine
//
// DESCRIPTION:
//
//  This method is called on program startup to parse out everything from
//  the command line and make sure its happy. The information is stored in
//  member data for later use.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
tCIDLib::TBoolean TFacMakeDocs::bParseCommandLine()
{
    // Get a quicky reference to the number of arguments
    const tCIDLib::TCard4 c4NumArgs = TSysInfo::c4CmdLineArgCount();

    //
    //  We need to have at least target directory and a source directory. If
    //  we don't get both, then we show an error and exit.
    //
    if (c4NumArgs < 3)
    {
        // Show the usage and exit
        __conOut << facMakeDocs.strMsg(kDocMsgs::midUsage) << DNewLn;
        return kCIDLib::False;
    }

    // Get the two parms out
    TSysInfo::CmdLineArg(1, __pathSourceDir);
    TSysInfo::CmdLineArg(2, __pathTargetDir);

    // If either of them end with slashes, then remove them
    __pathSourceDir.bRemoveTrailingSeparator();
    __pathTargetDir.bRemoveTrailingSeparator();

    // Check the two paths
    if (!__bCheckPath(__pathSourceDir, kCIDLib::False))
        return kCIDLib::False;

    if (!__bCheckPath(__pathTargetDir, kCIDLib::True))
        return kCIDLib::False;

    //
    //  If there are any more parameters, check them for valid options
    //  and do what they ask.
    //
    for (tCIDLib::TCard4 c4Index = 3; c4Index < c4NumArgs; c4Index++)
    {
        TString strParm;
        TSysInfo::CmdLineArg(c4Index, strParm);

        if (strParm == L"/FORCE")
            __bForceUpdate = kCIDLib::True;
    }

    return kCIDLib::True;
}


//
// FUNCTION/METHOD NAME: GenerateDocs
//
// DESCRIPTION:
//
//  This method is the top level workhorse method. This one does this:
//
//      1)  Scans the main source directory and gets the name of all of
//          subsystem names (i.e. the directory names.)
//
//      2)  For each subdirectory, it generates the main file which has a
//          link to each subsystem main page. This is done by calling the
//          __GenSubSystemPage().
//
//      3)  It goes back and iterates then again and calls __DoSubSystem()
//          for each subsystem. This will iterate all of the files in that
//          subsystem and generate output files for them.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacMakeDocs::GenerateDocs()
{
    //
    //  Create a directory iterator and iterate all of the subdirectories
    //  of the main source directory.
    //
    TDirIter    diterSubs;
    TFindBuf    fndbCurrent;

    TPathStr    pathToFind(__pathSourceDir);
    pathToFind.AddLevel(L"*");
    if (!diterSubs.bFindFirst
    (
        pathToFind
        , fndbCurrent
        , tCIDLib::EFileAttr_Directory
        , tCIDLib::EFileAttr_Directory))
    {
        __conOut << facMakeDocs.strMsg(kDocMsgs::midNoSubsInSource)
                 << NewLn;
        return;
    }

    //
    //  Lets create a text file stream for the main output file. This file
    //  contains links to the main subsystem pages. We know that all output
    //  files have to be ASCII format.
    //
    TPathStr pathOut(__pathTargetDir);
    pathOut.AddLevel(L"MainIndex");
    pathOut.AppendExtension(__ptransOut->strFileExtension());
    TTextFileStream strmTarget
    (
        pathOut
        , tCIDLib::ETextFmt_ASCII
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_ReplaceIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    // Set the output stream of the translator object
    __ptransOut->SetOutputStream(strmTarget);

    // Output the standard page header
    __ptransOut->OutputPageHeader(L"CIDLib Class and Member Documentation");

    //
    //  Output some canned text here. There is no need to parse a file for
    //  this because its always the same
    //
    __ptransOut->OutputTag(TTranslator::ETag_Heading1);
    strmTarget << L"CIDLib Class and Member Documentation";
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading1);

    strmTarget <<
        L"This is the main page of the online class and member documentation"
        L" of the CIDLib system. This file provides links to the main pages"
        L" of each subsystem (or facility as they are called in CIDLib"
        L" parlance.) These pages in turn provide links to each class available"
        L" within that facility.";

    // Output the heading for the link section
    __ptransOut->OutputTag(TTranslator::ETag_Heading2);
    strmTarget << L"CIDLib Facilities:";
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

    // Indent all of the links
    __ptransOut->OutputTag(TTranslator::ETag_Indent);

    //
    //  Now we loop through all of the subdirectories and generate the
    //  main page for each subsystem.
    //
    do
    {
        // Be sure to skip the special directories.
        if (!fndbCurrent.bIsSpecialDir())
        {
            TPathStr pathSubSys(fndbCurrent.pathFileName());
            pathSubSys.bRemovePath();
            pathSubSys.bRemoveExt();

            //
            //  Look for particular subdirectories which hold special files,
            //  which are to just be copied as is to the output directory.
            //  Otherwise, we want to process the class doc files in this
            //  subdirectory.
            //
            if ((pathSubSys == L"Images") || (pathSubSys == L"HandHewn"))
            {
                __CopyFiles(pathSubSys);
            }
             else
            {
                //
                //  Set the translator's output stream back to our stream and
                //  output a link for this subsystem.
                //
                __ptransOut->SetOutputStream(strmTarget);
                __ptransOut->OutputLink(pathSubSys, pathSubSys);
                __ptransOut->OutputTag(TTranslator::ETag_Break);

                //  Lets cause the main page for this subsystem to be generated.
                __GenSubSystemPage(pathSubSys);

                // And now process all of the files in this subsystem
                __DoSubSystem(pathSubSys);
            }
        }
    }   while (diterSubs.bFindNext(fndbCurrent));

    // Set the translator's output stream back to our stream
    __ptransOut->SetOutputStream(strmTarget);

    // Undo the indent from above
    __ptransOut->OutputTag(TTranslator::ETag_Outdent);

    // Output the standard page footer
    __ptransOut->OutputPageFooter();
}


//
// FUNCTION/METHOD NAME: ShowBlurb
//
// DESCRIPTION:
//
//  This method just shows a little program blurb.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacMakeDocs::ShowBlurb()
{
    TString strBlurb
    (
        kDocMsgs::midBlurb
        , facMakeDocs
        , 128
        , facCIDLib.strVersion()
    );
    __conOut << strBlurb << DNewLn;
}


// ----------------------------------------------------------------------------
//  TFacMakeDocs: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bCheckPath
//
// DESCRIPTION:
//
//  This method will check to see if the passed path exists. If it does, it
//  makes sure its a directory. If not, it returns False. If it does not
//  exist, it queries if the user wants to create it. If so, it creates the
//  directory. bCreateOk must be True in order for the directory to be
//  created. If bCreateOk is False, then its an error not to exist.
// -------------------------------------
//   INPUT: strToCheck is the path to check.
//
//  OUTPUT: None
//
//  RETURN: True if successful, else False.
//
tCIDLib::TBoolean
TFacMakeDocs::__bCheckPath( const   TString&            strToCheck
                            , const tCIDLib::TBoolean   bCreateOk)
{
    //
    //  Check to see if they both exist. If they do not, then query whether
    //  we should create them or not.
    //
    if (TFileSys::bExists(strToCheck))
    {
        // Make sure its a directory
        if (!TFileSys::bIsDirectory(strToCheck))
        {
            __conOut << facMakeDocs.strMsg(kDocMsgs::midMustBeDir, strToCheck)
                     << DNewLn;
            return kCIDLib::False;
        }
    }
     else
    {
        if (!bCreateOk)
        {
            __conOut << facMakeDocs.strMsg( kDocMsgs::midDoesNotExist
                                            , strToCheck) << DNewLn;
            return kCIDLib::False;
        }

        //
        //  See if the user wants to try to create this path and use it
        //  anyway.
        //
        TString strInput(kCIDLib::pszEmptyZStr, 64, 64);
        __conOut << facMakeDocs.strMsg(kDocMsgs::midDoesNotExistQ, strToCheck);

        if (!__conOut.c4GetLine(strInput))
            return kCIDLib::False;
        __conOut << NewLn;

        if (strInput.eICompare(L"Yes") && strInput.eICompare(L"Y"))
            return kCIDLib::False;

        try
        {
            TFileSys flsysTmp;
            flsysTmp.MakeDirectory(strToCheck, NUL_TString);
        }

        catch(const TError& errToCatch)
        {
            __conOut << facMakeDocs.strMsg(kDocMsgs::midDirMakeFailed)
                     << errToCatch.strErrText() << DNewLn;
            return kCIDLib::False;
        }
    }
    return kCIDLib::True;
}


//
// FUNCTION/METHOD NAME: __bOutOfDate
//
// DESCRIPTION:
//
//  This method will see if the first file is out of date with respect to the
//  second file. This is used to know when files need to get regenerated.
// -------------------------------------
//   INPUT: strFileToCheck is the file to check.
//          strSourceFile is the source file that the first file is checked
//              against.
//
//  OUTPUT: None
//
//  RETURN: kCIDLib::True if the first file has an earlier last modification
//              time than the source file.
//
tCIDLib::TBoolean
TFacMakeDocs::__bOutOfDate(const TString& strFileToCheck, const TString& strSourceFile)
{
    // If the user wants to force an update, just return true
    if (__bForceUpdate)
        return kCIDLib::True;

    //
    //  See if the file to check even exists. If not, then there is no need
    //  to go further, it must be regenerated. If so, then we will get back
    //  the file info on this file.
    //
    TFindBuf fndbToCheck;
    if (!TFileSys::bExists
    (
        strFileToCheck
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileAttr_None
        , fndbToCheck))
    {
        return kCIDLib::True;
    }

    // Ok, so get the file information on the source file
    TFindBuf fndbSource;
    if (!TFileSys::bExists
    (
        strSourceFile
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileAttr_None
        , fndbSource))
    {
        // This is an error
        throw TDocError
        (
            kDocErrs::errcSourceNotFound
            , 0
            , strSourceFile
        );
    }

    // See who has the earliest last modification time
    return (fndbToCheck.tmLastModify() < fndbSource.tmLastModify());
}


//
// FUNCTION/METHOD NAME: __CopyFiles
//
// DESCRIPTION:
//
//  This method is called from the main processing method above. Its called
//  to copy the contents of a subdirectory to the output directory.
// -------------------------------------
//   INPUT: strSubDir is the subdirectory to copy to the output.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacMakeDocs::__CopyFiles(const TString& strSubDir)
{
    // Build the path we want to iterate
    TPathStr    pathSource(__pathSourceDir);
    pathSource.AddLevel(strSubDir);
    pathSource.AddLevel(kCIDLib::pszAllFilesSpec);

    //
    //  Create a directory iterator and iterate all of the files in the
    //  source directory.
    //
    TDirIter    diterSource;
    TFindBuf    fndbCurrent;

    if (!diterSource.bFindFirst(pathSource, fndbCurrent))
    {
        __conOut << facMakeDocs.strMsg(kDocMsgs::midNoFilesIn, strSubDir)
                 << NewLn;
        return;
    }

    __conOut << facMakeDocs.strMsg(kDocMsgs::midCopyingFiles, strSubDir)
             << NewLn;
    TPathStr pathTmp;
    do
    {
        if (!fndbCurrent.bIsSpecialDir())
        {
            // Build the target path
            pathTmp = fndbCurrent.pathFileName();
            pathTmp.bRemovePath();
            TPathStr pathTarget(__pathTargetDir);
            pathTarget.AddLevel(pathTmp);

            //
            //  And copy the file if its out of date with its target
            //  version.
            //
            if (__bOutOfDate(pathTarget, fndbCurrent.pathFileName()))
                TFileSys::CopyFile(fndbCurrent.pathFileName(), pathTarget);
        }

    }   while (diterSource.bFindNext(fndbCurrent));
}


//
// FUNCTION/METHOD NAME: __DoSubSystem
//
// DESCRIPTION:
//
//  This is the second level method. When a subdirectory is found, which we
//  know that it is the name of a subsystem and contains document files for
//  that subsystem. So this method is called with the subsystem name. We
//  iterate the class files in it and generate docs for each class.
// -------------------------------------
//   INPUT: strSubName is the name of the subsystem that we are to do.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TFacMakeDocs::__DoSubSystem(const TString& strSubName)
{
    //
    //  Create a directory iterator and iterate all of the ClassDoc files
    //  in the source directory.
    //
    TDirIter    diterSource;
    TFindBuf    fndbCurrent;

    TPathStr    pathToFind(__pathSourceDir);
    pathToFind.AddLevel(strSubName);
    pathToFind.AddLevel(L"*.ClassDoc");
    if (!diterSource.bFindFirst(pathToFind, fndbCurrent))
    {
        __conOut << facMakeDocs.strMsg( kDocMsgs::midNoDocsInSource
                                        , __pathSourceDir) << NewLn;
        return;
    }

    __conOut    << facMakeDocs.strMsg(kDocMsgs::midXlatingFiles, strSubName)
                << NewLn;
    do
    {
        //
        //  Since there are potentiall a lot of files, put each loop into
        //  a faux block to force cleanup of everything on each round.
        //
        {
            //
            //  Create the path to the source file and the path to the target
            //  file and try to create them both. The source path is in the
            //  find buffer. The target extension is generated via the
            //  translator so it can do what is required for files of its
            //  target type.
            //
            TPathStr pathSrcFile(fndbCurrent.pathFileName());

            //
            //  Get the output file in a temp. Strip off its extension and
            //  path, leaving just the base name. We need this base name in
            //  other places below.
            //
            TPathStr pathTmp(fndbCurrent.pathFileName());
            pathTmp.bRemoveExt();
            pathTmp.bRemovePath();

            //
            //  Now start the target file with the target dir. Then tack on
            //  the temp name that we just created, and the extension that
            //  the translator wants.
            //
            TPathStr    pathTargetFile(__pathTargetDir);
            pathTargetFile.AddLevel(pathTmp);
            pathTargetFile.AppendExtension(__ptransOut->strFileExtension());

            //
            //  See if we need to update this file. We only need to if its
            //  out of date.
            //
            if (__bOutOfDate(pathTargetFile, pathSrcFile))
            {
                // Output the files that we are working on for this go around.
                __conOut << L"   " << pathSrcFile << NewLn;

                //
                //  Now lets open the source file and create the target file,
                //  overwriting any current contents.
                //
                TTextFileStream strmSource
                (
                    pathSrcFile
                    , tCIDLib::ETextFmt_UNICode
                    , tCIDLib::EAccessModes
                      (
                        tCIDLib::EAccess_Read
                        | tCIDLib::EAccess_DenyWrite
                      )
                    , tCIDLib::ECreateAct_OpenIfExists
                    , tCIDLib::EFileAttr_None
                    , tCIDLib::EFileFlag_SequentialScan
                );

                //
                //  We need to first see if this file is in UNICode or ASCII
                //  text format. We read in the first character and see if it
                //  is the UNICode marker. If it is, we set the text stream
                //  input format accordingly. Otherwise, we reset the stream.
                //
                tCIDLib::Tch chMarker;
                chMarker = strmSource.chGet();

                if ((chMarker != kCIDLib::chMarker)
                &&  (chMarker != kCIDLib::chSwappedMarker))
                {
                    strmSource.eTextFormat(tCIDLib::ETextFmt_ASCII);
                    strmSource.Reset();
                }

                // The target stream is always ASCII only
                TTextFileStream strmTarget
                (
                    pathTargetFile
                    , tCIDLib::ETextFmt_ASCII
                    , tCIDLib::EAccess_Excl_ReadWrite
                    , tCIDLib::ECreateAct_ReplaceIfExists
                    , tCIDLib::EFileAttr_None
                    , tCIDLib::EFileFlag_SequentialScan
                );

                // Tell the translator about the output stream
                __ptransOut->SetOutputStream(strmTarget);

                // Build up the title string
                TString strPageTitle("Class and Member Info for Class: ");
                strPageTitle.Append(pathTmp);

                //
                //  Ask the translator to dump out the standard page header for
                //  this file file
                //
                __ptransOut->OutputPageHeader(strPageTitle);

                //
                //  Create a stream parser for this source file. This is what
                //  provides us the easy parsing of the source text. Tell him
                //  the name of the class also.
                //
                TDocParser prsrSource(strmSource, pathTmp);

                //
                //  Ok, call the method that does the parsing of the source
                //  and outputs the translated text.
                //
                try
                {
                    __TranslateSource(prsrSource, strmTarget);
                }

                catch(...)
                {
                    // Close the output file stream
                    strmTarget.Close();

                    // And delete the output file
                    TFileSys::DeleteFile(pathTargetFile);

                    // And let the error propogate
                    throw;
                }

                //
                //  Ok, we are just about done with this page. Ask the
                //  translator to output the standard page footer info.
                //
                __ptransOut->OutputPageFooter();
            }
        }
    }   while (diterSource.bFindNext(fndbCurrent));
}


//
// FUNCTION/METHOD NAME: __GenSubSystemPage
//
// DESCRIPTION:
//
//  This is called from the main code generation method. This one generates
//  a main page for a subsystem. It iterates the classes files in the
//  subsystem and creates links to each of the classes.
// -------------------------------------
//   INPUT: strSubName is the name of the subsystem that we are to do.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFacMakeDocs::__GenSubSystemPage(const TString& strSubName)
{
    //
    //  Create a directory iterator and iterate all of the ClassDoc files
    //  in the source directory.
    //
    TDirIter    diterSource;
    TFindBuf    fndbCurrent;

    TPathStr    pathToFind(__pathSourceDir);
    pathToFind.AddLevel(strSubName);
    pathToFind.AddLevel(L"*.ClassDoc");
    if (!diterSource.bFindFirst(pathToFind, fndbCurrent))
    {
        __conOut << facMakeDocs.strMsg(kDocMsgs::midNoDocsInSource, strSubName)
                 << NewLn;
        return;
    }

    //
    //  There are files in here, so lets open the output file and put out
    //  some preliminary text. Output is always in ASCII format.
    //
    TPathStr pathOut(__pathTargetDir);
    pathOut.AddLevel(strSubName);
    pathOut.AppendExtension(__ptransOut->strFileExtension());
    TTextFileStream strmTarget
    (
        pathOut
        , tCIDLib::ETextFmt_ASCII
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_ReplaceIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    // Point the output translator at our output file
    __ptransOut->SetOutputStream(strmTarget);

    //
    //  There should be a file in the main source directory that has
    //  documentation for the subsystem. We translate it out to the
    //  output file, then generate the links to all of the classes.
    //
    TPathStr pathClassDoc(__pathSourceDir);
    pathClassDoc.AddLevel(strSubName);
    pathClassDoc.AppendExtension(L".ClassDoc");
    TTextFileStream strmSource
    (
        pathClassDoc
        , tCIDLib::ETextFmt_UNICode
        , tCIDLib::EAccessModes
          (
            tCIDLib::EAccess_Read
            | tCIDLib::EAccess_DenyWrite
          )
        , tCIDLib::ECreateAct_OpenIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    //
    //  We need to first see if this file is in UNICode or ASCII
    //  text format. We read in the first character and see if it
    //  is the UNICode marker. If it is, we set the text stream
    //  input format accordingly. Otherwise, we reset the stream.
    //
    tCIDLib::Tch chMarker;
    chMarker = strmSource.chGet();

    if ((chMarker != kCIDLib::chMarker)
    &&  (chMarker != kCIDLib::chSwappedMarker))
    {
        strmSource.eTextFormat(tCIDLib::ETextFmt_ASCII);
        strmSource.Reset();
    }

    // Create a stream parser for the source stream
    TDocParser prsrSource(strmSource, strSubName);

    // Create the title string and then output the standard page header
    TString strTitle("Subsystem: ");
    strTitle.Append(strSubName);
    __ptransOut->OutputPageHeader(strTitle);

    //
    //  The whole file is just a free form text that we just pass to the
    //  translator to translate. There are no tokens, only formatting
    //  tags.
    //
    __ptransOut->TranslateTagText(prsrSource);

    //
    //  And now generate the heading for the class list that we are about
    //  to generate.
    //
    __ptransOut->OutputTag(TTranslator::ETag_Heading2);
    strmTarget << L"Classes in this facility";
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

    // Indent the list of links
    __ptransOut->OutputTag(TTranslator::ETag_Indent);

    do
    {
        //
        //  Strip the name we found down to just the base name. This will
        //  be the name of the class.
        //
        TPathStr pathClass(fndbCurrent.pathFileName());
        pathClass.bRemovePath();
        pathClass.bRemoveExt();

        //
        //  And generate a link to this guy. The file and link text is the
        //  same in this case.
        //
        __ptransOut->OutputLink(pathClass, pathClass);

        // And do a break between them
        __ptransOut->OutputTag(TTranslator::ETag_Break);

    }   while (diterSource.bFindNext(fndbCurrent));

    // Undo the indent we did above
    __ptransOut->OutputTag(TTranslator::ETag_Outdent);

    // And now output the standard file footer
    __ptransOut->OutputPageFooter();
}


//
// FUNCTION/METHOD NAME: __TranslateOverview
//
// DESCRIPTION:
//
//  This is called when a @Overview tag is seen. It starts a new output file
//  and formats the overview text to the new file. The new file is made up of
//  the class name and a title string, e.g. MyClass_ClassOverviewInfo, so that
//  its easy  to find the file from the class name.
//
//  The overview consists of all tokens and their text up to the @EndOverview
//  token or the end of the file. However, the overview is assumed to only
//  include basic formatted text info.
// -------------------------------------
//   INPUT: prsrSource is the source stream parser that we use to parse the
//              text.
//          strFileName is the name of the file we should output this
//              method to. its just the raw name with no path or ext.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFacMakeDocs::__TranslateOverview(          TDocParser& prsrSource
                                    , const TString&    strFileName)
{
    static TString  strToken1(kCIDLib::pszEmptyZStr, 1024);

    // Build up the full path to the output file
    TPathStr pathTarget(__pathTargetDir);
    pathTarget.AddLevel(strFileName);
    pathTarget.AppendExtension(__ptransOut->strFileExtension());

    //
    //  Create the output stream for this method. All output files are in
    //  ASCII format.
    //
    TTextFileStream strmTarget
    (
        pathTarget
        , tCIDLib::ETextFmt_ASCII
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_ReplaceIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    // Install our stream in the translator object
    __ptransOut->SetOutputStream(strmTarget);

    // Output the standard file header
    __ptransOut->OutputTag(TTranslator::ETag_Paragraph);
    TString strTitle(L"Class Overview: ");
    strTitle.Append(prsrSource.strClassName());
    __ptransOut->OutputPageHeader(strTitle);

    // Output a little information
    __ptransOut->OutputTag(TTranslator::ETag_Heading1);
    strmTarget  << L"Overview of: " << prsrSource.strClassName();
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading1);

    //
    //  Now handle all of the free form text. It will stop when it hits a
    //  tag. All of the overview information is just free form text
    //  stuff, terminated by an @EndOverview tag.
    //
    __ptransOut->TranslateTagText(prsrSource);

    // Get the next token
    TDocParser::ETokens eToken = prsrSource.eNextToken();

    if (eToken != TDocParser::EToken_EndOverview)
    {
        throw TDocError
        (
            kDocErrs::errcExpected
            , prsrSource.c4CurLine()
            , prsrSource.strTokenText(TDocParser::EToken_EndOverview)
        );
    }
}


//
// FUNCTION/METHOD NAME: __TranslateMethod
//
// DESCRIPTION:
//
//  This is called when a @Method tag is seen. It starts a new output file
//  and formats the method to the new file. The new file is made up of the
//  class name and the method name, e.g. MyClass_MyMethod so that its easy
//  to find the file from the class and method name.
//
//  The method consists of all tokens and their text up to the @EndMethod
//  token or the end of the file.
// -------------------------------------
//   INPUT: prsrSource is the source stream parser that we use to parse the
//              text.
//          strMethodName is the name of the method we are doing.
//          strFileName is the name of the file we should output this
//              method to. its just the raw name with no path or ext.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFacMakeDocs::__TranslateMethod(        TDocParser& prsrSource
                                , const TString&    strMethodName
                                , const TString&    strFileName)
{
    static TString  strToken1(kCIDLib::pszEmptyZStr, 1024);

    // Build up the full path to the output file
    TPathStr pathTarget(__pathTargetDir);
    pathTarget.AddLevel(strFileName);
    pathTarget.AppendExtension(__ptransOut->strFileExtension());

    //
    //  Create the output stream for this method. All output files are
    //  in ASCII format.
    //
    TTextFileStream strmTarget
    (
        pathTarget
        , tCIDLib::ETextFmt_ASCII
        , tCIDLib::EAccess_Excl_ReadWrite
        , tCIDLib::ECreateAct_ReplaceIfExists
        , tCIDLib::EFileAttr_None
        , tCIDLib::EFileFlag_SequentialScan
    );

    // Install our stream in the translator object
    __ptransOut->SetOutputStream(strmTarget);

    // Output the standard file header
    TString strTitle(L"Class: ");
    strTitle.Append(prsrSource.strClassName());
    strTitle.Append(L", Method: ");
    strTitle.Append(strMethodName);
    __ptransOut->OutputPageHeader(strTitle);

    // The heading is in the first heading style
    __ptransOut->OutputTag(TTranslator::ETag_Heading1);
    strmTarget  << L"Method Name: " << strMethodName;
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading1);

    // Now handle any free form text of the general description
    __ptransOut->TranslateTagText(prsrSource);

    //
    //  Now enter the loop where we deal with all of the tokens that
    //  are for this method. We end when we hit the end of the file or
    //  the method end token.
    //
    tCIDLib::TBoolean bInParams = kCIDLib::False;
    tCIDLib::TBoolean bInExceptions = kCIDLib::False;
    while (1)
    {
        // Get the next token
        TDocParser::ETokens eToken = prsrSource.eNextToken();

        //
        //  If we are in a params block, then only a param or end params
        //  token is valid.
        //
        if (bInParams
        &&  ((eToken != TDocParser::EToken_Param)
        &&   (eToken != TDocParser::EToken_EndParams)))
        {
            throw TDocError
            (
                kDocErrs::errcExpected2
                , prsrSource.c4CurLine()
                , prsrSource.strTokenText(TDocParser::EToken_Param)
                , prsrSource.strTokenText(TDocParser::EToken_EndParams)
            );
        }

        //
        //  If we are in a exceptions block, then only an exception or end
        //  exceptions token is valid.
        //
        if (bInExceptions
        &&  ((eToken != TDocParser::EToken_Exception)
        &&   (eToken != TDocParser::EToken_EndExceptions)))
        {
            throw TDocError
            (
                kDocErrs::errcExpected2
                , prsrSource.c4CurLine()
                , prsrSource.strTokenText(TDocParser::EToken_Exception)
                , prsrSource.strTokenText(TDocParser::EToken_EndExceptions)
            );
        }

        if (eToken == TDocParser::EToken_Caveats)
        {
            __ptransOut->OutputTag(TTranslator::ETag_Heading2);
            strmTarget << eToken;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);
            __ptransOut->TranslateTagText(prsrSource);
        }
         else if ((eToken == TDocParser::EToken_Params)
              ||  (eToken == TDocParser::EToken_Exceptions))
        {
            if (bInParams || bInExceptions)
            {
                throw TDocError
                (
                    kDocErrs::errcAlreadyInBlock
                    , prsrSource.c4CurLine()
                    , prsrSource.strTokenText(eToken)
                );
            }

            if (eToken == TDocParser::EToken_Params)
                bInParams = kCIDLib::True;
            else
                bInExceptions = kCIDLib::True;

            __ptransOut->OutputTag(TTranslator::ETag_Heading2);
            strmTarget << eToken;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

            // Indent all the params
            __ptransOut->OutputTag(TTranslator::ETag_Indent);

            // Output the start of a table for the params
            __ptransOut->OutputTag(TTranslator::ETag_Table);
        }
         else if ((eToken == TDocParser::EToken_EndParams)
              ||  (eToken == TDocParser::EToken_EndExceptions))
        {
            if (eToken == TDocParser::EToken_EndParams)
            {
                if (!bInParams)
                {
                    throw TDocError
                    (
                        kDocErrs::errcNotInBlock
                        , prsrSource.c4CurLine()
                        , prsrSource.strTokenText(eToken)
                    );
                }
                bInParams = kCIDLib::False;
            }
            else
            {
                if (!bInExceptions)
                {
                    throw TDocError
                    (
                        kDocErrs::errcNotInBlock
                        , prsrSource.c4CurLine()
                        , prsrSource.strTokenText(eToken)
                    );
                }
                bInExceptions = kCIDLib::False;
            }

            __ptransOut->OutputTag(TTranslator::ETag_EndTable);

            // And outdent now that we are done with parms
            __ptransOut->OutputTag(TTranslator::ETag_Outdent);
        }
         else if ((eToken == TDocParser::EToken_Param)
              ||  (eToken == TDocParser::EToken_Exception))
        {
            //
            //  In this one, the first token is the parameter name. So we
            //  bold it. Then the rest is free form text.
            //
            prsrSource.GetNextToken(L" ", strToken1);
            __ptransOut->OutputTag(TTranslator::ETag_TableCol1);
            __ptransOut->OutputTag(TTranslator::ETag_Bold);
            strmTarget << strToken1;
            __ptransOut->OutputTag(TTranslator::ETag_EndBold);
            __ptransOut->OutputTag(TTranslator::ETag_EndTableCol1);

            __ptransOut->OutputTag(TTranslator::ETag_TableCol2);
            __ptransOut->TranslateTagText(prsrSource);
            __ptransOut->OutputTag(TTranslator::ETag_EndTableCol2);
        }
         else if (eToken == TDocParser::EToken_Return)
        {
            __ptransOut->OutputTag(TTranslator::ETag_Heading2);
            strmTarget << eToken;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

            __ptransOut->OutputTag(TTranslator::ETag_Indent);
            __ptransOut->TranslateTagText(prsrSource);
            __ptransOut->OutputTag(TTranslator::ETag_Outdent);
        }
         else if (eToken == TDocParser::EToken_End)
        {
            break;
        }
         else if (eToken == TDocParser::EToken_EndMethod)
        {
            break;
        }
         else if (eToken == TDocParser::EToken_None)
        {
            throw TDocError
            (
                kDocErrs::errcExpectedToken
                , prsrSource.c4CurLine()
            );
        }
         else
        {
            throw TDocError
            (
                kDocErrs::errcTokenBadHere
                , prsrSource.c4CurLine()
                , prsrSource.strTokenText(eToken)
            );
        }
    }

    // And output the standard footer
    __ptransOut->OutputPageFooter();
}


//
// FUNCTION/METHOD NAME: __TranslateSource
//
// DESCRIPTION:
//
//  This method does the actual parsing of a source file and outputs the
//  text that it translates to.
// -------------------------------------
//   INPUT: prsrSource is the source stream parser that we use to parse the
//              text.
//          strmTarget is the target text stream to output to.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid
TFacMakeDocs::__TranslateSource( TDocParser&    prsrSource
                                , TTextStream&  strmTarget)
{
    // Output a little header
    __ptransOut->OutputTag(TTranslator::ETag_Heading1);
    strmTarget << L"General Information";
    __ptransOut->OutputTag(TTranslator::ETag_EndHeading1);

    static TString strToken1(kCIDLib::pszEmptyZStr, 1024);

    while (1)
    {
        // Get the next token
        TDocParser::ETokens eToken = prsrSource.eNextToken();

        //
        //  We just go by what the token is and parse as we expect the
        //  text for that kind of token.
        //
        if ((eToken == TDocParser::EToken_Class)
        ||  (eToken == TDocParser::EToken_ParentClass)
        ||  (eToken == TDocParser::EToken_Facility)
        ||  (eToken == TDocParser::EToken_Mixins)
        ||  (eToken == TDocParser::EToken_Prefix)
        ||  (eToken == TDocParser::EToken_Header))
        {
            // All of these are just one liners
            prsrSource.GetLineRemainder(strToken1);
            strToken1.StripWhitespace();
            __ptransOut->OutputTag(TTranslator::ETag_Bold);
            strmTarget << eToken << L": ";
            __ptransOut->OutputTag(TTranslator::ETag_EndBold);

            //
            //  If this is the parent class and its not N/A, then out
            //  put it as a link. Otherwise just plain text.
            //
            if ((eToken == TDocParser::EToken_ParentClass)
            &&  (strToken1 != L"N/A"))
                __ptransOut->OutputLink(strToken1, strToken1);
             else
                strmTarget << strToken1;

            //
            //  If this is the class name, then make sure that it agrees
            //  with the name of the file.
            //
            if (eToken == TDocParser::EToken_Class)
            {
                if (!prsrSource.strClassName().bSameText(strToken1))
                {
                    throw TDocError
                    (
                        kDocErrs::errcBadClassName
                        , prsrSource.c4CurLine()
                        , prsrSource.strClassName()
                        , strToken1
                    );
                }
            }

            __ptransOut->OutputTag(TTranslator::ETag_Break);
        }
        else if (eToken == TDocParser::EToken_Caveats)
        {
            __ptransOut->OutputTag(TTranslator::ETag_Heading3);
            strmTarget << eToken;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading3);
            __ptransOut->TranslateTagText(prsrSource);
        }
         else if (eToken == TDocParser::EToken_Description)
        {
            __ptransOut->OutputTag(TTranslator::ETag_Heading3);
            strmTarget << eToken;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading3);
            __ptransOut->TranslateTagText(prsrSource);
        }
         else if (eToken == TDocParser::EToken_Group)
        {
            //
            //  These all have a heading text on one line, followed by
            //  optional free form text.
            //
            prsrSource.GetLineRemainder(strToken1);
            strToken1.StripWhitespace();

            // The heading is in the second heading style
            __ptransOut->OutputTag(TTranslator::ETag_Heading2);
            strmTarget << eToken << L": ";
            strmTarget << strToken1;
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

            // Now handle any free form text
            __ptransOut->TranslateTagText(prsrSource);

            // And output a break
            __ptransOut->OutputTag(TTranslator::ETag_Paragraph);
        }
         else if (eToken == TDocParser::EToken_Method)
        {
            // The rest of the line is the method name.
            prsrSource.GetLineRemainder(strToken1);
            strToken1.StripWhitespace();

            if (strToken1.bEmpty())
            {
                throw TDocError
                (
                    kDocErrs::errcNoMethodName
                    , prsrSource.c4CurLine()
                );
            }

            // Build up the base name of the new file for this method
            TPathStr pathLink(prsrSource.strClassName());
            pathLink.Append(L'_');
            pathLink.Append(strToken1);

            //
            //  We need to handle spaces in the name, which the NFS file
            //  systems that these files will be one wouldn't be happy with.
            //
            pathLink.ReplaceWith(kCIDLib::chSpace, kCIDLib::chUnderscore);

            // Output a link to this new file
            __ptransOut->OutputLink(strToken1, pathLink);
            __ptransOut->OutputTag(TTranslator::ETag_Break);

            // Now translate the method into this new file
            __TranslateMethod(prsrSource, strToken1, pathLink);

            // Put our stream back into the translator
            __ptransOut->SetOutputStream(strmTarget);
        }
         else if (eToken == TDocParser::EToken_Overview)
        {
            // Build up the base name of the new file for this method
            TPathStr pathLink(prsrSource.strClassName());
            pathLink.Append(L"_ClassOverviewInfo");

            // Output a link to this new file
            __ptransOut->OutputTag(TTranslator::ETag_Paragraph);
            __ptransOut->OutputLink(L"Class Overview", pathLink);
            __ptransOut->OutputTag(TTranslator::ETag_Break);

            // Now translate the overview info into this new file.
            __TranslateOverview(prsrSource, pathLink);

            // Put our stream back into the translator
            __ptransOut->SetOutputStream(strmTarget);
        }
         else if (eToken == TDocParser::EToken_Related)
        {
            //
            //  Each line after this tag indicates a link to a related
            //  class.
            //
            __ptransOut->OutputTag(TTranslator::ETag_Heading2);
            strmTarget << L"Related Topics";
            __ptransOut->OutputTag(TTranslator::ETag_EndHeading2);

            while (1)
            {
                prsrSource.GetNextToken(L" \r\n", strToken1);
                if (strToken1.bEmpty())
                    break;

                if (strToken1 == L"//")
                {
                    prsrSource.FlushLine();
                    continue;
                }

                __ptransOut->OutputLink(strToken1, strToken1);
                __ptransOut->OutputTag(TTranslator::ETag_Break);
            }
        }
         else if (eToken == TDocParser::EToken_End)
        {
            break;
        }
         else if (eToken == TDocParser::EToken_None)
        {
            throw TDocError
            (
                kDocErrs::errcExpectedToken
                , prsrSource.c4CurLine()
            );
        }
         else
        {
            throw TDocError
            (
                kDocErrs::errcTokenBadHere
                , prsrSource.c4CurLine()
                , prsrSource.strTokenText(eToken)
            );
        }
    }
}
