//
// NAME: MakeDocs_Parser.Cpp
//
// DESCRIPTION: 
//
//  This module implements the parser class that is used to parse the source
//  text files. It is built on the standard text stream parser, and just adds
//  some extra code to look for our particular synactical elements.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//

// -----------------------------------------------------------------------------
//  Include underlying headers.
// -----------------------------------------------------------------------------
#include    "MakeDocs.Hpp"


// ----------------------------------------------------------------------------
//  Local constant values
// ----------------------------------------------------------------------------
const tCIDLib::TCard4 __c4HashModulus = 17;


// ----------------------------------------------------------------------------
//  Local types
//
//  TTokenItem
//      This small structure represents a token that we look for. The tokens
//      are hashed for speed.
// ----------------------------------------------------------------------------
struct TTokenItem
{
    TDocParser::ETokens         eToken;
    tCIDLib::Tch*               pszToken;
    tCIDLib::THashVal           hshToken;
};


// ----------------------------------------------------------------------------
//  Local static data
//
//  __aszTokens
//      These are our syntax tokens. We tell the parser about them and it will
//      treat them as atomic tokens even when there is no whitespace around
//      them.
// ----------------------------------------------------------------------------
const   tCIDLib::Tch* __aszTokens[] =
{
    // This token starts a comment
    L"//"
    , L">"
};
const tCIDLib::TCard4  __c4SyntaxCount =   c4ArrayElems(__aszTokens);

TTokenItem __atokList[] =
{
        { TDocParser::EToken_Caveats        , L"@Caveats"       , 0 }
    ,   { TDocParser::EToken_Class          , L"@Class"         , 0 }
    ,   { TDocParser::EToken_Description    , L"@Description"   , 0 }
    ,   { TDocParser::EToken_EndExceptions  , L"@EndExceptions" , 0 }
    ,   { TDocParser::EToken_EndMethod      , L"@EndMethod"     , 0 }
    ,   { TDocParser::EToken_EndOverview    , L"@EndOverview"   , 0 }
    ,   { TDocParser::EToken_EndParams      , L"@EndParams"     , 0 }
    ,   { TDocParser::EToken_Exception      , L"@Exception"     , 0 }
    ,   { TDocParser::EToken_Exceptions     , L"@Exceptions"    , 0 }
    ,   { TDocParser::EToken_Facility       , L"@Facility"      , 0 }
    ,   { TDocParser::EToken_Group          , L"@Group"         , 0 }
    ,   { TDocParser::EToken_Header         , L"@Header"        , 0 }
    ,   { TDocParser::EToken_Method         , L"@Method"        , 0 }
    ,   { TDocParser::EToken_Mixins         , L"@Mixins"        , 0 }
    ,   { TDocParser::EToken_Overview       , L"@Overview"      , 0 }
    ,   { TDocParser::EToken_Param          , L"@Param"         , 0 }
    ,   { TDocParser::EToken_Params         , L"@Params"        , 0 }
    ,   { TDocParser::EToken_ParentClass    , L"@Parent"        , 0 }
    ,   { TDocParser::EToken_Prefix         , L"@Prefix"        , 0 }
    ,   { TDocParser::EToken_Related        , L"@Related"       , 0 }
    ,   { TDocParser::EToken_Return         , L"@Return"        , 0 }
    ,   { TDocParser::EToken_Comment        , L"//"             , 0 }
};
const tCIDLib::TCard4 __c4TokenCount = c4ArrayElems(__atokList);


// ----------------------------------------------------------------------------
//  CLASS: TDocParser
// PREFIX: prsr
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TDocParser: Global operators
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: operator<<
//
// DESCRIPTION:
//
//  This method will format the passed token to the passed stream.
// -------------------------------------
//   INPUT: strmOut is the output stream to format to
//          eToken is the token to format.
//
//  OUTPUT: None
//
//  RETURN: A reference to the output stream.
//
TTextStream& operator<<(TTextStream& strmOut, const TDocParser::ETokens eToken)
{
    if (eToken == TDocParser::EToken_Caveats)
        strmOut << L"Caveats";
    else if (eToken == TDocParser::EToken_Class)
        strmOut << L"Class";
    else if (eToken == TDocParser::EToken_Description)
        strmOut << L"Description";
    else if (eToken == TDocParser::EToken_EndExceptions)
        strmOut << L"End Exceptions";
    else if (eToken == TDocParser::EToken_EndMethod)
        strmOut << L"End Method";
    else if (eToken == TDocParser::EToken_EndOverview)
        strmOut << L"End Overview";
    else if (eToken == TDocParser::EToken_EndParams)
        strmOut << L"End Params";
    else if (eToken == TDocParser::EToken_Exception)
        strmOut << L"Exception";
    else if (eToken == TDocParser::EToken_Exceptions)
        strmOut << L"Exceptions";
    else if (eToken == TDocParser::EToken_Facility)
        strmOut << L"Facility";
    else if (eToken == TDocParser::EToken_Group)
        strmOut << L"Group";
    else if (eToken == TDocParser::EToken_Header)
        strmOut << L"Header";
    else if (eToken == TDocParser::EToken_Method)
        strmOut << L"Method";
    else if (eToken == TDocParser::EToken_Mixins)
        strmOut << L"Mixin Classes";
    else if (eToken == TDocParser::EToken_Overview)
        strmOut << L"Overview";
    else if (eToken == TDocParser::EToken_Param)
        strmOut << L"Parameter";
    else if (eToken == TDocParser::EToken_Params)
        strmOut << L"Parameters";
    else if (eToken == TDocParser::EToken_ParentClass)
        strmOut << L"Parent Class";
    else if (eToken == TDocParser::EToken_Prefix)
        strmOut << L"Class Prefix";
    else if (eToken == TDocParser::EToken_Related)
        strmOut << L"Related";
    else if (eToken == TDocParser::EToken_Return)
        strmOut << L"Return";
    else if (eToken == TDocParser::EToken_Comment)
        strmOut << L"//";
    else if (eToken == TDocParser::EToken_None)
        strmOut << L"None";
    else if (eToken == TDocParser::EToken_End)
        strmOut << L"End";
    else
    {
        facMakeDocs.LogErr
        (
            __FILE__
            , __LINE__
            , kDocErrs::errcPrsrBadToken
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_Internal
            , TInteger(eToken)
        );
    }

    return strmOut;
}

// ----------------------------------------------------------------------------
//  TDocParser: Constructors and destructors
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TDocParser
//
// DESCRIPTION:
//
// -------------------------------------
//   INPUT: strmSource is the source stream to parse from. It is not adopted
//              in this case.
//          strClassName is the name of the class whose source we are parsing.
//
//  OUTPUT: None
//
//  RETURN: None
//
TDocParser::TDocParser(TTextStream& strmSource, const TString& strClassName) :

    TStreamParser(&strmSource, tCIDLib::EAdoptOpt_NoAdopt)
    , __strClassName(strClassName)
{
    static tCIDLib::TBoolean bFirstTime = kCIDLib::True;
    if (bFirstTime)
    {
        bFirstTime = kCIDLib::False;

        // Do the hashes for all of the tokens.
        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4TokenCount; c4Ind++)
        {
            __atokList[c4Ind].hshToken = TRawStr::hshHashStr
            (
                __atokList[c4Ind].pszToken
                , __c4HashModulus
            );
        }
    }

    // Set up our token list
    for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4SyntaxCount; c4Ind++)
        AddSyntaxToken(__aszTokens[c4Ind]);
}

TDocParser::~TDocParser()
{
}


// ----------------------------------------------------------------------------
//  TDocParser: Public, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: eNextToken
//
// DESCRIPTION:
//
//  This method will look for a valid token in the text at the current
//  position.
// -------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: The token enum that represents the token found.
//
TDocParser::ETokens TDocParser::eNextToken()
{
    //
    //  We do this in a loop because we throw away any comment lines.
    //  We use a local string to read tokesn into.
    //
    TString strToken(kCIDLib::pszEmptyZStr, 128, 128);
    while (1)
    {
        // Get the next token, using whitespace as separators
        GetNextToken(kCIDLib::szWhitespace, strToken);

        //
        //  If the token is empty, then we are done so we return End
        //
        if (strToken.bEmpty())
            return TDocParser::EToken_End;

        //
        //  See if its one of our tags. If its the end, then just return
        //  that.
        //
        TDocParser::ETokens eToken = __eMapToken(strToken);

        if (eToken == TDocParser::EToken_End)
            return eToken;

        if (eToken == TDocParser::EToken_None)
        {
            UnGetToken(strToken);
            return eToken;
        }

        //
        //  If its a comment, then blow off the rest of the line and
        //  keep going.
        //
        if (eToken == TDocParser::EToken_Comment)
        {
            FlushLine();
            continue;
        }

        // Return the token
        return eToken;
    }
}


//
// FUNCTION/METHOD NAME: strTokenText
//
// DESCRIPTION:
//
//  This method will return the text for the given token.
// -------------------------------------
//   INPUT: eToken is the token to get the text for.
//
//  OUTPUT: None
//
//  RETURN: A by value string object with the text
//
TString TDocParser::strTokenText(const ETokens eToken) const
{
    // Check for special ones first
    if (eToken == EToken_None)
        return TString(L"None");
    else if (eToken == EToken_End)
        return TString(L"End");
    else if (eToken == EToken_Comment)
        return TString(L"//");

    // Else we just return the text from the token list
    return TString(__atokList[eToken].pszToken);
}


// ----------------------------------------------------------------------------
//  TDocParser: Private, non-virtual methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMapToken
//
// DESCRIPTION:
//
//  This method checks the passed text against our token list and maps
//  the text to a token enum.
// -------------------------------------
//   INPUT: strText is the text to map to a token enum
//
//  OUTPUT: None
//
//  RETURN: The token enum that represents the token found.
//
TDocParser::ETokens
TDocParser::__eMapToken(const TString& strText) const
{
    // Get the hash of the passed token text
    tCIDLib::THashVal hshToken = strText.hshCalcHash(__c4HashModulus);

    //
    //  Now use that to check each token for a possible match. If the
    //  hash matches, then do the actuall comparision.
    //
    for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4TokenCount; c4Index++)
    {
        if (__atokList[c4Index].hshToken == hshToken)
        {
            if (strText == __atokList[c4Index].pszToken)
                return __atokList[c4Index].eToken;
        }
    }
    return TDocParser::EToken_None;
}
