//
// NAME: MakeDocs.Cpp
//
// DESCRIPTION: 
//
//  This is the main module of the program. This program serves both as a
//  nice demonstration of a lot of the CIDLib system's non-GUI functionality
//  and as an important utility for the generation of the CIDLib HTML
//  documentation. All of the class and member documentation for CIDLib is
//  in the form of .ClassDoc files. These are text files with a canned format
//  that this program understands.
//
//  This ClassDoc files are parsed by this program, and it turns around and
//  outputs the text in some other format, such as HTML. Much of the output
//  is canned by this program, so it handles formatting for those parts.
//  Examples are the standard header panels, and headings in the standard
//  pages such as section titles, class names, member names, etc... It applies
//  standard formatting to these parts of the template, though it does it via
//  a plug in interface so that its done generically (see below.)
//
//  Besides the special headings and such that are generated by this program,
//  the .ClassDoc files use simple, HTML-like, formatting codes to control
//  the format of the text that is not part of the template text, e.g. to
//  indicate bold, source code, paragraph breaks, etc... This program uses
//  a plug in object interface to handle translation of these codes to the
//  required output language. This same interface is used when the canned
//  parts of the template are output also, so providing another derivative
//  and plugging it in could be used to quickly output the text as IPF text
//  or SGML or whatever. Since the format is so simple, it could be output
//  to pretty much anything.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 06/10/97
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  This program is an ad hoc program that suits exactly the needs of
//      the CIDLib documentation. You might be able to use it for other
//      things, but its not intended to be general purpose or handle any
//      extenuating circumstances.
//

// -----------------------------------------------------------------------------
//  Include underlying headers.
// -----------------------------------------------------------------------------
#include    "MakeDocs.Hpp"


// -----------------------------------------------------------------------------
//  Do RTTI macros for classes that have no Cpp files of their own
// -----------------------------------------------------------------------------
RTTIData(TDocError,TObject)



// -----------------------------------------------------------------------------
//  Local function prototypes
//
//  __eMainThead
//      This is the thread function for the main thread that provides the
//      execution for this program.
// -----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// -----------------------------------------------------------------------------
//  Glocal data declarations
//
//  facMakeDocs
//      This is our facility object. This program is complex enough that its
//      advantageous to have a facility class of its own. This allows for
//      easy coordination of any global data.
// -----------------------------------------------------------------------------
TFacMakeDocs    facMakeDocs;


// -----------------------------------------------------------------------------
//  Include magic main module code
// -----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"MainThread", __eMainThreadFunc))



//
// FUNCTION/METHOD NAME: __excMainThread
//
// DESCRIPTION:
//
//  This is the main program thread. 
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator. We don't use it here.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    // Display our little opening blurb.
    facMakeDocs.ShowBlurb();

    //
    //  Now lets work on the command line parms. This is done by the
    //  facility object, which stores away all of the information if it
    //  goes well.
    //
    if (!facMakeDocs.bParseCommandLine())
        return tCIDLib::EExit_BadParameters;

    //
    //  Ok, kick off the work. We handle any exceptions here by just dumping
    //  out the info
    //
    try
    {
        facMakeDocs.GenerateDocs();
    }

    catch(const TError& errToCatch)
    {
        facMakeDocs.strmOut()
            << facMakeDocs.strMsg(kDocMsgs::midError) << DNewLn
            << facMakeDocs.strMsg(kDocMsgs::midInfo) << NewLn
            << L"---------------------" << NewLn
            << errToCatch
            << DNewLn;
        return tCIDLib::EExit_FatalError;

    }

    // Watch for parser errors
    catch(const TDocError& errToCatch)
    {
        facMakeDocs.strmOut()
            << NewLn << facMakeDocs.strMsg(kDocMsgs::midParseError) << NewLn
            << facMakeDocs.strMsg(kDocMsgs::midLineNum)
            << errToCatch.c4Line() << NewLn
            << facMakeDocs.strMsg(kDocMsgs::midErrorPrefix)
            << errToCatch.strText()
            << DNewLn;
        return tCIDLib::EExit_FatalError;

    }

    //
    //  This is partly a test program so, just in case, catch any kernel
    //  errors so we can know when any are leaking through (they are not
    //  supposed to be seen by end user code!)
    //
    catch(const TKrnlError& kerrToCatch)
    {
        facMakeDocs.strmOut()
            << facMakeDocs.strMsg(kDocMsgs::midKrnlError) << DNewLn
            << facMakeDocs.strMsg(kDocMsgs::midErrorPrefix)
            << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;

    }

    // And watch for system errors
    catch(...)
    {
        facMakeDocs.strmOut()
            << facMakeDocs.strMsg(kDocMsgs::midSysError) << DNewLn;
        return tCIDLib::EExit_SystemException;

    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
