//
// NAME: Fractal1.Cpp
//
// DESCRIPTION:
//
//  This is a small demo program which demonstrates the use of the fractal
//  engine, for simple single threaded programs. It just does a canned
//  rendering of a Mandelbrot fractal in full view. If you want to get a
//  differnt fractal or coordinate space, change it and recompile.
//
//  The output is a Dis format file, which the little DispRGB program
//  understands how to view.
//
//  Since all rendering is done in the background, we have to let the
//  rendering thread complete before we try to write the data. So we use
//  a standard TStatusControllerTTY object, which was designed basically
//  for this type of app. We give the calc engine the object and he uses
//  it to keep us informed. We just ask the controller to block us until
//  the work is complete.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 09/20/96
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//
// MODIFICATION LOG:
//


// -----------------------------------------------------------------------------
//  CIDLib includes
// -----------------------------------------------------------------------------
#include    "Fractal1.Hpp"


// -----------------------------------------------------------------------------
//  Local function prototypes
//
//  __excMainThead
//      This is the thread function for the main thread. This thread is
//      started magically and provides the entry point for the application
//      code.
// -----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// -----------------------------------------------------------------------------
//  Local data declarations
//
//  __conOut
//      This is our console object for user interaction. TConsole is a
//      specialized text stream class that provides command recall and
//      editing (i.e. it allows you to turn a program into a command shell
//      type of interface easily.)
//
//  __strTitle
//      This is the program title. It has replacement parms that we will
//      replace using version info from the build environment. The second
//      token says format to a 3 char field, right justified, with '0' for
//      fill. It will have an initial allocation of 64 chars, which should be
//      enough for the replacement parametes to fit (so it won't have to be
//      reallocated.)
// -----------------------------------------------------------------------------
static TConsole     __conOut;
static TString      __strTitle
                    (
                        L"Fractal Demo 1, Version %(1).%(2,3,0), 'CIDCorp"
                        , 64
                    );


// -----------------------------------------------------------------------------
//  Include magic main module code
// -----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"MainThread", __eMainThreadFunc))


//
// FUNCTION/METHOD NAME: __eMainThread
//
// DESCRIPTION:
//
//  This is the main program thread. It just displays its info and exits.
// -------------------------------------
//   INPUT: thrThis is a reference to the thread object for this thread
//          pData is a pointer to the optional data buffer passed by our
//              creator.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid* const)
{
    // Let our caller go
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Do the replacement of the version token in the title. The major and
    //  minor versions are the standard CIDLib release values. This is just
    //  a convention used by all of CIDLib code to easily mark them all
    //  according to version.
    //
    __strTitle.ReplaceToken(kCIDLib::c4MajVersion, L'1');
    __strTitle.ReplaceToken(kCIDLib::c4MinVersion, L'2');

    //
    //  These are the parameters we use for the fractal. Change these and
    //  recompile to get different outputs.
    //
    const tCIDLib::TCard4   c4Height        = 480;
    const tCIDLib::TCard4   c4MaxIters      = 1024;
    const tCIDLib::TCard4   c4Width         = 480;
    TPathStr                strFileName(L"OutPut");

    // Create the fractal we want to do
    TMandelbrot* pfracToRender = new TMandelbrot;

    //
    //  Ask the fractal what its default fractal coordinates are. We just
    //  use the default coordinates of the fractal that we are to render.
    //  This is usually the 'big picture' view from which zooms can be done
    //  into the image.
    //
    TFArea  fareaFractalSpace(pfracToRender->fareaDefSpace());

    // Set its max iterations
    pfracToRender->c4MaxIterations(c4MaxIters);

    //
    //  Create the status controller that we will give to the fractal
    //  engine to allow it to interact with us. We create a TTY type
    //  controller, which just outputs status info to an output stream.
    //  We give it the console output stream to output to.
    //
    TStatusControllerTTY* pstatcToUse = new TStatusControllerTTY(&__conOut);

    //
    //  Create a single threaded calculation engine. We tell it the pixel
    //  size of the image, the fractal space we want to overlay on that
    //  pixel grid, a pointer to a dyanmically allocated fractal object
    //  that he will use to do the calculations, and the status controller
    //  object he will use to communicate with. He adopts the fractal
    //  object and status controller, so we are not responsible for them
    //  after this.
    //
    TFracSingleEngine fengWork
    (
        TSize(c4Width, c4Height)
        , fareaFractalSpace
        , pfracToRender
        , pstatcToUse
    );

    //
    //  Tell the engine to start working. This will not come back until
    //  its all done.
    //
    fengWork.StartProcessing();

    //
    //  Now we just block on the status controller until the work is
    //  is complete. The background thread will, via the TTY controller,
    //  output its status to our standard out.
    //
    pstatcToUse->WaitForComplete(kCIDLib::c4MaxWait);

    //
    //  Ok, the image is done, so lets save it out as a simple Dis dump
    //  file. Create a binary file stream.
    //
    strFileName.Append(L".Dis");

    // Create a stream for the file
    TBinaryFileStream strmTarget;
    try
    {
        strmTarget.Open
        (
            strFileName
            , tCIDLib::EAccess_Excl_Write
            , tCIDLib::ECreateAct_ReplaceIfExists
            , tCIDLib::EFileAttr_None
            , tCIDLib::EFileFlag_SequentialScan
        );
    }

    catch(const TError& errToCatch)
    {
        __conOut    << L"Error occured while opening file "
                    << strFileName << DNewLn
                    << L"Err: " << errToCatch.strErrText()
                    << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    //
    //  First thing written is the image size. We have to force them to
    //  be 2 byte cardinal values here!
    //
    strmTarget << tCIDLib::TCard2(c4Width);
    strmTarget << tCIDLib::TCard2(c4Height);

    // Create a palette object to map to colors
    TClrPalette palImage(tCIDLib::EDefPalette_256Default);

    //
    //  Create a log map to map the big iteration values down to the
    //  limited palette indexes.
    //
    TLogMap vmapColors(255, c4MaxIters-1);

    __conOut << L"Writing out file..." << NewLn;

    //
    //  We create one buffer as a line buffer. The Dis format we are
    //  writing has separate sections for each line's RGB values. So
    //  we have an array of bytes with 3 bytes per pixel in a row. We
    //  tell it not to store its line count when streamed out, since we
    //  are writing out to a particular format not just streaming out
    //  a CIDLib object.
    //
    //  We create an array of TCard4 values in order to pull each row
    //  out of the cache.
    //
    TBaseArray<tCIDLib::TCard1> baLineBuf(c4Width*3);
    baLineBuf.bStoreCount(kCIDLib::False);
    TBaseArray<tCIDLib::TCard4> baRowData(c4Width);

    //
    //  Run through all of the rows in the image and query the escape
    //  time info into an array of TCard4 values. We use an instantiation
    //  of the TBaseArray template class. It comes with instantations
    //  for all of the fundamental types. , for your convenience.
    //
    TRGBClr rgbPel;
    for (tCIDLib::TCard4 c4RowInd = 0; c4RowInd < c4Height; c4RowInd++)
    {
        // Write out the row number as a Card2 value
        strmTarget << tCIDLib::TCard2(c4RowInd);

        // Get this line's data from the cache
        fengWork.QueryCacheLine(c4RowInd, baRowData);

        //
        //  Loop through this row, map each value to a color, and write
        //  out the color value to the file.
        //
        for (tCIDLib::TCard4 c4Pel = 0; c4Pel < c4Width; c4Pel++)
        {
            //
            //  Map this pel to a color from the palette. We index the
            //  row data array to get the escape value, which is fed into
            //  to the log map to get a color index, which is fed into the
            //  palette to get the color.
            //
            const TRGBClr& rgbPel
                        = palImage[vmapColors.c4MapIndex(baRowData[c4Pel])];

            // Put each color component into the line buffer
            baLineBuf[c4Pel] = rgbPel.c1Red();
            baLineBuf[c4Pel + c4Width] = rgbPel.c1Green();
            baLineBuf[c4Pel + (c4Width * 2)] = rgbPel.c1Blue();
        }

        // Now lets write this buffer. 
        strmTarget << baLineBuf;
    }

    __conOut << L"File Creation complete" << NewLn;

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
