//
//  FILE NAME: FileSys3.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 09/09/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the third of the file system oriented demo
//  programs. This program is a file renamer. It will take all of the files
//  in a directory and rename them according to a pattern (with a continuously
//  incrementing values.)
//
//  For instance, if you wanted to name all of the files in a directory
//  with names like File0001.Txt, File0002.Txt, File0003.Txt, etc..., this
//  program will do that automatically. You give it a starting directory and a
//  file name pattern to use for renaming. The pattern uses a standard CIDLib
//  replacement token. For the above example, the pattern would be:
//
//      File%(1,4,0).Txt
//
//  This replacement token says that token #1 (which the program looks for)
//  should be right justified in a 4 character string, 0 filled.
//
//  If the pattern has spaces in it, then put it in double quotes. That's
//  legal as long as the target file system allows spaces.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This program assumes that the target file system is case insensitive
//      but case preserving (as NTFS and OS/2's HPFS are.)
//


// ----------------------------------------------------------------------------
//  Includes. We just include our main header which brings in anything we
//  we need. This header is set up as the precompiled header file.
// ----------------------------------------------------------------------------
#include    "FileSys3.Hpp"


// ----------------------------------------------------------------------------
//  Typedef our collection type. We use a bag of find buffer objects. In
//  many cases we would use a hash map for this kind of thing, but we have
//  special case sensitive and insensitive name comparisons. Hash maps don't
//  do that well since the hashing scheme takes into account the case of the
//  text of the file name.
// ----------------------------------------------------------------------------
typedef TBag<TFindBuf>  TFileCol;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::EExitCodes __eMainThreadFunc
(
            TThread&            thrThis
    ,       tCIDLib::TVoid*     pData
);

static const TPathStr& __pathGetKey
(
    const   TFindBuf&           fndbData
);

static tCIDLib::TVoid __RenameFiles
(
            TFileCol&           colFiles
    , const TString&            strPattern
    , const TString&            strTargetDir
);

static tCIDLib::TVoid __ShowUsage();



// ----------------------------------------------------------------------------
//  Local static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class.
//
//  __facFileSys3
//      This is our facility object, which we need in order to log errors and
//      load message resources. Since we have no particular needs, we just
//      create a regular TFacility object.
// ----------------------------------------------------------------------------
static TConsole     __conOut;
static TFacility    __facFileSys3
                    (
                        L"FileSys3"
                        , tCIDLib::EModType_Exe
                        , kCIDLib::c4MajVersion
                        , kCIDLib::c4MinVersion
                    );


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"FileSys3MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
static tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and testing program, we'd like to catch
    //  all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        // Display the program blurb
        __conOut << __facFileSys3.strMsg(kFSMsgs::midBlurb1) << NewLn;

        //
        //  Create a hash map of TPathName objects. This will let us look
        //  up all of the existing files in the directory before we starting
        //  renaming them.
        //
        //  The TFileCol name is just a typedef set up above, so we don't have
        //  to use the grotesquely long template names.
        //
        TFileCol colFileList;

        //
        //  Ok, lets first check out the command line parameters. We need
        //  to make sure that we got both of them, and that they directory
        //  exists and has entries and that the pattern is legal.
        //
        if (TSysInfo::c4CmdLineArgCount() < 3)
        {
            __ShowUsage();
            facCIDLib.ExitProcess(tCIDLib::EExit_FatalError);
        }

        // Get the two parameters out
        TString strTargetDir;
        TString strPattern;
        TSysInfo::CmdLineArg(2, strPattern);

        //
        //  If the target directory is a partial directory, then we need to
        //  complete it.
        //
        TString strTmp;
        TSysInfo::CmdLineArg(1, strTmp);
        TFileSys::QueryFullPath(strTmp, strTargetDir);

        // See if the target is a directory
        if (!TFileSys::bIsDirectory(strTargetDir))
        {
            __conOut << __facFileSys3.strMsg(kFSErrs::errcTargetDirNotFound, strTargetDir)
                     << DNewLn;
            return tCIDLib::EExit_BadParameters;
        }

        //
        //  See if there is anything in the target. If so, then this will
        //  return us a collection of TFindBuf objects for them. We need
        //  to create a search spec to find all files.
        //
        TPathStr pathSearch(strTargetDir);
        pathSearch.AddLevel(kCIDLib::pszAllFilesSpec);

        // Search the directory for all entries
        TFileSys::c4SearchDir(pathSearch, colFileList);

        //
        //  First of all, scarf out any directories from the list. We don't
        //  want to deal with them.
        //
        if (colFileList.bResetIter())
        {
            while (1)
            {
                if (colFileList.objCur().bIsDirectory())
                {
                    if (!colFileList.bFlushCur())
                        break;
                }
                 else
                {
                    if (!colFileList.bNext())
                        break;
                }
            }
        }

        // If that was all we had, then its empty
        if (colFileList.bIsEmpty())
        {
            __conOut << __facFileSys3.strMsg(kFSErrs::errcEmptyTarget)
                     << DNewLn;
            return tCIDLib::EExit_NotFound;
        }

        // Make sure that the pattern contains the '1' token
        if (!strPattern.bTokenExists(L'1'))
        {
            __conOut << __facFileSys3.strMsg(kFSErrs::errcInvalidPattern
                                            , strPattern)
                     << DNewLn;
            return tCIDLib::EExit_BadParameters;
        }

        // Say what we are about to do
        __conOut << __facFileSys3.strMsg
        (
            kFSMsgs::midRenaming
            , strTargetDir
            , strPattern
        ) << DNewLn;

        //
        //  Ok, it looks basically ok. So lets call the function that actually
        //  does the work.
        //
        __RenameFiles(colFileList, strPattern, strTargetDir);
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut <<  L"A CIDLib runtime error occured during processing. "
                 <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing. Error="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}


//
// FUNCTION/METHOD NAME: __pathGetKey
//
// DESCRIPTION:
//
//  A key extraction function to pass to the hash map. It pulls the path field
//  out of the TFindBuf objects that we get back from the directory search.
// ---------------------------------------
//   INPUT: fndbData is the data object
//
//  OUTPUT: None
//
//  RETURN: A const reference to the path field of the passed find buffer.
//
static const TPathStr& __pathGetKey(const TFindBuf& fndbData)
{
    return fndbData.pathFileName();
}


//
// FUNCTION/METHOD NAME: __RenameFiles
//
// DESCRIPTION:
//
//  This function does the grunt work of the program. The parameters have
//  been validated and we are given a collect with the list of files in the
//  target directory, and the target directory and the pattern.
// ---------------------------------------
//   INPUT: colFiles is the collection of files that already exist in the
//              target directory.
//          strPattern is the pattern that the user wants to rename to
//          strTargetDir is the target directory that is being operated
//              upon.
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __RenameFiles(        TFileCol&   colFiles
                                    , const TString&    strPattern
                                    , const TString&    strTargetDir)
{
    // Create a counter that will be used for the numbering
    tCIDLib::TCard4 c4Counter = 0;
    TPathStr        pathNewName;

    //
    //  Reset the internal iterator and use it for the outer loop to run
    //  through all the files in the list.
    //
    colFiles.bResetIter();

    //
    //  And create a cursor to use for the internal loop where we search
    //  the collection for each created pattern.
    //
    TFileCol::TCursor cursFind(&colFiles);

    // And now loop until we handle each file in the collection
    while (1)
    {
        //
        //  Create the file name for this counter value. This is our new
        //  target name to rename to.
        //
        pathNewName = strTargetDir;
        pathNewName.AddLevel(strPattern);
        pathNewName.ReplaceToken(TCardinal(c4Counter), L'1');

        //
        //  First see if this guy is already in the collection. If so, then
        //  the file is already there and we don't want to try to rename it.
        //
        //  Note that we use a cursor for this because we don't want to
        //  affect the internal iterator being used for the outer loop.
        //
        if (!cursFind.bReset())
            break;

        tCIDLib::TBoolean bFound = kCIDLib::False;
        do
        {
            if (!pathNewName.eICompare(cursFind.objCur().pathFileName()))
            {
                //
                //  One nicety here. If the file exists, but is not spelled
                //  in the same case as the pattern, then rename it to the
                //  correct case.
                //
                if (!pathNewName.bSameText(cursFind.objCur().pathFileName()))
                {
                    TFileSys::Rename
                    (
                        cursFind.objCur().pathFileName()
                        , pathNewName
                    );
                }

                //
                //  Note that this flush adjusts the internal iterator up to
                //  the next node, so its safe to do. If we had used a
                //  cursor for the outer loop, then we would have had copy
                //  it, adjust the copy upwards, and either keep it or throw
                //  it away according to whether we found thie file or not.
                //
                colFiles.FlushAt(cursFind);
                bFound = kCIDLib::True;
                break;
            }
        }   while (cursFind.bNext());

        //
        //  If the new file pattern did not already exist, then we need
        //  to rename the current file in the collection to the new pattern.
        //  Else we need to check and see whether we just did the last file
        //  in the collection.
        //
        if (!bFound)
        {
            TFileSys::Rename(colFiles.objCur().pathFileName(), pathNewName);

            //
            //  Move up to the next name in the collection. If no more,
            //  then we are done.
            //
            if (!colFiles.bNext())
                break;
        }
         else
        {
            // If we flushed the last node, then we are done
            if (colFiles.bIsEmpty())
                break;
        }

        // Move the counter up to the next value
        c4Counter++;
    }
}


//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  Shows the parameter usage for the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowUsage()
{
    __conOut << NewLn << __facFileSys3.strMsg(kFSMsgs::midUsage) << DNewLn;
}
