//
//  FILE NAME: FileSys2.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the second of the file system oriented demo
//  programs. This one queries TVolumeInfo objects for all of the available
//  volumes. It then prints out a nice report on all of them. It demonstrates
//  the use of collections, since we query a collection of volume info objects,
//  collection iteration, and formatted text stream output.
//
//  Note that this program will only get info for those volumes that are
//  ready. Floppies without disks or CD-ROMs without a CD in them are not
//  reported. Also, any unsupported filesystem types are not reported.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This program is so simple that it does not create a facility object
//      for itself, or have a main facility header.
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class.
// ----------------------------------------------------------------------------
static TConsole __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"FileSys2MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and testing program, we'd like to catch
    //  all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        // Lets create a bag of TString objects
        typedef TBag<TString>   TBagOPaths;
        TBagOPaths              colVolumes;

        //
        //  Ask the file system class to give us a list of all of the
        //  available volume paths. It returns how many it found, so indicate
        //  to user that none were available if it returns 0.
        //
        if (!TFileSys::c4QueryAvailableVolumePaths(colVolumes))
        {
            __conOut << NewLn << L"Could not find any volumes to report on"
                     << DNewLn;
            return tCIDLib::EExit_Normal;
        }

        // Indicate what we are displaying
        __conOut << NewLn << L"Information on available volumes:"
                 << DNewLn;

        //
        //  We need to create stream format objects for each of the columns
        //  that we want to deal with.
        //
        //  The columns are:
        //
        //  1) Volume path. A string, left justified in a width of 8
        //  2) Volume label. A string. Left justified in a width of 12
        //  3) File system type. A string. Left justified in a width of 10
        //  4) Bytes used. Numeric, right justified in a width of 16
        //  5) Total bytes. Numeric, right justified in a width of 16
        //  6) Max path length. Numeric, right justified in a width of 8
        //
        TStreamFmt strmfPath(8, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);
        TStreamFmt strmfLabel(12, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);
        TStreamFmt strmfType(10, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);
        TStreamFmt strmfUsed(16, 0, tCIDLib::EHJustify_Right, kCIDLib::chSpace);
        TStreamFmt strmfTotal(16, 0, tCIDLib::EHJustify_Right, kCIDLib::chSpace);
        TStreamFmt strmfPathLen(8, 0, tCIDLib::EHJustify_Right, kCIDLib::chSpace);

        //
        //  Output the headers for the colums with dashed underlines under
        //  each one.
        //
        __conOut << strmfPath << L"Volume" << TTextStream::ESpaces(2)
                 << strmfLabel << L"Label" << TTextStream::ESpaces(2)
                 << strmfType << L"Type" << TTextStream::ESpaces(2)
                 << strmfUsed << L"Bytes Used" << TTextStream::ESpaces(2)
                 << strmfTotal << L"Total Bytes" << TTextStream::ESpaces(2)
                 << strmfPathLen << L"Max Path"  << TTextStream::ESpaces(2)
                 << NewLn;

        __conOut << strmfPath << L"--------" << TTextStream::ESpaces(2)
                 << strmfLabel << L"------------" << TTextStream::ESpaces(2)
                 << strmfType << L"----------" << TTextStream::ESpaces(2)
                 << strmfUsed << L"----------------" << TTextStream::ESpaces(2)
                 << strmfTotal << L"----------------" << TTextStream::ESpaces(2)
                 << strmfPathLen << L"--------"  << TTextStream::ESpaces(2)
                 << NewLn;

        //
        //  Create a cursor for our bag and iterate the volumes, outputing
        //  a report for each one.
        //
        TBagOPaths::TCursor cursVolumes(&colVolumes);

        do
        {
            //
            //  Try to create a TVolumeInfo object for each one. If it fails,
            //  see if its because its either not ready (a floppy or CD with
            //  no disk in it) or its an unsupported file system.
            //
            try
            {
                const TVolumeInfo volCur(cursVolumes.objCur());

                // Calculate the used and total bytes
                const tCIDLib::TCard4 c4BytesPerUnit = 
                (
                    volCur.c4SectorsPerUnit() * volCur.c4BytesPerSector()
                );

                const tCIDLib::TFloat4 f4Total = 
                (
                    tCIDLib::TFloat4(volCur.c4TotalUnits() * c4BytesPerUnit)
                );

                const tCIDLib::TFloat4 f4Used =
                (
                    f4Total - (volCur.c4UnitsAvailable() * c4BytesPerUnit)
                );

                // Output the info for this volume
                __conOut << strmfPath << volCur.strVolumePath() << TTextStream::ESpaces(2)
                         << strmfLabel << volCur.strVolumeLabel() << TTextStream::ESpaces(2)
                         << strmfType << volCur.strFileSysType() << TTextStream::ESpaces(2)
                         << strmfUsed << TLocFloat(f4Used, 0) << TTextStream::ESpaces(2)
                         << strmfTotal << TLocFloat(f4Total, 0) << TTextStream::ESpaces(2)
                         << strmfPathLen << volCur.c4MaxPathLenComp() << TTextStream::ESpaces(2)
                         << NewLn;
            }

            catch(const TError& errToCatch)
            {
                //
                //  If its a volume query error from CIDLib, then see what the
                //  underlying kernel error is.
                //
                __conOut << strmfPath << cursVolumes.objCur()
                         << TTextStream::ESpaces(2);
                __conOut.SetDefaultFormat();
                if (errToCatch.bCheckError(facCIDLib, kCIDErrs::errcFile_VolumeQuery))
                {
                    if (errToCatch.errcKrnlId() == kKrnlErrors::errcUnsupportedFileSystem)
                        __conOut << L"Unknown file system type" << NewLn;
                    else if (errToCatch.errcKrnlId() == kKrnlErrors::errcNotReady)
                        __conOut << L"Device not ready" << NewLn;
                    else
                        throw;
                }
                 else
                {
                    // Let it propogate
                    throw;
                }
            }

        }   while (cursVolumes.bNext());
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut <<  L"A CIDLib runtime error occured during processing. "
                 <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing. Error="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
