//
//  FILE NAME: FileSys1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first of the file system oriented demo
//  programs. This one just provides a very simple directory listing program.
//  This demonstrates directory iteration, recursive directory searching,
//  and some formatting of objects and text stream stuff.
//
//  It outputs pretty much the same as the built in NT "DIR" command.
//
//  CAVEATS/GOTCHAS:
//
//  1)  This program is so simple that it does not create a facility object
//      for itself, or have a main facility header.
//
//  2)  We create a TFindBuf object on each level of the recursive search.
//      In a 'real' program we'd probably be a little more paranoid about
//      stack overflow and just allocate one on each level, using a janitor
//      object to clean it up.
//
//  3)  This program does not attempt to be language independent.
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local, static data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class.
// ----------------------------------------------------------------------------
TConsole        __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"FileSys1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __RecursiveSearch
//
// DESCRIPTION:
//
//  This method is called if the user wants to do a recursive directory
//  operation. We just recurse our way down the chain.
// ---------------------------------------
//   INPUT: pathFullPath represents the path for this level of the search.
//              It does not end with a slash.
//          pathWildCard is the wildcard to search for.
//
//  OUTPUT: fposTotalBytes, c4TotalFiles are kept updated with the totals
//              for files and total bytes in those files.
//
//  RETURN: None
//
static tCIDLib::TVoid
__RecursiveSearch(  const   TPathStr&           pathFullPath
                    , const TPathStr&           pathWildCard
                    ,       tCIDLib::TFilePos&  fposTotalBytes
                    ,       tCIDLib::TCard4&    c4TotalFiles)
{
    //
    //  First we want to iterate this whole directory and display anything
    //  that matches the wildcard.
    //
    TDirIter    diterLevel;
    TFindBuf    fndbSearch;

    if (diterLevel.bFindFirst(pathFullPath, pathWildCard, fndbSearch))
    {
        __conOut  << DNewLn << L"  Directory of " << pathFullPath << DNewLn;

        //
        //  Just iterate the rest of files, displaying each one. Keep
        //  up with how many files and how many bytes total.
        //
        do
        {
            //
            //  Just do the default format of a find buffer by just
            //  dumping it to the console.
            //
            __conOut << fndbSearch << NewLn;

            c4TotalFiles++;
            fposTotalBytes += fndbSearch.fposSize();

        }   while (diterLevel.bFindNext(fndbSearch));
    }

    //
    //  Now start a new search, for directories only. For each one, we
    //  want to recurse and handle it the same way. Our canbe and mustbe
    //  search attributes both are 'directory', which insures that this
    //  is all we will get. We pass our own search spec, which won't limit
    //  the search in any way. pszAllFilesSpec is a predefined spec that
    //  has this effect, so its safer to do than something hard coded.
    //
    tCIDLib::EFileAttrs eCanAndMust = tCIDLib::EFileAttr_Directory;
    if (diterLevel.bFindFirst
    (
        pathFullPath
        , kCIDLib::pszAllFilesSpec
        , fndbSearch
        , eCanAndMust
        , eCanAndMust))
    {
        //
        //  Ok, for all of the rest of the finds, we want to recurse upon
        //  ourselves.
        //
        do
        {
            // Don't do special directories, or we will freak out forever
            if (!fndbSearch.bIsSpecialDir())
            {
                // Now handle this level
                __RecursiveSearch
                (
                    fndbSearch.pathFileName()
                    , pathWildCard
                    , fposTotalBytes
                    , c4TotalFiles
                );
            }
        }   while (diterLevel.bFindNext(fndbSearch));
    }
}


//
// FUNCTION/METHOD NAME: __ShowUsage
//
// DESCRIPTION:
//
//  Shows the parameter usage for the program.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static tCIDLib::TVoid __ShowUsage()
{
    __conOut    <<  L"Usage: FileSys1 wildcard [/s]" << DNewLn
                <<  L"  wildcard  The file search specification, e.g. *.*" << NewLn
                <<  L"  /s        Indicates a recursive directory search." << NewLn
                <<  L"              Otherwise just the current directory is listed"
                << DNewLn;
}


//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  Since this is a demo and testing program, we'd like to catch
    //  all exceptions cleanly and report them. So put the whole thing
    //  in a try.
    //
    try
    {
        // Check out the command line parms for valid count
        tCIDLib::TCard4 c4Args = TSysInfo::c4CmdLineArgCount();
        if (c4Args > 3)
        {
            __ShowUsage();
            return tCIDLib::EExit_BadParameters;
        }

        //
        //  If there is a single user parm and its /?, then show the program
        //  usage and exit. If its not, then it must be the wildcard. If
        //  there is none, then set wildcard to default.
        //
        TPathStr pathWildCard;
        if (c4Args >= 2)
        {
            TSysInfo::CmdLineArg(1, pathWildCard);

            if (pathWildCard == L"/?")
            {
                __ShowUsage();
                return tCIDLib::EExit_Normal;
            }
        }
         else
        {
            // Set it to the default 'all files' spec
            pathWildCard = kCIDLib::pszAllFilesSpec;
        }

        //
        //  If we have another parm, then get it. If it is /s, then set
        //  the bRecursive flag. Else show usage and exit.
        //
        tCIDLib::TBoolean bRecursive = kCIDLib::False;
        if (c4Args == 3)
        {
            TString strTmp;
            TSysInfo::CmdLineArg(2, strTmp);

            if (strTmp != L"/s")
            {
                __ShowUsage();
                return tCIDLib::EExit_BadParameters;
            }
            bRecursive = kCIDLib::True;
        }

        //
        //  Indicate what we are list the directory for. We get the full
        //  path of the partial path, which is display as what we are
        //  searching. We remove the name and extension, and trailing
        //  separator to just get the actual directory that we are
        //  searching.
        //
        TPathStr    pathFullPath;
        TFileSys::QueryFullPath(pathWildCard, pathFullPath);
        pathFullPath.bRemoveNameExt();
        pathFullPath.bRemoveTrailingSeparator();

        // Now get the volume info out the path
        TPathStr    pathVolume;
        pathFullPath.bQueryVolume(pathVolume);

        // Pull the path part out of the wildcard, if any
        pathWildCard.bRemovePath();

        // Query the volume information for this volume
        TString strVolumeLabel;
        TString strSerialNum;
        try
        {
            TFileSys::QueryVolumeLabel(strVolumeLabel, strSerialNum, pathVolume);
        }

        catch(const TError& errToCatch)
        {
            //
            //  Since this is the first thing we do using the info we got,
            //  lets check to see if the drive is valid at all. If the 
            //  underlying kernel error was 'path not found', then we give
            //  a specific error, else we rethrow it.
            //
            if (errToCatch.errcKrnlId() == kKrnlErrors::errcPathNotFound)
            {
                __conOut    << NewLn << L"The volume: " << pathVolume
                            << L" was not valid" << NewLn;
                return tCIDLib::EExit_ResourceAccess;
            }
            throw;
        }

        // Now output the preamble stuff
        __conOut    << L"  Volume in drive " << pathVolume
                    << L" is " << strVolumeLabel << NewLn;
        __conOut    << L"  Volume Serial Number is " << strSerialNum << DNewLn;

        //
        //  Ok, lets do the directory listing. If this is a recursive search
        //  we kick off a local recursive function. If its not, we just do
        //  it right here since its so trivial.
        //
        tCIDLib::TFilePos   fposTotalBytes = 0;
        tCIDLib::TCard4     c4TotalFiles = 0;
        if (bRecursive)
        {
            __RecursiveSearch
            (
                pathFullPath
                , pathWildCard
                , fposTotalBytes
                , c4TotalFiles
            );
        }
         else
        {
            __conOut << L"  Directory of " << pathFullPath << DNewLn;

            TDirIter    diterSearch;
            TFindBuf    fndbSearch;
            if (diterSearch.bFindFirst(pathWildCard, fndbSearch))
            {
                //
                //  Just iterate the rest of files, displaying each one. Keep
                //  up with how many files and how many bytes total.
                //
                do
                {
                    //
                    //  Just do the default format of a find buffer by just
                    //  dumping it to the console.
                    //
                    __conOut << fndbSearch << NewLn;

                    c4TotalFiles++;
                    fposTotalBytes += fndbSearch.fposSize();

                }   while (diterSearch.bFindNext(fndbSearch));
            }
        }

        if (!c4TotalFiles)
        {
            __conOut    << L"There were no files found at: "
                        << pathFullPath << DNewLn;
            return tCIDLib::EExit_NotFound;
        }

        // Output the totals
        __conOut    << NewLn
                    << L"   Total Files: " << c4TotalFiles << NewLn
                    << L"   Total Bytes: " << fposTotalBytes << DNewLn;
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing. "
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my test programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing. Error="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
