//
//  FILE NAME: DispRGB.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 03/23/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the very simple RGB dump display program. Some
//  of the demo programs output in a very simple dump program used by some
//  public domain programs. This is a simple program, which does not use
//  any CIDLib functionality, to display them. There is also a CIDDispRGB
//  demo program that does the same thing, using CIDLib functionality.
//
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    <windows.h>
#include    <stdio.h>
#include    "DispRGB_ResourceIds.H"


// ----------------------------------------------------------------------------
//  Local, constant data
// ----------------------------------------------------------------------------
const COLORREF          __rgbWindowBgn = RGB(128,128,128);
const TCHAR* const      __pszAppName = L"DispRGB.Exe";
const TCHAR* const      __pszAppTitle = L"CIDLib RGB Dump Viewer";
const ULONG             __ulMaxFileLen = 1024;


// ----------------------------------------------------------------------------
//  Local data
//
//  __hbmpImage
//      This is the memory bitmap that the image is stored in. The paint
//      event for the main window just blits from this bitmap.
//
//  __hInstance
//      The program instance passed by the system to our main
//
//  __hwndMain
//      The handle of the main window.
//
//  __szFileName
//      This is the name of the file that we are currently displaying, if
//      any. It is set via the common file dialog.
//
//  __ulImageCX, __ulImageCY
//      These are the sizes of the image as read from the dump file. These
//      are the pixel sizes of the image, with each pixel being made up
//      of 3 bytes (one each for RGB values.)
// ----------------------------------------------------------------------------
static  HBITMAP     __hbmpImage;
static HINSTANCE    __hInstance;
static HWND         __hwndMain;
static TCHAR        __szFileName[__ulMaxFileLen+1];
static ULONG        __ulImageCX, __ulImageCY;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static LRESULT CALLBACK __lresWndProc
(
            HWND            hwndThis
    ,       UINT            uMsg
    ,       WPARAM          wParam
    ,       LPARAM          lParam
);

static VOID __OpenDisFile();

static VOID __ShowMessage
(
    const   TCHAR* const    pszMessage
    , const TCHAR* const    pszToken1 = 0
    , const TCHAR* const    pszToken2 = 0
    , const BOOL            bError = TRUE
);


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __LoadImage
//
// DESCRIPTION:
//
//  This is called to read in a new image file and create the memory bitmap
//  used to hold it. If the operation fails, it will either leave the old
//  bitmap intact or the bitmap handle set to 0.
// ---------------------------------------
//   INPUT: 
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __LoadImage()
{
    // Save the current image size
    const ULONG ulOldCX = __ulImageCX;
    const ULONG ulOldCY = __ulImageCY;

    //
    //  Lets parse the file first because if its not right, we can't really
    //  do much.
    //
    HANDLE hflImage = CreateFile
    (
        __szFileName
        , GENERIC_READ
        , 0
        , 0
        , OPEN_EXISTING
        , FILE_FLAG_SEQUENTIAL_SCAN
        , 0
    );

    if (hflImage == (HANDLE)INVALID_HANDLE_VALUE)
    {
        __ShowMessage(L"The file %s could not be opened.", __szFileName);
        return;
    }

    USHORT  usTmp;
    ULONG   ulActual;
    try
    {
        //
        //  The first information in the file is two words with the width
        //  and height of the image data.
        //
        if (!ReadFile(hflImage, &usTmp, sizeof(usTmp), &ulActual, 0))
        {
            throw(L"Could not read the image horizontal size");
        }
        __ulImageCX = (ULONG)usTmp;

        if (!ReadFile(hflImage, &usTmp, sizeof(usTmp), &ulActual, 0))
        {
            throw(L"Could not read the image vertical size");
        }
        __ulImageCY = (ULONG)usTmp;

        //
        //  Make sure that the values seem reasonable. For this simple
        //  program, reasonable means less than an arbitary value.
        //
        if ((__ulImageCX < 8) || (__ulImageCX > 2048)
        ||  (__ulImageCY < 8) || (__ulImageCY > 2048))
        {
            throw(L"The image size was invalid");
        }
    }

    catch(const TCHAR* pszErrMessage)
    {
        __ShowMessage(pszErrMessage);
        CloseHandle(hflImage);
        return;
    }

    //
    //  Calculate the memory required to hold the image date that we
    //  will use to initialize the bitmap. Not that the scan lines have
    //  to be word aligned.
    //
    BOOL        bPadded = (((__ulImageCX * 3) & 1) == 1);
    const ULONG ulScanLineBytes = __ulImageCX * 3 + (bPadded ? 1 : 0);

    //
    //  Ok, so allocate a buffer of the right size now that we know how
    //  much space we need for a scan line.
    //
    const ULONG ulBufferBytes = ulScanLineBytes * __ulImageCY;
    BYTE* pbBuf = new BYTE[ulBufferBytes];

    //
    //  And allocate a buffer to read in scan lines from the file. They
    //  are not in the order that we need for the bitmap init data.
    //
    const ULONG ulFileLineBytes = __ulImageCX * 3;
    BYTE* pbFileLine = new BYTE[ulFileLineBytes];

    try
    {
        //
        //  Ok, now we read in the data from the file and put it into the
        //  image.
        //
        for (ULONG ulRow = 0; ulRow < __ulImageCY; ulRow++)
        {
            // Read in the row number
            if (!ReadFile(hflImage, &usTmp, sizeof(usTmp), &ulActual, 0))
            {
                throw(L"Could not read the row number from the file");
            }

            // If it is not the same as our row index, then we are screwed
            if (usTmp != ulRow)
            {
                throw(L"The row number from the file was != to our row");
            }

            //
            //  Read in the next line from the file. It is always the same
            //  size so we can read the whole thing in and work on the
            //  whole line at once.
            //
            if (!ReadFile(hflImage, pbFileLine, ulFileLineBytes, &ulActual, 0))
            {
                throw(L"Could not read the next line from the file");
            }

            //
            //  Calculate the destiation buffer offset. The problem here is
            //  that the file is in bottom up format, but the bitmap for
            //  Windoze must be top down.
            //
            ULONG ulBufIndex = (__ulImageCY - (ulRow+1)) * ulScanLineBytes;

            if (ulBufIndex + ulScanLineBytes > ulBufferBytes)
            {
                throw(L"File content overran the bitmap color buffer");
            }

            //
            //  Ok, so lets pull out the bytes and put them into the bitmap
            //  buffer. The format is all of the red bytes, all of the green
            //  bytes, and all the red bytes.
            //
            //  For reasons unknown to me, I have to put them into the
            //  bitmap buffer in the order BGR, i.e. reversed.
            //
            for (ULONG ulCol = 0; ulCol < __ulImageCX; ulCol++)
            {
                pbBuf[ulBufIndex++] = pbFileLine[ulCol + (__ulImageCX * 2)];
                pbBuf[ulBufIndex++] = pbFileLine[ulCol + __ulImageCX];
                pbBuf[ulBufIndex++] = pbFileLine[ulCol];
            }
        }

        //
        //  If there is an existing bitmap, we need to destroy it. But,
        //  as an optimization, see if the new file is the same size as
        //  the old one. If so, then we can keep it.
        //
        if (__hbmpImage && ((ulOldCX != __ulImageCX) || (ulOldCY != __ulImageCY)))
            DeleteObject(__hbmpImage);

        //
        //  Now we want to create the backing bitmap that we will put the
        //  actual data into. The paint event for the window basically just
        //  amounts to a blit from this bitmap to the window.
        //
        __hbmpImage = CreateBitmap(__ulImageCX, __ulImageCY, 1, 24, pbBuf);
        if (!__hbmpImage)
        {
            throw(L"Could not create the new bitmap");
        }
    }

    catch(const TCHAR* pszErrMessage)
    {
        __ShowMessage(pszErrMessage);
    }

    // Clean up the image buffers
    delete pbBuf;
    delete pbFileLine;

    // We are done with the file now, so close it
    CloseHandle(hflImage);
}


//
// FUNCTION/METHOD NAME: __bInitApplication(hInstance)
//
// DESCRIPTION:
//
//  Initializes the application by doing the grunt work setup of
//  registering the window class and creating the main window.
// ---------------------------------------
//   INPUT: hInstance if the program instance
//
//  OUTPUT: None
//
//  RETURN: TRUE if successful, else FALSE
//
static BOOL __bInitApplication(HINSTANCE hInstance)
{
    // Store instance handle in a global variable
	__hInstance = hInstance;

    //
    //  Fill in window class structure with parameters that describe
    //  the main window.
    //
    WNDCLASSEX  wcData;
    wcData.cbSize           = sizeof(WNDCLASSEX);
    wcData.style            = CS_HREDRAW | CS_VREDRAW;
    wcData.lpfnWndProc      = (WNDPROC)__lresWndProc;
    wcData.cbClsExtra       = 0;
    wcData.cbWndExtra       = 0;
    wcData.hInstance        = hInstance;
    wcData.hIcon            = 0; // LoadIcon(hInstance, szAppName);
    wcData.hCursor          = LoadCursor(NULL, MAKEINTRESOURCE(IDI_APPLICATION));
    wcData.hbrBackground    = 0;
	wcData.lpszMenuName     = MAKEINTRESOURCE(MID_MAIN);
    wcData.lpszClassName    = __pszAppName;
    wcData.hIconSm          = 0;

	if (!RegisterClassEx(&wcData))
        return FALSE;

	//
    //	Create the window. Make it a popup since we don't want any title
	//	bar or anything like that.
	//
	__hwndMain = CreateWindowEx
    (
        0
        , __pszAppName
        , __pszAppTitle
        , WS_OVERLAPPEDWINDOW | WS_VISIBLE
        , CW_USEDEFAULT
        , CW_USEDEFAULT
        , CW_USEDEFAULT
        , CW_USEDEFAULT
        , NULL
        , NULL
        , hInstance
        , NULL
    );

	if (!__hwndMain)
		return (FALSE);

    return TRUE;
}


//
// FUNCTION/METHOD NAME: __lresWndProc(hwndThis, uMsg, wParam, lParam)
//
// DESCRIPTION:
//
//  This is the window proc for the program.
// ---------------------------------------
//   INPUT: hwndThis the program's frame window handle
//          uMsg is the message being recieved
//          wParam, lParam are the window parameters.
//
//  OUTPUT: None
//
//  RETURN: Depends on the message.
//
static LRESULT CALLBACK __lresWndProc(  HWND        hwndThis
                                        , UINT      uMsg
                                        , WPARAM    wParam
                                        , LPARAM    lParam)
{
    if (uMsg == WM_PAINT)
    {
	    PAINTSTRUCT psData;
        HDC hDC = BeginPaint(hwndThis, &psData);

        // Fill in the invalid area if needed
        if (psData.fErase)
        {
            HBRUSH hbrBgn = CreateSolidBrush(__rgbWindowBgn);
            FillRect(hDC, &psData.rcPaint, hbrBgn);
            DeleteObject(hbrBgn);
        }

        //
        //  Blit the invalid area from the backing bitmap to the window
        //  if there is a bitmap. The invalid rectangle was given to us
        //  in the paint structure.
        //
        if (__hbmpImage)
        {
            HDC hdcImage = CreateCompatibleDC(0);
            if (hdcImage)
            {
                HANDLE hOld = SelectObject(hdcImage, __hbmpImage);
                BitBlt
                (
                    hDC
                    , psData.rcPaint.left
                    , psData.rcPaint.top
                    , psData.rcPaint.right - psData.rcPaint.left
                    , psData.rcPaint.bottom - psData.rcPaint.top
                    , hdcImage
                    , psData.rcPaint.left
                    , psData.rcPaint.top
                    , SRCCOPY
                );

                SelectObject(hdcImage, hOld);
                DeleteDC(hdcImage);
            }
        }

        EndPaint(hwndThis, &psData);
        return 0;
    }
     else if (uMsg == WM_DESTROY)
    {
        PostQuitMessage(0);
    }
     else if (uMsg == WM_COMMAND)
    {
        if (LOWORD(wParam) == MID_FILE_OPEN)
        {
            // Open the file, read it, and create the image bitmap
            __OpenDisFile();

            // Force an update of the client area
            InvalidateRect(__hwndMain,  0, TRUE);
        }
         else if (LOWORD(wParam) == MID_FILE_EXIT)
        {
            PostQuitMessage(0);
        }
    }
     else
    {
        return DefWindowProc(hwndThis, uMsg, wParam, lParam);
    }
    return 0;
}


//
// FUNCTION/METHOD NAME: __OpenDisFile
//
// DESCRIPTION:
//
//  This function is called from the window procedure when the user
//  selects the file open option. If all goes well, then this ends with
//  the image in the new file being loaded into the global bitmap that is
//  used by the paint event.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __OpenDisFile()
{
    //
    //  Display the standard file dialog to get the file that the user wants
    //  to view.
    //
    OPENFILENAME OPInfo = {0};
    OPInfo.lStructSize = sizeof(OPENFILENAME);
    OPInfo.hwndOwner  = __hwndMain;
    OPInfo.lpstrFilter = L"Dump Files\0*.Dis\0\0";
    __szFileName[0] = 0;
    OPInfo.lpstrFile = __szFileName;
    OPInfo.nMaxFile = __ulMaxFileLen;
    OPInfo.lpstrTitle = L"Open A Dis File";
    OPInfo.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_LONGNAMES;
    OPInfo.lpstrDefExt = L"Dis";
    if (!GetOpenFileName(&OPInfo))
        return;

    //
    //  The user provided a file name, so call the image loading function
    //  which will do the rest of the work.
    //
    __LoadImage();
}


//
// FUNCTION/METHOD NAME: __ShowMessage
//
// DESCRIPTION:
//
//  This function will do a message box popup to show an error or info
//  message. If bError is FALSE, then its assumed to be informational.
// ---------------------------------------
//   INPUT: pszMessage is the message text. It can have printf replacement
//              tokens in it.
//          pszTokenX are the tokens to replace in the message.
//          bError indicates whether this message is an error message or
//              an informational messasge. Defaults to TRUE.
//
//  OUTPUT: None
//
//  RETURN: None
//
static VOID __ShowMessage(  const   TCHAR* const    pszMessage
                            , const TCHAR* const    pszToken1
                            , const TCHAR* const    pszToken2
                            , const BOOL            bError)
{
    static TCHAR szDisplay[2048];

    if (pszToken2)
        swprintf(szDisplay, pszMessage, pszToken1, pszToken2);
     else if (pszToken1)
        swprintf(szDisplay, pszMessage, pszToken1);
     else
        wcscpy(szDisplay, pszMessage);

    // Build up the correct message box flags
    UINT mbFlags = MB_OK | MB_APPLMODAL;
    if (bError)
        mbFlags |= MB_ICONSTOP;
    else
        mbFlags |= MB_ICONINFORMATION;

    // And do the popup
    MessageBox(__hwndMain, szDisplay, __pszAppTitle, mbFlags);
}



// ----------------------------------------------------------------------------
//  Global functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: WinMain
//
// DESCRIPTION:
//
//  This guy just takes the single command line parameter, which is the
//  file name, and stores it away. Then it sets up the windows, reads the
//  file contents, and displays it.
// ---------------------------------------
//   INPUT: hInstance is the process instance of this program
//          lpCmdLine is the command line string
//          nShowCmd indicates whether the window is to be initially
//              displayed.
//
//  OUTPUT: None
//
//  RETURN: 0 if successful, 1 if a failure
//
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE, LPSTR, int nShowCmd)
{
    // Init the app
    if (!__bInitApplication(hInstance))
        return FALSE;

	ShowWindow(__hwndMain, nShowCmd);
	UpdateWindow(__hwndMain);

    MSG qmsgGet;
	while (GetMessage(&qmsgGet, NULL, 0, 0) == TRUE)
        DispatchMessage(&qmsgGet);

	return (qmsgGet.wParam);
}
