//
//  FILE NAME: Collect4.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/07/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the fourth of the collection oriented demo
//  programs. This program demonstrates more of the type safety of the
//  collections, plus it demonstrates the TQueue collection which has not
//  be used yet.
//
//  This program simulates what a queue would be used for, a situation where
//  there are multiple 'producer' threads that are putting objects into a
//  queue which is read from one element at a time by a 'consumer' thread
//  that handles each element one at a time.
//
//  There are number of reasons for this kind of scheme. One is that it
//  naturally serializes access to a resource while allowing multiple threads
//  to make requests or perform actions against it. For instance, multiple
//  threads might want to log messages to a server. You don't want to have
//  a separate network connection for each thread so you can set up a thread
//  that monitors the queue and pulls elements out and sends them to the
//  server. The threads that send messages just dump them to the queue and
//  go on about their business without waiting around since they know that
//  the queue consumer thread will handle it.
//
//  There are cases where the thread does want to wait, in which case the
//  element put into the queue can have an event object in it. The consumer
//  thread can post the event after handling the thread's request.
//
//  In this program, we just simulate this kind of situation. The worker
//  threads sleep for a given period of time, then they put their own
//  thread name on the queue. This is a useless program but it just illustrates
//  the issue. In reality, the worker threads might be waiting on a client
//  machine to send a request which they would put onto the queue.
//
//  Note that the sleep time is generated as a random number by the main
//  thread and passed to the worker thread. This demonstrates how nice the
//  CIDLib thread start mechanism is. Since the the thread that starts a new
//  thread is in sync mode, it won't return until the started thread calls
//  its Sync() method. So the main thread can just pass a pointer to the
//  counter that is using. The worker thread will get its copy before calling
//  Sync(), so there is no need to dynamically allocate a value or provide
//  another synchronization mechanism just to pass data to each new thread.
//
//  The queue consumer thread just waits for information to arrive and, when
//  it does, it pulls the string out and writes it out. It will wait for X
//  number of elements to be gotten from the worker threads, then it asks them
//  to shut down and then it exits.
//
//  Note that no external synchronization is used here because the writers
//  do a single operation and the read does a single operation. The previous
//  demo required that each thread do multiple operations on the collection,
//  which had to be atomic, so it locked the collection itself and did all
//  of the operations while the collection was locked. This one can just
//  depend on the natural safety of a thread safe collection.
//  


// ----------------------------------------------------------------------------
//  Includes.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);

tCIDLib::EExitCodes __eWorkerThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __colTest
//      A pointer to the queue that will be used for this test. We make it
//      a thread safe collection.
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
//
//  __unamThreads
//      This is a unique name generator that is used to generate names for
//      the threads we spin up. Threads must have unique names with the
//      process.
// ----------------------------------------------------------------------------
static TQueue<TString>  __colTest(tCIDLib::EMTState_Safe);
static TConsole         __conOut;
static TUniqueName      __unamThreads(L"WorkerThread%(1)");


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Collect4MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. 
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    //
    //  This is the info on our threads. How many we want, the node type
    //  which is a counted pointer to a thread (because threads are not
    //  copyable we have to store them as pointers in the collection), and
    //  an object array of that many threads.
    //
    //  The constructor of the object array kicks of default constructors
    //  for the elements, but the elements are TCntPtr objects, not threads
    //  themselves, so this does not actually create any threads yet.
    //
    //  This is done outside the try block so that the exception code below
    //  can ask the threads to stop if an exception occurs.
    //
    const tCIDLib::TCard4 c4ThreadCount = 5;
    typedef TCntPtr<TThread> TThreadPtr;
    TObjArray<TThreadPtr> colOfThreads(c4ThreadCount);

    // Default return value
    tCIDLib::EExitCodes eRet = tCIDLib::EExit_Normal;
    try
    {
        // Seed the random number generator with some bogus number
        TMathLib::SeedRandom(0xF19DA31);

        //
        //  Create a list of worker threads. We just create 4 of them and
        //  let them run for a while. Threads are not copyable so we have
        //  to deal with them by reference, which means that the nodes are
        //  really pointer wrapper objects. We use the counted pointer class
        //  since its convenient for this type of thing, though we really
        //  do require all of its power.
        //
        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        {
            // Create the thread, give it the next unique name
            colOfThreads[c4Index].SetPointer
            (
                new TThread
                (
                    __unamThreads.strQueryNewName()
                    , __eWorkerThreadFunc
                )
            );
        }

        //
        //  Now start them all up. We could have started them above. But,
        //  in general, this is safer. If something happened in the loop
        //  above that caused an exception, some would have just  been
        //  started for nothing. At this point we know for sure that all
        //  of them are there and ready to rock and roll.
        //
        //  Note that, since we are using a pointer wrapper, the [] operator
        //  just gets us the wrapper object. objData() gets us a reference
        //  to the object its wrapping.
        //
        for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        {
            //
            //  Get a random number between 1000 and 2000. This will be the
            //  sleep time for this thread.
            //
            tCIDLib::TCard4 c4SleepTime = 0;
            while ((c4SleepTime < 1000) || (c4SleepTime > 2000))
                c4SleepTime = TMathLib::c4RandomNum();

            //
            //  Start the thread up now. We don't come back until its up
            //  and got a copy of its passed start time.
            //
            colOfThreads[c4Index].objData().Start(&c4SleepTime);
        }

        //
        //  Now enter our main loop. We wait for 32 elements to arrive then
        //  we exit.
        //
        tCIDLib::TCard4 c4Count = 0;
        while (c4Count < 32)
        {
            //
            //  Wait on something to arrive in the queue. We tell it we are
            //  willing to wait forever.
            //
            TString strNextElem = __colTest.objGetNext(kCIDLib::c4MaxWait);
            __conOut << L"    Got element from thread: " << strNextElem
                     << NewLn;

            c4Count++;
        }
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        eRet = tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        eRet = tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        eRet = tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    //
    //  Now lets go through the list and ask each thread to shutdown. We
    //  also then wait for it to really die, since returning from the
    //  RequestShutdown() just means that the thread sees the request and
    //  is on the way out.
    //
    __conOut << L"Asking threads to stop..." << NewLn;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCount; c4Index++)
    {
        // Wait for up to 3 seconds for it to respond
        colOfThreads[c4Index].objData().RequestShutdown(3 Seconds);

        // And now wait for it to die, up to 1 seconds
        colOfThreads[c4Index].objData().eWaitForDeath(1 Seconds);
    }

    //
    //  When colOfThreads goes out of scope it will destroy its elements.
    //  The reference counted threads in the elements only have 1 reference
    //  so this will cause the reference counts to go to 0, so the thread
    //  objects will be deleted.
    //
    //  That's why we could not just do a RequestShutdown() and fall right
    //  out. We had to wait explictly for them to die because otherwise
    //  we might fall out before the last couple actually got to fully
    //  terminate. That would cause an error because you cannot destruct
    //  a thread object while its running.
    //

    return tCIDLib::EExit_Normal;
}



//
// FUNCTION/METHOD NAME: __eWorkerThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the worker threads. Each instance
//  just sleeps some random amount of time. When it wakes up, it puts its
//  name on the global test queue collection.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//          pData is the optional data for this thread. In this case it is
//              a pointer to a TCard4 value that holds our sleep time.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes __eWorkerThreadFunc(TThread&            thrThis
                                        , tCIDLib::TVoid*   pData)
{
    //
    //  Get our sleep time out that was passed to us. We need to do this
    //  before we sync and let the caller go.
    //
    const tCIDLib::TCard4 c4SleepTime = *(tCIDLib::TCard4*)pData;

    // Now let the main thread go against
    thrThis.Sync();

    //
    //  Enter our loop. We just sleep for our sleep time, then we wake up
    //  and put our string name onto the test queue. We check for shutdown
    //  requests each time.
    //
    while (1)
    {
        // Check for a sync request
        thrThis.Sync();

        // If we get an exit signal, then exit
        if (thrThis.bCheckShutdownRequest())
            break;

        // Sleep for our appointed time
        TThread::Sleep(c4SleepTime);

        // Add our name to the queue
        __colTest.Add(thrThis.strName());
    }

    return tCIDLib::EExit_Normal;
}
