//
//  FILE NAME: Collect3_Thread.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/07/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the simple thread class derivative that the program
//  uses for its test. Each thread is given a pointer to a sorted bag of
//  TCardinal values, which is stores away for use. There are two sorts of
//  threads, adders and removers, which is controlled by a parameter in their
//  constructor.
//
//  Each thread started sleeps for some small random time. Then it wakes up
//  and creates another randum number. It checks the the bag to see if that
//  random value is in the bag. If not, then it adds the value if its an adder.
//  If it is, and its a remover, then it removes that element.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes.
// ----------------------------------------------------------------------------
#include    "Collect3.Hpp"


// ----------------------------------------------------------------------------
//  Local static data
//
//  __unamThreads
//      This is a unique name object that is used to create the unique names
//      required for the multiple instances of the threads.
// ----------------------------------------------------------------------------
static TUniqueName  __unamThreads(L"ColThread%(1)");


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TColThread,TThread)


// ----------------------------------------------------------------------------
//   CLASS: TColThread
//  PREFIX: thr
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
//  TColThread: Constructors and destructors
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: TColThread
//
// DESCRIPTION:
//
//  The only available constructor takes the info that the thread needs. It
//  will be started later by the caller.
// ---------------------------------------
//   INPUT: pcolToUse is the collection that we should use for the test.
//              We just reference it, but don't adopt it.
//          eType is the type of thread that this instance should be, an adder
//              or remover.
//          pstrmOut is the stream that should be used for text output. We
//              just reference it, not adopt it.
//
//  OUTPUT: None
//
//  RETURN: None
//
TColThread::TColThread(         TTestCol* const             pcolToUse
                        , const TColThread::EThreadTypes    eType
                        ,       TTextStream* const          pstrmOut) :

    TThread(__unamThreads.strQueryNewName())
    , __eType(eType)
    , __pcolToUse(pcolToUse)
    , __pstrmOut(pstrmOut)
{
}

TColThread::~TColThread()
{
}


// ----------------------------------------------------------------------------
//  TColThread: Protected, inherited methods
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _eProcess
//
// DESCRIPTION:
//
//  This is the thread function for the test thread. It works on the collection
//  stored during construction.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes TColThread::_eProcess()
{
    //
    //  Seed the random number generator differently for each thread. Note that
    //  this does not require synchronization because the main thread won't
    //  return from the Start() until we call Sync(), so it cannot start another
    //  thread. And any started before us must also have made it to Sync() before
    //  this instance could start.
    //
    static tCIDLib::TCard4 c4Seed = 0xFACEABCD;
    TMathLib::SeedRandom(c4Seed <<= 1);

    // Let the starting thread go
    Sync();

    //
    //  Loop until we are asked to shutdown. The main thread will let us
    //  play for a while then ask us to shutdown.
    //
    while (1)
    {
        // Check for a sync request
        Sync();

        // If we get an exit signal, then exit
        if (bCheckShutdownRequest())
            break;

        // Generate a semi random number between 500 and 2000
        tCIDLib::TCard4 c4Value = 0;
        while (c4Value < 500)
            c4Value = TMathLib::c4RandomNum() % 2000;

        // Lets sleep for this long
        TThread::Sleep(c4Value);

        //
        //  Now we need to create a new random number and see if this value is
        //  in the bag. If not, we need to add it. For this, we need to lock
        //  the collection because this requires multiple operations.
        //
        {
            TLocker lockCol(__pcolToUse);

            // Create a new value to add, between 0 and 500
            c4Value = TMathLib::c4RandomNum() % 500;

            //
            //  If no elements, then no need to even search. Just put the
            //  new element in if an adder. Do nothing if a remover.
            //
            //  Otherwise we have to see the element is already in the
            //  collection.
            //
            if (!__pcolToUse->c4ElemCount())
            {
                if (__eType == EThreadType_Adder)
                {
                    __pcolToUse->Add(TCardinal(c4Value));

                    // Say what we did
                    *__pstrmOut << L"Thread: " << strName() << NewLn
                                << L"        Added value: " << c4Value << NewLn; 
                }
            }
             else
            {
                //
                //  Since we have the collection locked, we can just use the
                //  internal interator.
                //
                __pcolToUse->bResetIter();
                tCIDLib::TBoolean bFound = kCIDLib::False;
                do
                {
                    if (__pcolToUse->objCur() == c4Value)
                    {
                        // If we are a remover, then remove it
                        if (__eType == EThreadType_Remover)
                        {
                            __pcolToUse->bFlushCur();

                            // Indicate what we did
                            *__pstrmOut << L"Thread: " << strName() << NewLn
                                        << L"        Removed value: "
                                        << c4Value << NewLn;
                        }
                        bFound = kCIDLib::True;
                        break;
                    }
                }   while (__pcolToUse->bNext());

                if (!bFound && (__eType == EThreadType_Adder))
                {
                    __pcolToUse->Add(TCardinal(c4Value));

                    // Say what we did
                    *__pstrmOut << L"Thread: " << strName() << NewLn
                                << L"        Added value: " << c4Value << NewLn; 
                }
            }
        }
    }

    return tCIDLib::EExit_Normal;
}
