//
//  FILE NAME: Collect3.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 08/07/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the third of the collection oriented demo
//  programs. This program demonstrates the ability of all collections to
//  be thread safe. Any collection can optionally be thread safe if that is
//  desired. Thread safe collections make all of their methods atomic that
//  need to be (i.e. the ones that are not naturally thread safe.) They can
//  also be locked (via a standard TLocker since they implement the standard
//  MLockable mixin interface) from the outside for an extended atomic
//  operation.
//
//  This program simulates a situation where there are threads that are adding
//  elements to a collection and other threads are removing them. It creates
//  a set of threads, some of which are 'adders' and some of which are 'removers'
//  and lets them add and remove elements from a sorted bag of TInteger objects.
//
//  Each thread will sleep for some random small amount of time. Then it will
//  generate a random number. If its an adder it will see if this value is 
//  already in the collection. If so, it does nothing, else it adds it. If its
//  a remover, it sees if this value is in the bag and removes it if so.
//
//  The main thread just sleeps for a while to let them do their thing, then it
//  wakes up and asks them to shutdown, waiting until they are all dead. It
//  then displays any values that survived, which should occur in sorted order
//  since the collection is a sorted bag.
//
//  This program also demonstrates how to do a TThread derivative. Threads can
//  either take a thread function (good for ad hoc threads), or you can create
//  a derivative and override the _eProcess() method. This is better for 
//  creating threads that do some canned functionality.
//
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes.
// ----------------------------------------------------------------------------
#include    "Collect3.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Collect3MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. 
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    try
    {
        //
        //  Create a sorted bag of TCardinal objects. We have to provide
        //  it with an object comparator, so provide it with the standard
        //  one which uses the <, >, and == operators. We make it a thread
        //  safe collection by passing EMTState_Safe.
        //
        TSortedBag<TCardinal> colTest
        (
            new TStdObjComp<TCardinal>
            , tCIDLib::EMTState_Safe
        );

        //
        //  Create a list of worker threads. We just create 16 of them and
        //  let them run for a while. Threads are not copyable so we have
        //  to deal with them by reference, which means that the nodes are
        //  really pointer wrapper objects. We use the counted pointer class
        //  since its convenient for this type of thing, though we really
        //  do require all of its power.
        //
        const tCIDLib::TCard4 c4ThreadCount = 16;
        typedef TCntPtr<TColThread> TColThreadPtr;
        TObjArray<TColThreadPtr> colOfThreads(c4ThreadCount);

        for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        {
            // Make most of them removers
            TColThread::EThreadTypes eType;
            if (!(c4Index % 6))
                eType = TColThread::EThreadType_Adder;
            else
                eType = TColThread::EThreadType_Remover;

            // Create the new thread and store its pointer in this node
            colOfThreads[c4Index].SetPointer
            (
                new TColThread(&colTest, eType, &__conOut)
            );
        }

        //
        //  Now start them all up. We could have started them above. But,
        //  in general, this is safer. If something happened in the loop
        //  above that caused an exception, some would have just  been
        //  started for nothing. At this point we know for sure that all
        //  of them are there and ready to rock and roll.
        //
        //  Note that, since we are using a pointer wrapper, the [] operator
        //  just gets us the wrapper object. objData() gets us a reference
        //  to the object its wrapping.
        //
        for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
            colOfThreads[c4Index].objData().Start();

        //
        //  Now lets sleep for 10 seconds and let the threads work their
        //  magic.
        //
        TThread::Sleep(10 Seconds);

        //
        //  Now lets go through the list and ask each thread to shutdown
        //  and wait for them to actually die.
        //
        __conOut << L"Asking threads to stop..." << NewLn;
        for (c4Index = 0; c4Index < c4ThreadCount; c4Index++)
        {
            // Wait for up to 5 seconds for it to die
            colOfThreads[c4Index].objData().RequestShutdown(5 Seconds);

            //
            //  And wait for it to fully die. We don't care about the
            //  return value from the thread, and we say we will wait
            //  for up to 2 seconds. This should be plenty of time since
            //  the thead is already on the way down.
            //
            colOfThreads[c4Index].objData().eWaitForDeath(2 Seconds);
        }

        //
        //  Now lets get a cursor for the collection and print out the
        //  list of elements that survived.
        //
        __conOut << L"The elements left in the collection are:" << NewLn;

        TSortedBag<TCardinal>::TCursor cursDisplay(&colTest);
        if (cursDisplay.bReset())
        {
            do
            {
                __conOut << L"   Value: " << cursDisplay.objCur() << NewLn;
            }   while (cursDisplay.bNext());
        }
         else
        {
            __conOut << L"   No elements found" << DNewLn;
        }
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
