//
//  FILE NAME: Collect2.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 07/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the second of the collection oriented demo
//  programs. This one demonstrates the hash map collection. It creates a
//  simple class, TNameInfo, which is the element type for the test. This
//  class has a name string, the key field, and a counter. It is used to
//  track how many times a name is found in a list. A simple array of strings
//  is used as test data.
//
//  The logic of the program is to test to see if the next name is in the
//  set already. If not, its added. If so, then its counter is bumped up.
//  At the end, the information is dumped out.
//
//  If this program had to deal with a huge list of names, it would still be
//  pretty efficient because the hash map is made for just this kind of
//  operation. It uses a hash table to store the elements, so a relatively
//  random set of names will create a pretty balanced set of entries in each
//  hash table slot. And the use of hashing to find the keys means that only
//  a couple of string compares is likely to occur to find any particular
//  name, even in a large list.
//
//  The hash map requires a 'key ops' object to handle hashing and comparing
//  key objects. Since our key field is a TString, we can use a canned one
//  that is provided by CIDLib, TStringKeyOps. We also have to provide a
//  small function that will pull out the key from the data element. Its
//  just done as a simple static method, __strGetKey, and passed to the
//  hash map.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Collect2MainThread", __eMainThreadFunc))


// ----------------------------------------------------------------------------
//  Local, static data
//
//  __apszNames
//      This is just a list of dummy names that serves as test data for the
//      program.
//
//  __c4NameCount
//      The count of elements in the array. We use the c4ArrayElems() macro
//      that does the usual calculation for us.
// ----------------------------------------------------------------------------
static const tCIDLib::Tch* __apszNames[] = 
{
    L"Habib, Amat"
    , L"Roddey, Dean"
    , L"Smith, John"
    , L"Wiedersatz, Michelle"
    , L"Heninger, Andy"
    , L"Werts, Michael"
    , L"Smith, John"
    , L"Clinton, Bill"
    , L"Clinton, Hillary"
    , L"Habib, Amat"
    , L"Smith, John"
    , L"Hendrix, Jimi"
    , L"Hahn, Jessie"
    , L"Joplin, Janis"
    , L"Hendrix, Jimi"
    , L"Smith, John"
};

static const tCIDLib::TCard4 __c4NameCount = c4ArrayElems(__apszNames);


// ----------------------------------------------------------------------------
//   CLASS: TNameInfo
//  PREFIX: nmi
//
//  This is a small test class that is used as the element type for our
//  hash map.
// ----------------------------------------------------------------------------
class TNameInfo : public TObject
{
    public :
        // --------------------------------------------------------------------
        //  Constructors and Destructors
        // --------------------------------------------------------------------
        TNameInfo() :
            __c4Counter(0)
        {
        }

        TNameInfo(const TString& strName) :
            __c4Counter(1)
            , __strName(strName)
        {
        }

        ~TNameInfo() {}

        // --------------------------------------------------------------------
        //  Public operators
        //
        //  The hash map requires equality operators for its elements
        // --------------------------------------------------------------------
        tCIDLib::TBoolean operator==(const TNameInfo& nmiToCompare) const
        {
            if ((__c4Counter != nmiToCompare.__c4Counter)
            ||  (__strName != nmiToCompare.__strName))
            {
                return kCIDLib::False;
            }
            return kCIDLib::True;
        }

        tCIDLib::TBoolean operator!=(const TNameInfo& nmiToCompare) const
        {
            return !operator==(nmiToCompare);
        }


        // --------------------------------------------------------------------
        //  Public, non-virtual methods
        //
        //  The objKey() methods are required by the hash map for its
        //  elements. They return the key field as a const or non-const
        //  reference.
        // --------------------------------------------------------------------
        tCIDLib::TCard4 c4Counter() const
        {
            return __c4Counter;
        }

        tCIDLib::TVoid IncCounter()
        {
            __c4Counter++;
        }

        TString& strName()
        {
            return __strName;
        }

        const TString& strName() const
        {
            return __strName;
        }


    private :
        // --------------------------------------------------------------------
        //  Private data members
        //
        //  __c4Counter
        //      This is the counter of times we've hit this name.
        //
        //  __strName
        //      This is the name that we are counting in this object.
        // --------------------------------------------------------------------
        tCIDLib::TCard4     __c4Counter;
        TString             __strName;


        // --------------------------------------------------------------------
        //  Do any needed macro stuff
        // --------------------------------------------------------------------
        RTTIMacros(TNameInfo,TObject)
};

// ----------------------------------------------------------------------------
//  Do any out of line macro stuff
// ----------------------------------------------------------------------------
RTTIData(TNameInfo,TObject)


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __strGetKey
//
// DESCRIPTION:
//
//  This is our key extraction function for the hash map. The key field is
//  inside the element object's themselves, so you have to provide a method
//  to pull it out.
// ---------------------------------------
//   INPUT: nmiSource is the source object to pull the key from.
//
//  OUTPUT: None
//
//  RETURN: A reference to the key field object.
//
static const TString& __strGetKey(const TNameInfo& nmiSource)
{
    return nmiSource.strName();
}


//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread. This thread just
//  runs through the local list of test data names, adding them to the map
//  if they are not already and bumping up their counts if so. At the end,
//   a report is generated.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    try
    {
        //
        //  Create a hash map of name info objects. The modulus divisor for
        //  the hash algorithm will be 11, which is fine for our small sample.
        //  The default maximum element count is kCIDLib::c4MaxCard which is
        //  basically unlimited. We use the standard string key ops class
        //  since it does what we want already.
        //
        THashMap<TNameInfo,TString> colOfNames
        (
            11
            , new TStringKeyOps<TString>
            , __strGetKey
        );

        //
        //  Loop through our list of names and create a name info object
        //  for each one we find. We test each one to see if its there, and
        //  add it if not. If so, we bump its counter.
        //
        for (tCIDLib::TCard4 c4Index = 0; c4Index < __c4NameCount; c4Index++)
        {
            TString strName = __apszNames[c4Index];

            // See if its in the map yet
            TNameInfo& nmiFound = colOfNames.objFindByKey(strName);

            // If not found, add it. Else bump its count
            if (!&nmiFound)
                colOfNames.Add(TNameInfo(strName));
             else
                nmiFound.IncCounter();
        }

        //
        //  Create a cursor for the map so that we can iterate it and
        //  dump out the results.
        //
        THashMap<TNameInfo,TString>::TCursor cursNames(&colOfNames);

        //
        //  Reset the cursor to get it ready for iteration. Just for paranoia
        //  check that it indicates that there are elements.
        //
        if (!cursNames.bReset())
        {
            __conOut << L"The cursor reset indicated no elements in hash map!"
                     << NewLn;
        }

        //
        //  Set up two stream format objects. One represents the name column
        //  and one represents the count column. We use these to control the
        //  output of each column.
        //
        TStreamFmt strmfCount(8, 0, tCIDLib::EHJustify_Right, kCIDLib::chSpace);
        TStreamFmt strmfName(48, 0, tCIDLib::EHJustify_Left, kCIDLib::chSpace);

        //
        //  Do a header line for the data we are going to dump out. Note
        //  that the TTextStream::ESpaces() thingie outputs 4 spaces, which
        //  are not affected by the formatting. I.e. it will just output
        //  4 spaces, regardless of the set field width. We use this to
        //  space the columns apart.
        //
        __conOut << NewLn << strmfCount << L"Count"
                 << TTextStream::ESpaces(4)
                 << strmfName << L"Name"
                 << NewLn
                 << strmfCount << L"------"
                 << TTextStream::ESpaces(4)
                 << strmfName << L"---------------------"
                 << NewLn;

        // Now iterate all the elements and dump their info out
        do
        {
            __conOut << strmfCount << cursNames.objCur().c4Counter()
                     << TTextStream::ESpaces(4)
                     << strmfName << cursNames.objCur().strName()
                     << NewLn;
           
        }   while (cursNames.bNext());
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
