//
//  FILE NAME: Collect1.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 07/24/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module for the first of the collection oriented demo
//  programs. This one just demonstrates the most basic collection, the
//  bag. It just does a directory iteration of the current directory and puts
//  the resulting find buffer objects into the bag. Then it iterates the bag
//  and dumps out the results to the standard out.
//
//  Note that there are much easier ways to do all of this stuff, via the
//  TFileSys class, but this is just for demonstration purposes.
//
//  CAVEATS/GOTCHAS:
//


// ----------------------------------------------------------------------------
//  Includes. This program is so simple that we don't even have a header of
//  our own. So just include CIDLib, which is all we need.
// ----------------------------------------------------------------------------
#include    "CIDLib.Hpp"


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
tCIDLib::EExitCodes __eMainThreadFunc
(
        TThread&            thrThis
        , tCIDLib::TVoid*   pData
);


// ----------------------------------------------------------------------------
//  Local data
//
//  __conOut
//      This is a console object which we use in this program for our standard
//      output. Its a specialized text stream class. By default consoles are
//      redirectable, but they can be forced to use the physical console. The
//      default constructor disables the interactive aspects (command recall
//      and editing), which is appropriate for this simple program.
// ----------------------------------------------------------------------------
static TConsole     __conOut;


// ----------------------------------------------------------------------------
//  Do the magic main module code
// ----------------------------------------------------------------------------
CIDLib_MainModule(TThread(L"Collect1MainThread", __eMainThreadFunc))



// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __eMainThreadFunc
//
// DESCRIPTION:
//
//  This is the the thread function for the main thread.
// ---------------------------------------
//   INPUT: thrThis is a reference to the thread instance this is the
//              function for.
//
//  OUTPUT: None
//
//  RETURN: One of the tCIDLib::EExitCodes values.
//
tCIDLib::EExitCodes __eMainThreadFunc(TThread& thrThis, tCIDLib::TVoid*)
{
    // We have to let our calling thread go first
    thrThis.Sync();

    //  Set the processes state to up and running
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Ready);

    try
    {
        //
        //  Create a bag of find buffer objects. Find buffer objects are
        //  the results of directory searches.
        //
        TBag<TFindBuf>  colOfFiles;

        // Create a directory iterator for the search
        TDirIter  diterCur;

        //
        //  Create a temporary find buffer object for use in the loop. Set
        //  up a format string for it. This tells it how to format itself
        //  when dumped to a text stream. We use one of the prefab ones
        //  that is supplies.
        //
        //  Note that the find operations below won't override the format
        //  they just fill in the file find information.
        // 
        TFindBuf fndbTmp(TFindBuf::strNameAndSize);

        //
        //  Query the current directory. We use a path string to make it
        //  easy to manipulate as a path.
        //
        TPathStr pathCurDir;
        TFileSys::QueryCurrentDir(pathCurDir);

        //
        //  Add on a search extension that will find all matches on the
        //  host platform.
        //
        pathCurDir.AddLevel(kCIDLib::pszAllFilesSpec);

        // Indicate what we are searching for
        __conOut << NewLn << L"Searching for files: " << pathCurDir << DNewLn;

        //
        //  Now start a search on the directory iterator. We use the default
        //  file attributes, which are to just find everything. That's fine
        //  for us here.
        //
        if (!diterCur.bFindFirst(pathCurDir, fndbTmp))
        {
            // If no match, then we can't do very much
            __conOut << L"There were no files found." << DNewLn;
        }

        // Continue while we are still finding matches
        do
        {
            // Add this find to the collection
            colOfFiles.Add(fndbTmp);
        }   while (diterCur.bFindNext(fndbTmp));

        //
        //  Ok, we have a bag of TFindBuf objects. So lets get a cursor for
        //  the collection, and iterate all the elements, dumping each one
        //  out to the console. We have to pass a pointer to the collection
        //  because the cursor keeps a pointer to the collection. This is a
        //  strict CIDLib convention.
        //
        TBag<TFindBuf>::TCursor cursFiles(&colOfFiles);

        //
        //  Reset the cursor to get it ready for iteration. Just for paranoia
        //  check that it indicates that there are elements.
        //
        if (!cursFiles.bReset())
        {
            __conOut << L"The cursor reset indicated no elements in bag!"
                     << NewLn;
        }

        do
        {
            __conOut << cursFiles.objCur() << NewLn;
        }   while (cursFiles.bNext());
    }

    // Catch any CIDLib runtime errors
    catch(const TError& errToCatch)
    {
        __conOut    <<  L"A CIDLib runtime error occured during processing.\r\n"
                    <<  L"Error: " << errToCatch.strErrText() << DNewLn;
        return tCIDLib::EExit_RuntimeError;
    }

    //
    //  Kernel errors should never propogate out of CIDLib, but I test
    //  for them in my demo programs so I can catch them if they do
    //  and fix them.
    //
    catch(const TKrnlError& kerrToCatch)
    {
        __conOut    << L"A kernel error occured during processing.\r\nError="
                    << kerrToCatch.errcId() << DNewLn;
        return tCIDLib::EExit_FatalError;
    }

    // Catch a general exception
    catch(...)
    {
        __conOut << L"A general exception occured during processing" << DNewLn;
        return tCIDLib::EExit_SystemException;
    }

    // Set the processes state to terminating
    TProcessRegistry::SetProcessState(tCIDLib::EProcState_Terminating);

    return tCIDLib::EExit_Normal;
}
