//
//  FILE NAME: CIDWnd_Window.Hpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/27/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the header for the CIDWnd_Window.Cpp module, which implements
//  the TWindow class. All of the window classes from CIDWnd.Lib derive from
//  this class in order to get all of the host specific capabilities.
//
//  We provide methods that can be called to manipulate the window, and we
//  provide a set of protected virtual methods via which events are passed to
//  the window (and which can be overridden by the derivatives of TWindow in
//  order to handle the events.)
//
//  CAVEATS/GOTCHAS:
//


#pragma pack(push, CIDLIBPACK)


// -----------------------------------------------------------------------------
//  CLASS: TWindow
// PREFIX: wnd
// -----------------------------------------------------------------------------
class CIDWNDEXP TWindow : public TObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and destructors
        // ---------------------------------------------------------------------
        ~TWindow();


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TBoolean bMatchesMnemonic
        (
            const   tCIDLib::Tch            chToCheck
        );

        virtual tCIDLib::TVoid QueryColors();

        virtual const TRGBClr& rgbActiveBorder() const;

        virtual const TRGBClr& rgbBgn() const;

        virtual const TRGBClr& rgbBorder() const;

        virtual const TRGBClr& rgbBorderContrast() const;

        virtual const TRGBClr& rgbInactiveBorder() const;

        virtual const TRGBClr& rgbRaised() const;

        virtual const TRGBClr& rgbSunken() const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        TArea areaBorder() const;

        TArea areaInsideBorder() const;

        TArea areaInvalid() const;

        TArea areaWnd() const;

        TArea areaWnd
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        TArea areaWndSize() const;

        TArea areaWndSize
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDLib::TBoolean bGotFocus() const;

        tCIDLib::TBoolean bGotFocus
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDLib::TBoolean bHasABorder
        (
                    TClass&                 clsBorder = NUL_TClass
        );

        tCIDLib::TBoolean bIsActivated() const;

        tCIDLib::TBoolean bIsChild
        (
            const   TWindow* const          pwndTest
        );

        tCIDLib::TBoolean bIsEnabled() const;

        tCIDLib::TBoolean bIsEnabled
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDLib::TBoolean bIsOnTop() const;

        tCIDLib::TBoolean bIsMaximized() const;

        tCIDLib::TBoolean bIsMinimized() const;

        tCIDLib::TBoolean bIsValid() const;

        tCIDLib::TBoolean bIsVisible() const;

        tCIDLib::TBoolean bIsVisible
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDLib::TBoolean bMouseCaptured() const;

        tCIDLib::TBoolean bParentIsDesktop() const;

        tCIDLib::TBoolean bPntOverBorder
        (
            const   TPoint&                 pntToCheck
            ,       tCIDWnd::EBorderAreas&  eBorderArea
        );

        tCIDLib::TCard4 c4BorderWidth() const;

//        tCIDLib::TCard4 c4WaitMode() const;

//        tCIDLib::TCard4 c4WaitMode
//        (
//            const   tCIDLib::TBoolean       bNewState
//        );

        tCIDLib::TVoid CaptureMouse();

        tCIDLib::TVoid CenterInParent();

        tCIDLib::TVoid CenterInParent
        (
            const   tCIDWnd::TWindowId      widChild
        );

        tCIDLib::TVoid DepressBorder() const;

        tCIDLib::TVoid DestroyChild
        (
            const   tCIDWnd::TWindowId      widChild
        );

        tCIDLib::TVoid Disable();

        tCIDLib::TVoid Disable
        (
            const   tCIDWnd::TWindowId      widChild
        );

        tCIDWnd::EWndStyles eWndStyles() const;

        tCIDWnd::EWndStyles eWndStyles
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDWnd::EWndStyles eWndStyles
        (
           const    tCIDWnd::EWndStyles     eNewStyles
           , const  tCIDWnd::EWndStyles     eRelevant = tCIDWnd::EWndStyles_AllBits
        );

        tCIDWnd::EWndStyles eWndStyles
        (
           const    tCIDWnd::TWindowId      widChild
           , const  tCIDWnd::EWndStyles     eNewStyles
           , const  tCIDWnd::EWndStyles     eRelevant = tCIDWnd::EWndStyles_AllBits
        );

        tCIDLib::TVoid Enable();

        tCIDLib::TVoid Enable
        (
            const   tCIDWnd::TWindowId      widChild
        );

        tCIDLib::TVoid ForceRepaint();

        tCIDWnd::TWindowHandle hwndThis() const;

        tCIDLib::TVoid Hide();

        tCIDLib::TVoid Hide
        (
            const   tCIDWnd::TWindowId      widChild
        );

        tCIDLib::TVoid InvalidateArea
        (
            const   TArea&                  areaToInvalidate
        );

        tCIDLib::TVoid InvalidateArea
        (
            const   tCIDWnd::TWindowId      widChild
            , const TArea&                  areaToInvalidate
        );

        tCIDLib::TVoid InvalidateArea
        (
            const   TPoint&                 pntUL
            , const TPoint&                 pntLR
        );

        tCIDLib::TVoid InvalidateArea
        (
            const   tCIDWnd::TWindowId      widChild
            , const TPoint&                 pntUL
            , const TPoint&                 pntLR
        );

        tCIDLib::TVoid MoveAndSize
        (
            const   TArea&                  areaNew
        );

        TPoint pntOrg() const;

        TPoint pntOrg
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        const TWindow* pwndCastTo
        (
            const   tCIDWnd::TWindowId      widChild
            , const TClass&                 clsCastTo
            , const tCIDLib::Tch* const     pszFile
            , const tCIDLib::TCard4         c4Line
        );

        TWindow* pwndChildUnderPnt
        (
            const   TPoint&                 pntPos
        );

        TWindow* pwndDescendantWithFocus() const;

        TWindow* pwndFromId
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        const TWindow* pwndParent
        (
            const   TWindow* const          pwndNewParent
        );

        TWindow* pwndParent() const;

        TWindow* pwndTopChild() const;

        tCIDLib::TVoid RaiseBorder() const;

        tCIDLib::TVoid SetBorder
        (
                    TWndBorder* const       pwndbToAdopt
        );

        tCIDLib::TVoid SetPos
        (
            const   TPoint&                 pntNewOrg
        );

        tCIDLib::TVoid SetPos
        (
            const   tCIDWnd::TWindowId      widChild
            , const TPoint&                 pntNewOrg
        );

        tCIDLib::TVoid SetPos
        (
            const   tCIDLib::TInt4          i4X
            , const tCIDLib::TInt4          i4Y
        );

        tCIDLib::TVoid SetPos
        (
            const   tCIDWnd::TWindowId      widChild
            , const tCIDLib::TInt4          i4X
            , const tCIDLib::TInt4          i4Y
        );

        tCIDLib::TVoid SetSize
        (
            const   tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid SetSize
        (
            const   tCIDWnd::TWindowId      widChild
            , const tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid SetSize
        (
            const   TArea&                  areaSize
        );

        tCIDLib::TVoid SetSize
        (
            const   tCIDWnd::TWindowId      widChild
            , const TArea&                  areaSize
        );

        tCIDLib::TVoid SetSize
        (
            const   TSize&                  szNew
        );

        tCIDLib::TVoid SetSize
        (
            const   tCIDWnd::TWindowId      widChild
            , const TSize&                  szNew
        );

        tCIDLib::TVoid SetSizePos
        (
            const   TArea&                  areaNew
        );

        tCIDLib::TVoid SetSizePos
        (
            const   tCIDWnd::TWindowId      widChild
            , const TArea&                  areaNew
        );

        tCIDLib::TVoid SetSizePos
        (
            const   TPoint&                 pntOrg
            , const tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid SetSizePos
        (
            const   tCIDWnd::TWindowId      widChild
            , const TPoint&                 pntOrg
            , const tCIDLib::TCard4         c4CX
            , const tCIDLib::TCard4         c4CY
        );

        tCIDLib::TVoid Show
        (
            const   tCIDWnd::EShowStates    eShowState
        );

        tCIDLib::TVoid SizeTo
        (
            const   TSize&                  szNew
        );

        const TString& strFlyoverText() const;

        const TString& strFlyoverText
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        const TString& strFlyoverText
        (
            const   TString&                strNewText
        );

        const TString& strFlyoverText
        (
            const   tCIDWnd::TWindowId      widChild
            , const TString&                strNewText
        );

        const TString& strText() const;

        const TString& strText
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        const TString& strText
        (
            const   TString&                strNewText
        );

        const TString& strText
        (
            const   tCIDWnd::TWindowId      widChild
            , const TString&                strNewText
        );

        tCIDLib::TVoid ValidateArea
        (
            const   TArea&                  areaToValidate
        );

        tCIDWnd::TWindowId widThis() const;


    protected   :
        // ---------------------------------------------------------------------
        //  This is an internal structure used to pass needed information to
        //  the new window. It will change according to platform.
        // ---------------------------------------------------------------------
        struct TWndData
        {
            tCIDLib::TCard2 c2Size;
            TWindow*        pwndThis;
        };


        // ---------------------------------------------------------------------
        //  Protected constructors
        // ---------------------------------------------------------------------
        TWindow
        (
            const   tCIDWnd::TWindowHandle  hwndToUse
            ,       TWndBorder* const       pwndBorderToAdopt = 0
            , const tCIDLib::EAdoptOpts     eAdopt = tCIDLib::EAdoptOpt_Adopt
        );

        TWindow
        (
            const   tCIDWnd::TWindowId      widToUse
            ,       TWndBorder* const       pwndBorderToAdopt = 0
        );


        // ---------------------------------------------------------------------
        //  Protected, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TVoid _ActivationChanged
        (
            const   tCIDLib::TBoolean       bNewState
        );

        virtual TArea _areaBorder() const;

        virtual tCIDLib::TBoolean _bQuerySizeLimits
        (
                    TArea&                  areaMin
            ,       TArea&                  areaMax
        );

        virtual tCIDLib::TBoolean _bSetPointer();

        virtual tCIDLib::TVoid _BorderChanged
        (
            const   TWndBorder* const       pwndbNew
            , const TWndBorder* const       pwndbOld
        );

        virtual tCIDLib::TVoid _Created
        (
            const   TArea&                  areaInit
        );

        virtual tCIDLib::TVoid _Destroy();

        virtual tCIDWnd::EEventResponses _eClick
        (
            const   tCIDWnd::EMouseButtons  eButton
            , const tCIDWnd::EMouseClicks   eClickType
            , const TPoint&                 pntAt
            , const tCIDLib::TBoolean       bControlDown
            , const tCIDLib::TBoolean       bShiftDown
        );

        virtual tCIDLib::TVoid _EnableStateChanged
        (
            const   tCIDLib::TBoolean       bNewState
        );

        virtual tCIDLib::TVoid _EndSizeMove();

        virtual tCIDLib::TVoid _EraseBgn
        (
                    TGraphicDevice&         gdevToUse
            , const TArea&                  areaUpdate
        );

        virtual tCIDLib::TVoid _LostCapture();

        virtual tCIDLib::TVoid _MouseMove
        (
            const   TPoint&                 pntAt
            , const tCIDLib::TBoolean       bControlDown
            , const tCIDLib::TBoolean       bShiftDown
        );

        tCIDLib::TVoid _MoveComplete
        (
            const   TPoint&                 pntEndPos
        );

        virtual tCIDLib::TVoid _Moving
        (
            const   TArea&                  areaCurrent
        );

        virtual tCIDLib::TVoid _Paint
        (
                    TGraphicDevice&         gdevToUse
            , const TArea&                  areaUpdate
        );

        virtual tCIDLib::TVoid _SizeChanged
        (
            const   TArea&                  areaNewSize
            , const TArea&                  areaOldSize
        );

        virtual tCIDLib::TVoid _StartSizeMove();

        virtual tCIDLib::TVoid _VisibilityChanged
        (
            const   tCIDLib::TBoolean       bNewState
        );

        virtual tCIDLib::TVoid _WndStylesChanged
        (
            const   tCIDWnd::EWndStyles     eNewStyles
            , const tCIDWnd::EWndStyles     eOldStyle
        );

        virtual tCIDLib::TVoid _WndTextChanged
        (
            const   TString&                strNewText
            , const TString&                strOldText
        );


        // ---------------------------------------------------------------------
        //  Protected, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _Create
        (
            const   TWindow&                wndParent
            , const TString&                strText
            , const tCIDWnd::EWndStyles     eStyles
            , const TArea&                  areaInit = NUL_TArea
            , const tCIDLib::TCard4         c4ExtraHostStyles = 0
            , const TString&                strClass = NUL_TString
        );

        TWndClr& _wclrBgn();

        TWndClr& _wclrActiveBorder();

        TWndClr& _wclrBorderContrast();

        TWndClr& _wclrInactiveBorder();

        TWndClr& _wclrRaised();

        TWndClr& _wclrSunken();


    private :
        // ---------------------------------------------------------------------
        //  Declare our friends
        // ---------------------------------------------------------------------
        friend static tCIDWnd::TMsgResult WNDPROCAPI __mresDispatch
        (
            tCIDWnd::TWindowHandle  hwndThis
            , tCIDLib::TUInt        uMsg
            , tCIDLib::TUInt        uParam
            , tCIDLib::TInt4        i4Param
        );

        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TWindow();

        TWindow(const TWindow&);

        tCIDLib::TVoid operator=(const TWindow&);


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDWnd::TWindowHandle __hwndGetChild
        (
            const   tCIDWnd::TWindowId      widChild
            , const tCIDLib::TCard4         c4Line
        )   const;

        tCIDWnd::TWindowHandle __hwndGetChild
        (
            const   tCIDWnd::TWindowId      widChild
        )   const;

        tCIDLib::TVoid __QueryInitialState();


        // ---------------------------------------------------------------------
        //  Private, static methods
        // ---------------------------------------------------------------------
        static tCIDLib::TCard4 __c4MapStyles
        (
            const   tCIDWnd::EWndStyles     eStyles
        );

        static tCIDLib::TCard4 __c4MapExStyles
        (
            const   tCIDWnd::EWndStyles     eStyles
        );

        static tCIDLib::TVoid __InsertPtr
        (
            const   tCIDWnd::TWindowHandle  hwndTarget
            ,       TWindow* const          pwndToInsert
        );

        static TWindow* __pwndExtractPtr
        (
            const   tCIDWnd::TWindowHandle  hwndSource
        );


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __bDestructing
        //      This is set by the destructor, so that the processing for
        //      the destruction message won't turn around and try to destroy
        //      the window again.
        //
        //  __bIsActive
        //      Indicates whether the window is currently active or not. It
        //      is maintained by activation events. It is set/cleared before
        //      the ActivationChange() event method is called.
        //
        //  __bEnabled
        //      The enabled state of the window. This is kept up to date
        //      by the event handler method internally.
        //
        //  __bMouseCaptured
        //      This is the capture state of the mouse. This is so common a
        //      thing, that support for keeping this flag is provided at
        //      this fundamental.
        //
        //  __eAdopt
        //      This indicates whether we own the window handle or not. If
        //      not, then we don't destroy it when we die. If we do, then we
        //      destroy it. Wrapper objects for system defined windows are
        //      the type that would not own the handle.
        //
        //  __hwndThis
        //      This is our window handle. This class is basically just a
        //      wrapper for this handle.
        //
        //  __pwndbThis
        //      This is the border object installed on the window. If it
        //      is installed, then the default version of _Paint() will
        //      draw it.
        //
        //  __strFlyoverText
        //      This is the status bar text for this window. When the most
        //      moves over this window, its status bar text will be sent to
        //      the first direct frame parent of this window. Defaults to
        //      and empty string.
        //
        //  __strWndText
        //      We store the window text here.
        //
        //  __wclrBgn
        //      Used to fill the invalid area in our default _Paint()
        //      processing.
        //
        //  __wclrActiveBorder
        //  __wclrInactiveBorder
        //      These are used for the active and inactive border colors for
        //      solid type borders. A single line of the contrast color is
        //      used on the inside and outside of them.
        //
        //  __wclrBorderContrast
        //      This is the color of the contrast line that is used around
        //      the outside and inside of some border types. This provides
        //      contrast against what they are put against.
        //
        //  __wclrRaised
        //  __wclrSunken
        //      The colors used for 3D border effects.
        //
        //  __widThis
        //      This is the id of this window object.
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean       __bDestructing;
        tCIDLib::TBoolean       __bEnabled;
        tCIDLib::TBoolean       __bMouseCaptured;
        tCIDLib::TBoolean       __bIsActive;
        tCIDLib::EAdoptOpts     __eAdopt;
        tCIDWnd::EWndStyles     __eStyles;
        tCIDWnd::TWindowHandle  __hwndThis;
        TWndBorder*             __pwndbThis;
        TString                 __strFlyoverText;
        TString                 __strWndText;
        TWndClr                 __wclrBgn;
        TWndClr                 __wclrActiveBorder;
        TWndClr                 __wclrBorderContrast;
        TWndClr                 __wclrInactiveBorder;
        TWndClr                 __wclrRaised;
        TWndClr                 __wclrSunken;
        tCIDWnd::TWindowId      __widThis;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TWindow,TObject)
};

#pragma pack(pop)


// -----------------------------------------------------------------------------
//  TWindow: Public, virtual methods
// -----------------------------------------------------------------------------

inline const TRGBClr& TWindow::rgbActiveBorder() const
{
    return __wclrActiveBorder.rgbColor();
}

inline const TRGBClr& TWindow::rgbBgn() const
{
    return __wclrBgn.rgbColor();
}

inline const TRGBClr& TWindow::rgbBorderContrast() const
{
    return __wclrBorderContrast.rgbColor();
}

inline const TRGBClr& TWindow::rgbInactiveBorder() const
{
    return __wclrInactiveBorder.rgbColor();
}

inline const TRGBClr& TWindow::rgbRaised() const
{
    return __wclrRaised.rgbColor();
}

inline const TRGBClr& TWindow::rgbSunken() const
{
    return __wclrSunken.rgbColor();
}


// -----------------------------------------------------------------------------
//  TWindow: Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TWindow::bIsActivated() const
{
    return __bIsActive;
}

inline tCIDLib::TBoolean TWindow::bMouseCaptured() const
{
    return __bMouseCaptured;
}

inline tCIDWnd::TWindowHandle TWindow::hwndThis() const
{
    return __hwndThis;
}

inline tCIDLib::TVoid TWindow::DepressBorder() const
{
    if (__pwndbThis)
        __pwndbThis->Depress();
}

inline tCIDLib::TVoid
TWindow::InvalidateArea(const TPoint& pntUL, const TPoint& pntLR)
{
    InvalidateArea(TArea(pntUL, pntLR));
}

inline tCIDLib::TVoid
TWindow::InvalidateArea(const   tCIDWnd::TWindowId  widChild
                        , const TPoint&             pntUL
                        , const TPoint&             pntLR)
{
    InvalidateArea(widChild, TArea(pntUL, pntLR));
}

inline tCIDLib::TVoid TWindow::RaiseBorder() const
{
    if (__pwndbThis)
        __pwndbThis->Raise();
}

inline tCIDLib::TVoid TWindow::SetPos(  const   tCIDLib::TInt4  i4X
                                        , const tCIDLib::TInt4  i4Y)
{
    SetPos(TPoint(i4X, i4Y));
}

inline tCIDLib::TVoid TWindow::SetPos(  const   tCIDWnd::TWindowId  widChild
                                        , const tCIDLib::TInt4      i4X
                                        , const tCIDLib::TInt4      i4Y)
{
    SetPos(widChild, TPoint(i4X, i4Y));
}

inline tCIDLib::TVoid TWindow::SetSize(const TArea& areaSize)
{
    SetSize(areaSize.c4Width(), areaSize.c4Height());
}

inline tCIDLib::TVoid TWindow::SetSize( const   tCIDWnd::TWindowId  widChild
                                        , const TArea&              areaSize)
{
    SetSize(widChild, areaSize.c4Width(), areaSize.c4Height());
}

inline tCIDLib::TVoid TWindow::SetSize(const TSize& szNew)
{
    SetSize(szNew.c4Width(), szNew.c4Height());
}

inline tCIDLib::TVoid TWindow::SetSize( const   tCIDWnd::TWindowId  widChild
                                        , const TSize&              szNew)
{
    SetSize(widChild, szNew.c4Width(), szNew.c4Height());
}

inline tCIDLib::TVoid
TWindow::SetSizePos(const   TPoint&                 pntOrg
                    , const tCIDLib::TCard4         c4CX
                    , const tCIDLib::TCard4         c4CY)
{
    SetSizePos(TArea(pntOrg, c4CX, c4CY));
}

inline tCIDLib::TVoid
TWindow::SetSizePos(const   tCIDWnd::TWindowId      widChild
                    , const TPoint&                 pntOrg
                    , const tCIDLib::TCard4         c4CX
                    , const tCIDLib::TCard4         c4CY)
{
    SetSizePos(widChild, TArea(pntOrg, c4CX, c4CY));
}

inline tCIDWnd::TWindowId TWindow::widThis() const
{
    return __widThis;
}


// -----------------------------------------------------------------------------
//  TWindow: Protected, non-virtual methods
// -----------------------------------------------------------------------------
inline TWndClr& TWindow::_wclrBgn()
{
    return __wclrBgn;
}

inline TWndClr& TWindow::_wclrActiveBorder()
{
    return __wclrActiveBorder;
}

inline TWndClr& TWindow::_wclrBorderContrast()
{
    return __wclrBorderContrast;
}

inline TWndClr& TWindow::_wclrInactiveBorder()
{
    return __wclrInactiveBorder;
}

inline TWndClr& TWindow::_wclrRaised()
{
    return __wclrRaised;
}

inline TWndClr& TWindow::_wclrSunken()
{
    return __wclrSunken;
}


// -----------------------------------------------------------------------------
//  A very useful way to cast a TWindow to what it really is. It handles
//  checking (in debug builds) that the cast is valid.
// -----------------------------------------------------------------------------
template <class T> inline TWindow*
pwndCastChild(const TWindow& wndParent, const tCIDWnd::TWindowId widChild)
{
    #if CID_DEBUG_ON
    return wndParent.pwndCastTo(widChild, T::clsThis, __FILE__, __LINE__);
    #else
    return (T*)wndParent.pwndFromId(widChild);
    #endif
}
