//
//  FILE NAME: CIDWnd_Window.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/27/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TWindow class, which is the host abstraction
//  class for windows. All higher level window classes derive from this
//  class.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDWnd_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TWindow,TObject)



// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
static tCIDWnd::TMsgResult WNDPROCAPI __mresDispatch
(
    tCIDWnd::TWindowHandle  hwndThis
    , tCIDLib::TUInt        uMsg
    , tCIDLib::TUInt        uParam
    , tCIDLib::TInt4        i4Param
);



// -----------------------------------------------------------------------------
//  Local functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __mresDispatch
//
// DESCRIPTION:
//
//  This is the internal window dispatch routine. All windows created via
//  CIDLib are given this window function. This function knows how to
//  get the TWindow object from the window, translate the message to
//  the canonical format, and dispatch it to the window.
//
//  The window data for all CIDLib programs is the window function object
//  for this window. We store it on the window and extract it any time that
//  we need to dispatch a message.
// ---------------------------------------
//   INPUT: 
//
//  OUTPUT: None
//
//  RETURN: 
//
static tCIDWnd::TMsgResult WNDPROCAPI
__mresDispatch( tCIDWnd::TWindowHandle  hwndThis
                , tCIDLib::TUInt        uMsg
                , tCIDLib::TUInt        uParam
                , tCIDLib::TInt4        i4Param)
{
    //
    //  Check specially for the create message. This gets us the window
    //  object that we need. We store it on the window.
    //
    if (uMsg == WM_NCCREATE)
    {
        DefWindowProc(hwndThis, uMsg, uParam, i4Param);

        // Get the pointer to the create data
        const CREATESTRUCT* const pCreate = (CREATESTRUCT*)i4Param;

        // get the window creation data out, which is our needed info
        struct TWindow::TWndData UNALIGNED* pData
                                = (TWindow::TWndData*)pCreate->lpCreateParams;
        //
        //  Put our window handle into the window object. It will be set
        //  again when the create returns, but this gets it in early as
        //  possible. We are a friend so we have direct access to the
        //  window data members.
        //
        pData->pwndThis->__hwndThis = hwndThis;

        // Query any initial state that we need to get
        pData->pwndThis->__QueryInitialState();

        // Store the pointer to the window function on our window
        TWindow::__InsertPtr(hwndThis, pData->pwndThis);
        if (!TKrnlThread::errcGetLast())
            return 1;
    }

    // Get out the window object
    TWindow* pwndThis = TWindow::__pwndExtractPtr(hwndThis);

    // If it has not gotten set yet, then just send to default handler
    if (!pwndThis)
        return DefWindowProc(hwndThis, uMsg, uParam, i4Param);

    switch(uMsg)
    {
        //
        //  The activation state of the window is changing, so tell it 
        //  what its new state is.
        //
        case WM_ACTIVATE :
        case WM_ACTIVATEAPP :
        {
            // Get the new activation state
            tCIDLib::TBoolean bNewState;
            if (uMsg == WM_ACTIVATE)
                bNewState = (TRawBits::c2Low16From32(uParam) != WA_INACTIVE);
            else
                bNewState = (uParam != WA_INACTIVE);

            // First update the activation member of the window
            pwndThis->__bIsActive = bNewState;

            // Then trigger the activation change event
            pwndThis->_ActivationChanged(bNewState);
        }
        break;

        //
        //  The window is being created, so tell it where its initial
        //  position is.
        //
        case WM_CREATE :
        {
            const CREATESTRUCT* const pCreate = (CREATESTRUCT*)i4Param;

            // Convert the passed create size to an area structure
            TArea areaInit
            (
                pCreate->x
                , pCreate->y
                , pCreate->cx
                , pCreate->cy
            );

            //
            //  Now we can call the window function. We call its special init
            //  function.
            //
            pwndThis->_Created(areaInit);
            return 1;
        }
        break;

        //
        //  The window is being destroyed so it needs to clean up any
        //  data it has.
        //
        case WM_DESTROY :
        {
            // Let all of the derived classes of this window cleanup
            pwndThis->_Destroy();

            //
            //  If the __bDestructing flag is not set, then we are being
            //  destroyed via the system's mechanism, not by way of our
            //  window object being deleted. So we need to kick off the
            //  destructor of the window object.
            //
            if (!pwndThis->__bDestructing)
            {
                //
                //  Invalidate the handle so the destructor won't try to
                //  destroy the window again.
                //
                pwndThis->__hwndThis = kCIDWnd::hwndInvalid;
                delete pwndThis;
            }
        }
        break;

        // The enabled state of this window has changed
        case WM_ENABLE :
        {
            // Update our internal status
            pwndThis->__bEnabled = (uParam != 0);

            // And call the event method
            pwndThis->_EnableStateChanged(pwndThis->__bEnabled);
        }
        break;

        // A sizing or moving event is about to start
        case WM_ENTERSIZEMOVE :
        {
            pwndThis->_StartSizeMove();
        }
        break;

        // The window needs to erase its background
        case WM_ERASEBKGND :
        {
            // Just an empty default implementation
            tCIDLib::THostRectl rectlUpdate;
            if (GetUpdateRect(hwndThis, (RECT*)&rectlUpdate, FALSE))
            {
                TArea areaUpdate(rectlUpdate, tCIDLib::ERectl_Inclusive);
                TGraphicDevice gdevErase
                (
                    tCIDWnd::TDeviceHandle(uParam)
                    , TGraphicDevice::EDevType_Paint
                    , tCIDLib::EAdoptOpt_NoAdopt
                );
                pwndThis->_EraseBgn(gdevErase, areaUpdate);
                return 1;
            }
        }
        break;

        // A sizing/moving event has ended
        case WM_EXITSIZEMOVE :
        {
            pwndThis->_EndSizeMove();
        }
        break;

        //
        //  Ask the window object what the minimum and maximum sizes it wants
        //  to be sized to are.
        //
        case WM_GETMINMAXINFO :
        {
            TArea areaMin;
            TArea areaMax;

            //
            //  If the overrider honors this event, then we need to put their
            //  returned min/max info into the system's passed structure.
            //
            if (pwndThis->_bQuerySizeLimits(areaMin, areaMax))
            {
                MINMAXINFO* pInfo = (MINMAXINFO*)i4Param;

                pInfo->ptMinTrackSize.x = areaMin.c4Width();
                pInfo->ptMinTrackSize.y = areaMin.c4Height();
                pInfo->ptMaxTrackSize.x = areaMax.c4Width();
                pInfo->ptMaxTrackSize.y = areaMax.c4Height();
            }
        }
        break;

        //
        //  A button click occured. So figure out which one and what kind
        //  and pass along the shift keys pressed at the time.
        //
        case WM_LBUTTONDBLCLK :
        case WM_LBUTTONDOWN :
        case WM_LBUTTONUP :
        case WM_MBUTTONDBLCLK :
        case WM_MBUTTONDOWN :
        case WM_MBUTTONUP :
        case WM_RBUTTONDBLCLK :
        case WM_RBUTTONDOWN :
        case WM_RBUTTONUP :
        {
            tCIDWnd::EMouseButtons eButton = tCIDWnd::EMouseButton_Left;
            if ((uMsg == WM_MBUTTONDBLCLK)
            ||  (uMsg == WM_MBUTTONUP)
            ||  (uMsg == WM_MBUTTONDOWN))
            {
                eButton = tCIDWnd::EMouseButton_Middle;
            }
             else if ((uMsg == WM_RBUTTONDBLCLK)
                  ||  (uMsg == WM_RBUTTONUP)
                  ||  (uMsg == WM_RBUTTONDOWN))
            {
                eButton = tCIDWnd::EMouseButton_Right;
            }

            tCIDWnd::EMouseClicks eClick = tCIDWnd::EMouseClick_Down;
            if ((uMsg == WM_LBUTTONDBLCLK)
            ||  (uMsg == WM_MBUTTONDBLCLK)
            ||  (uMsg == WM_RBUTTONDBLCLK))
            {
                eClick = tCIDWnd::EMouseClick_Double;
            }
             else if ((uMsg == WM_LBUTTONUP)
                  ||  (uMsg == WM_MBUTTONUP)
                  ||  (uMsg == WM_RBUTTONUP))
            {
                eClick = tCIDWnd::EMouseClick_Up;
            }

            // Create a point that represents the click position
            TPoint pntPos
            (
                TRawBits::c2Low16From32(i4Param)
                , TRawBits::c2High16From32(i4Param)
            );

            // And call the virtual
            tCIDWnd::EEventResponses eResponse = pwndThis->_eClick
            (
                eButton
                , eClick
                , pntPos
                , (uParam & MK_CONTROL) != 0
                , (uParam & MK_SHIFT) != 0
            );

            if (eResponse == tCIDWnd::EEventResponse_Propogate)
                return 1;
        }
        break;

        //
        //  The move is moving. Pass along the new mouse position and whether
        //  either of the control or shift keys was down.
        //
        case WM_MOUSEMOVE :
        {
            // Get the point information out to pass on
            TPoint pntPos
            (
                TRawBits::c2Low16From32(i4Param)
                , TRawBits::c2High16From32(i4Param)
            );

            pwndThis->_MouseMove
            (
                pntPos
                , (uParam & MK_CONTROL) != 0
                , (uParam & MK_SHIFT) != 0
            );
        }
        break;

        // 
        //  A move has just completed. Tell the window object where it
        //  ended up.
        //
        case WM_MOVE :
        {
            TPoint pntPos
            (
                TRawBits::c2Low16From32(i4Param)
                , TRawBits::c2High16From32(i4Param)
            );
            pwndThis->_MoveComplete(pntPos);
        }
        break;

        //
        //  The window is being moved. Tell the window object where it is
        //  right now.
        //
        case WM_MOVING :
        {
            // Construct the current area of the move
            TArea areaPos
            (
                *(tCIDLib::THostRectl*)i4Param
                , tCIDLib::ERectl_Inclusive
            );
            pwndThis->_Moving(areaPos);
        }
        break;

        // We need to paint ourselve
        case WM_PAINT :
        {
            tCIDLib::THostRectl rectlUpdate;
            if (!GetUpdateRect(hwndThis, (RECT*)&rectlUpdate, FALSE))
                return 0;

            PAINTSTRUCT PaintStruct;
            HDC hdcToUse = BeginPaint(hwndThis, &PaintStruct);
            if (!hdcToUse)
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcWnd_BeginPaint
                    , kerrToLog
                    , tCIDLib::ESev_ProcessFatal
                    , tCIDLib::EClass_Internal
                );
            }

            // Convert the update rect to an area structure
            TArea areaUpdate(rectlUpdate, tCIDLib::ERectl_Inclusive);

            //
            //  Create a device object for the device. We indicate that its
            //  a paint device and that the object does not adopt it since
            //  we have to clean it up.
            //
            TGraphicDevice gdevPaint(hdcToUse, PaintStruct);

            // Call the window's painter
            pwndThis->_Paint(gdevPaint, areaUpdate);
        }
        break;


        // Handle a set of the cursor
        case WM_SETCURSOR :
        {
            //
            //  Pass it on to the window. If it does not handle it, then
            //  we let it go on to the default handler who will propogate
            //  it up the parent chain.
            //
            if (!pwndThis->_bSetPointer())
                return DefWindowProc(hwndThis, uMsg, uParam, i4Param);
        }       
        break;

        // The visibility state has changed
        case WM_SHOWWINDOW :
        {
            // Pass along the new visibility state flag
            pwndThis->_VisibilityChanged(uParam != 0);
        }
        break;

        default :
            return DefWindowProc(hwndThis, uMsg, uParam, i4Param);
    }
    return 0;
}


// -----------------------------------------------------------------------------
//  Intrafacility functions
// -----------------------------------------------------------------------------

tCIDLib::TVoid _InitTermWindow( const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
    &&  (eGlobals == tCIDLib::EGlobalState_Before))
    {
        //
        //  Register our window class. This is the one and only class we
        //  need. It insures that there is a space for the pointer to the
        //  associated window object.
        //
        WNDCLASSEX  wcData = {0};
        wcData.cbSize           = sizeof(WNDCLASSEX);
        wcData.style            = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
        wcData.lpfnWndProc      = __mresDispatch;
        wcData.cbWndExtra       = sizeof(tCIDLib::TVoid*);
        wcData.hInstance        = modInit.kmodThis().hmodThis();
        wcData.hCursor          = LoadCursor(NULL, MAKEINTRESOURCE(IDI_APPLICATION));
        wcData.lpszClassName    = kCIDWnd_::pszCIDLibClass;
//        wcData.hbrBackground    = (HBRUSH)(COLOR_WINDOW+1);

	    if (!RegisterClassEx(&wcData))
        {
            TRawStr::CopyStr
            (
                pszFailReason
                , L"Could not register window class"
                , c4MaxChars
            );
            throw TKrnlError();
        }
    }
}


// -----------------------------------------------------------------------------
//  Global functions
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDWnd_InitShowState
//
// DESCRIPTION:
//
//  This method is called from the magic startup code for a GUI app. It stores
//  away the initial show state that was provided to the program entry point.
//  this is used in the initial display of the first window created.
// ---------------------------------------
//   INPUT: sShowState is the initial show state passed to the program
//              entry point.
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid CIDWnd_InitShowState(const tCIDLib::TSInt sShowState)
{
    // <TBD> Store this and use it as needed
}


// -----------------------------------------------------------------------------
//   CLASS: TWindow
//  PREFIX: wnd
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TWindow: Constructors and destructors
// -----------------------------------------------------------------------------

TWindow::TWindow(   const   tCIDWnd::TWindowHandle  hwndToUse
                    ,       TWndBorder* const       pwndBorderToAdopt
                    , const tCIDLib::EAdoptOpts     eAdopt) :

    __bDestructing(kCIDLib::False)
    , __bEnabled(kCIDLib::False)
    , __bIsActive(kCIDLib::False)
    , __bMouseCaptured(kCIDLib::False)
    , __eAdopt(eAdopt)
    , __eStyles(tCIDWnd::EWndStyle_None)
    , __hwndThis(hwndToUse)
    , __pwndbThis(pwndBorderToAdopt)
    , __wclrBgn(tCIDLib::ESysClr_Background)
    , __wclrActiveBorder(tCIDLib::ESysClr_ActiveBorder)
    , __wclrBorderContrast(tCIDLib::ESysClr_BorderContrast)
    , __wclrInactiveBorder(tCIDLib::ESysClr_InactiveBorder)
    , __wclrRaised(tCIDLib::ESysClr_3DLight)
    , __wclrSunken(tCIDLib::ESysClr_3DDark)
    , __widThis(kCIDWnd::widInvalid)
{
    // Get the id of the window
    __widThis = GetWindowLong(hwndToUse, GWL_ID);

    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_WindowCount, 1);
}

TWindow::TWindow(   const   tCIDWnd::TWindowId  widToUse
                    ,       TWndBorder* const   pwndBorderToAdopt) :

    __bDestructing(kCIDLib::False)
    , __bEnabled(kCIDLib::False)
    , __bIsActive(kCIDLib::False)
    , __bMouseCaptured(kCIDLib::False)
    , __eAdopt(tCIDLib::EAdoptOpt_NoAdopt)
    , __eStyles(tCIDWnd::EWndStyle_None)
    , __hwndThis(kCIDWnd::hwndInvalid)
    , __pwndbThis(pwndBorderToAdopt)
    , __wclrBgn(tCIDLib::ESysClr_Background)
    , __wclrActiveBorder(tCIDLib::ESysClr_ActiveBorder)
    , __wclrBorderContrast(tCIDLib::ESysClr_BorderContrast)
    , __wclrInactiveBorder(tCIDLib::ESysClr_InactiveBorder)
    , __wclrRaised(tCIDLib::ESysClr_3DLight)
    , __wclrSunken(tCIDLib::ESysClr_3DDark)
    , __widThis(widToUse)
{
    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_WindowCount, 1);
}

TWindow::~TWindow()
{
    // If the handle is filled in then destroy it.
    if (__hwndThis != kCIDWnd::hwndInvalid)
    {
        //
        //  We need to destroy the window, but make sure that the WM_DESTROY
        //  message knows that it does not have to in turn call the
        //  destructor. So, set the __bDestructing flag, then destroy the
        //  window.
        //
        __bDestructing = kCIDLib::True;
        if (!DestroyWindow(__hwndThis))
        {
            // <TBD> Log a warning error here
        }

        // Now we can waste the handle
        __hwndThis = kCIDWnd::hwndInvalid;
    }

    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_WindowCount, -1);
    delete __pwndbThis;
}


// -----------------------------------------------------------------------------
//  TWindow: Public, non-virtual methods
// -----------------------------------------------------------------------------

TArea TWindow::areaBorder() const
{
    return _areaBorder();
}


TArea TWindow::areaInvalid() const
{
    tCIDLib::THostRectl  rectlInvalid;

    if (!GetUpdateRect(__hwndThis, (RECT*)&rectlInvalid, FALSE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_InvalidArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    return TArea(rectlInvalid, tCIDLib::ERectl_Inclusive);
}


TArea TWindow::areaWnd() const
{
    tCIDLib::THostRectl rectlArea;
    if (!GetWindowRect(__hwndThis, (RECT*)&rectlArea))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_QueryArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    return TArea(rectlArea, tCIDLib::ERectl_Inclusive);
}

TArea TWindow::areaWnd(const tCIDLib::TCard4 c4ChildId) const
{
    tCIDWnd::TWindowHandle hwndChild = __hwndGetChild(c4ChildId, __LINE__);

    tCIDLib::THostRectl rectlArea;
    if (!GetWindowRect(hwndChild, (RECT*)&rectlArea))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_QueryArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    return TArea(rectlArea, tCIDLib::ERectl_Inclusive);
}


TArea TWindow::areaWndSize() const
{
    tCIDLib::THostRectl rectlArea;
    if (!GetWindowRect(__hwndThis, (RECT*)&rectlArea))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_QueryArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    TArea areaRet(rectlArea, tCIDLib::ERectl_Inclusive);
    areaRet.ZeroOrg();
    return areaRet;
}

TArea TWindow::areaWndSize(const tCIDLib::TCard4 c4ChildId) const
{
    tCIDWnd::TWindowHandle hwndChild = __hwndGetChild(c4ChildId, __LINE__);

    tCIDLib::THostRectl rectlArea;
    if (!GetWindowRect(__hwndThis, (RECT*)&rectlArea))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_QueryArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    TArea areaRet(rectlArea, tCIDLib::ERectl_Inclusive);
    areaRet.ZeroOrg();
    return areaRet;
}


tCIDLib::TBoolean TWindow::bGotFocus() const
{
    if (GetFocus() == __hwndThis)
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TWindow::bGotFocus(const tCIDWnd::TWindowId widChild) const
{
    tCIDWnd::TWindowHandle hwndChild;

    hwndChild = __hwndGetChild(widChild, __LINE__);
    if (GetFocus() == hwndChild)
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bHasABorder(TClass& clsBorder)
{
    if (__pwndbThis)
    {
        if (&clsBorder)
            clsBorder = __pwndbThis->clsIsA();

        return kCIDLib::True;
    }
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsChild(const TWindow* const pwndTest)
{
    if (IsChild(__hwndThis, pwndTest->__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsEnabled() const
{
    if (IsWindowEnabled(__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TWindow::bIsEnabled(const tCIDWnd::TWindowId widChild) const
{
    tCIDWnd::TWindowHandle hwndChild;
    hwndChild = __hwndGetChild(widChild, __LINE__);
    if (IsWindowEnabled(hwndChild))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsOnTop() const
{
    // Get the parent window
    tCIDWnd::TWindowHandle hwndParent = GetParent(__hwndThis);

    if (GetTopWindow(hwndParent) == __hwndThis)
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsMaximized() const
{
    if (IsZoomed(__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsMinimized() const
{
    if (IsIconic(__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsValid() const
{
    if (IsWindow(__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bIsVisible() const
{
    if (IsWindowVisible(__hwndThis))
        return kCIDLib::True;
    return kCIDLib::False;
}

tCIDLib::TBoolean TWindow::bIsVisible(const tCIDWnd::TWindowId widChild) const
{
    tCIDWnd::TWindowHandle hwndChild;
    hwndChild = __hwndGetChild(widChild, __LINE__);

    if (IsWindowVisible(hwndChild))
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean TWindow::bParentIsDesktop() const
{
    if (__hwndThis == GetDesktopWindow())
        return kCIDLib::True;
    return kCIDLib::False;
}


tCIDLib::TBoolean
TWindow::bPntOverBorder(const   TPoint&                 pntToCheck
                        ,       tCIDWnd::EBorderAreas&  eRegion)
{
    // Default the region
    eRegion = tCIDWnd::EBorderArea_None;

    //
    //  First quick test is to see if its outside the window altogether. If
    //  so it obviously ain't in the size border.
    //
    if ((pntToCheck.i4Y() < 0) || (pntToCheck.i4X() < 0))
        return kCIDLib::False;

    //
    //  For border detection purposes, we make sure that visually borderless
    //  windows still have a pixel sizing border.
    //
    tCIDLib::TCard4 c4Border = c4BorderWidth();
    if (!c4Border)
        c4Border = 1;

    //
    //  Get the overall window area into one area. Then make a copy of that
    //  and bump it in by the border width. Anything outside the outer
    //  area or inside the inner one is not on the border.
    //
    TArea   areaOuter(areaWndSize());
    TArea   areaInner(areaOuter);
    areaInner.Deflate(c4Border);

    if (pntToCheck.bInArea(areaInner))
        return kCIDLib::False;

    if (!pntToCheck.bInArea(areaOuter))
        return kCIDLib::False;

    //
    //  Its in the border somewhere, so figure out the particular region.
    //
    if (pntToCheck.i4X() <= areaInner.i4X())
    {
        // Its one of the left side somethings
        if (pntToCheck.i4Y() <= areaInner.i4Y())
            eRegion = tCIDWnd::EBorderArea_LowerLeft;
        else if (pntToCheck.i4Y() >= areaInner.i4Top())
            eRegion = tCIDWnd::EBorderArea_UpperLeft;
        else
            eRegion = tCIDWnd::EBorderArea_LeftSide;
    }
     else if (pntToCheck.i4X() >= areaInner.i4Right())
    {
        // Its a right side something
        if (pntToCheck.i4Y() <= areaInner.i4Y())
            eRegion = tCIDWnd::EBorderArea_LowerRight;
        else if (pntToCheck.i4Y() >= areaInner.i4Top())
            eRegion = tCIDWnd::EBorderArea_UpperRight;
        else
            eRegion = tCIDWnd::EBorderArea_RightSide;
    }
     else if (pntToCheck.i4Y() <= areaInner.i4Y())
    {
        // Its a bottom side
        eRegion = tCIDWnd::EBorderArea_Bottom;
    }
     else
    {
        // Gotta be a top side
        eRegion = tCIDWnd::EBorderArea_Top;
    }

    if (eRegion == tCIDWnd::EBorderArea_None)
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::TCard4 TWindow::c4BorderWidth() const
{
    if (!__pwndbThis)
        return 0;
    return __pwndbThis->c4Width();
}


tCIDLib::TVoid TWindow::CaptureMouse()
{
    if (!SetCapture(__hwndThis))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_MouseCapture
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    __bMouseCaptured = kCIDLib::True;
}


tCIDLib::TVoid TWindow::CenterInParent()
{
    TArea   areaMe(areaWndSize());
    areaMe.CenterIn(pwndParent()->areaWndSize());
    SetSizePos(areaMe);
}

tCIDLib::TVoid TWindow::CenterInParent(const tCIDWnd::TWindowId widChild)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->CenterInParent();
}


tCIDLib::TVoid TWindow::DestroyChild(const tCIDWnd::TWindowId widChild)
{
    TWindow* pwndChild = pwndFromId(widChild);
    delete pwndChild;
}


tCIDLib::TVoid TWindow::Disable()
{
    if (!EnableWindow(__hwndThis, 0))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_Enable
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(L"disable")
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid TWindow::Disable(const tCIDWnd::TWindowId widChild)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->Disable();
}


tCIDWnd::EWndStyles TWindow::eWndStyles() const
{
    return __eStyles;
}

tCIDWnd::EWndStyles 
TWindow::eWndStyles(const tCIDWnd::TWindowId widChild) const
{
    TWindow* pwndChild = pwndFromId(widChild);
    return pwndChild->eWndStyles();
}

tCIDWnd::EWndStyles
TWindow::eWndStyles(const   tCIDWnd::EWndStyles eNewStyles
                    , const tCIDWnd::EWndStyles eRelevant)
{
    // Get the current styles into a TCard4 for easy bit fiddlin
    tCIDLib::TCard4 c4RawStyles = __eStyles;

    // And save a copy
    tCIDWnd::EWndStyles eOldStyles = __eStyles;

    //
    //  Turn off the relevant bits. This means we only have to reset those
    //  that the user wants set. The cleared state is now a default.
    //
    c4RawStyles &= ~tCIDLib::TCard4(eRelevant);

    // And now turn on the relevant styles that are on
    c4RawStyles |= (eNewStyles & eRelevant);

    // Store this as the new styles
    __eStyles = tCIDWnd::EWndStyles(c4RawStyles);

    if (tCIDWnd::EWndStyles(c4RawStyles) != eOldStyles)
    {
        //
        //  Set the new styles on the host window. We have to convert them
        //  to the host bits first.
        //
        tCIDLib::TCard4 c4Styles = __c4MapStyles(__eStyles);
        tCIDLib::TCard4 c4ExStyles = __c4MapExStyles(__eStyles);

        // Let everyone know that they changed
        _WndStylesChanged(tCIDWnd::EWndStyles(c4RawStyles), eOldStyles);

        // Then do a size recalc and force a repaint
        _SizeChanged(areaWndSize(), areaWndSize());
        ForceRepaint();
    }

    // Return the new styles
    return __eStyles;
}

tCIDWnd::EWndStyles
TWindow::eWndStyles(const   tCIDWnd::TWindowId  widChild
                    , const tCIDWnd::EWndStyles eNewStyles
                    , const tCIDWnd::EWndStyles eRelevant)
{
    TWindow* pwndChild = pwndFromId(widChild);
    return pwndChild->eWndStyles(eNewStyles, eRelevant);
}


tCIDLib::TVoid TWindow::Enable()
{
    if (!EnableWindow(__hwndThis, 1))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_Enable
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(L"enable")
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid TWindow::Enable(const tCIDWnd::TWindowId widChild)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->Enable();
}


tCIDLib::TVoid TWindow::ForceRepaint()
{
    if (!InvalidateRect(__hwndThis, 0, 0))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_Invalidate
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TArea()
            , TCardinal(__widThis)
        );
    }
}


tCIDLib::TVoid TWindow::Hide()
{
    Show(tCIDWnd::EShowState_Hide);
}

tCIDLib::TVoid TWindow::Hide(const tCIDWnd::TWindowId widChild)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->Show(tCIDWnd::EShowState_Hide);
}


tCIDLib::TVoid TWindow::InvalidateArea(const TArea& areaToInvalidate)
{
    tCIDLib::THostRectl rectlInvalidate;
    areaToInvalidate.ToRectl(rectlInvalidate, tCIDLib::ERectl_Inclusive);
    if (!InvalidateRect(__hwndThis, (RECT*)&rectlInvalidate, kCIDLib::False))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_Invalidate
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaToInvalidate
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid
TWindow::InvalidateArea(const   tCIDWnd::TWindowId  widChild
                        , const TArea&              areaToInvalidate)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->InvalidateArea(areaToInvalidate);
}


tCIDLib::TVoid TWindow::MoveAndSize(const TArea& areaNew)
{
    if (!SetWindowPos
    (
        __hwndThis
        , 0
        , areaNew.i4X()
        , areaNew.i4Y()
        , areaNew.c4Width()
        , areaNew.c4Height()
        , SWP_NOOWNERZORDER | SWP_NOZORDER | SWP_NOREPOSITION))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_WindowPos
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaNew
            , TCardinal(__widThis)
        );
    }
}


TPoint TWindow::pntOrg() const
{
    tCIDLib::THostRectl rectlArea;
    if (!GetWindowRect(__hwndThis, (RECT*)&rectlArea))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_QueryArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TCardinal(__widThis)
        );
    }
    return TPoint(rectlArea.i4Left, rectlArea.i4Top);
}

TPoint TWindow::pntOrg(const tCIDWnd::TWindowId widChild) const
{
    TWindow* pwndChild = pwndFromId(widChild);
    return pwndChild->pntOrg();
}

// <TBD> Could this now be done with the casting template methods?
const TWindow*
TWindow::pwndCastTo(const   tCIDWnd::TWindowId  widChild
                    , const TClass&             clsCastTo
                    , const tCIDLib::Tch* const pszFile
                    , const tCIDLib::TCard4     c4Line)
{
    TWindow*  pwndChild = pwndFromId(widChild);

    if (!pwndChild)
    {
        facCIDWnd.LogErr
        (
            pszFile
            , c4Line
            , kCIDWErrs::errcWnd_NoChildWithId
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(widChild)
        );
    }

    if (!pwndChild->bIsDescendantOf(clsCastTo))
    {
        facCIDWnd.LogErr
        (
            pszFile
            , c4Line
            , kCIDWErrs::errcWnd_CastTo
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , pwndChild->clsIsA()
            , clsCastTo
        );
    }
    return pwndChild;
}


// <TBD> Update to support skipping of invisible, disabled, ect...
TWindow* TWindow::pwndChildUnderPnt(const TPoint& pntPos)
{
    tCIDWnd::TWindowHandle hwndRet;

    // Convert the point to a host point
    THostPoint ptPos;
    pntPos.ToHostPt(ptPos);

    hwndRet = ChildWindowFromPoint(__hwndThis, *(POINT*)&ptPos);

    if (!hwndRet)
        return 0;

    // Extract the window object from this window
    TWindow* pwndRet = __pwndExtractPtr(hwndRet);

    // If not one, then its not a CIDLib window
    if (!pwndRet)
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NotCIDLibWindow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(GetWindowLong(hwndRet, GWL_ID))
        );
    }
    return pwndRet;
}


TWindow* TWindow::pwndDescendantWithFocus() const
{
    HWND hwndFocus = GetFocus();

    if (!hwndFocus)
        return 0;

    if (!IsChild(__hwndThis, hwndFocus))
        return 0;

    // Extract the window object from this window
    TWindow* pwndRet = __pwndExtractPtr(hwndFocus);

    // If not one, then its not a CIDLib window
    if (!pwndRet)
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NotCIDLibWindow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(GetWindowLong(hwndFocus, GWL_ID))
        );
    }
    return pwndRet;
}


TWindow* TWindow::pwndFromId(const tCIDWnd::TWindowId widChild) const
{
    tCIDWnd::TWindowHandle hwndChild = __hwndGetChild(widChild);
    if (!hwndChild)
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NoChildWithId
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TCardinal(widChild)
        );
    }

    // Extract the window object pointer that should be there
    TWindow* pwndThis = __pwndExtractPtr(__hwndThis);

    // If not, then it ain't no CIDLib window.
    if (!pwndThis)
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NotCIDLibWindow
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , TCardinal(widChild)
        );
    }
    return pwndThis;
}


const TWindow*
TWindow::pwndParent(const TWindow* const pwndNewParent)
{
    if (!SetParent(__hwndThis, pwndNewParent->hwndThis()))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_SetParent
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return pwndNewParent;
}

TWindow* TWindow::pwndParent() const
{
    tCIDWnd::TWindowHandle hwndParent = GetParent(__hwndThis);

    if (!hwndParent)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_GetParent
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // <TBD> See if its the desktop or object window
//    if (hwndParent == GetDesktopWindow())
//        return (TWindow*)facCIDGui.pwndDesktop;

    // Not a special window, so extract the object pointer
    TWindow* pwndTmp = __pwndExtractPtr(__hwndThis);

    if (!pwndTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NotCIDLibWindow
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return pwndTmp;
}


TWindow* TWindow::pwndTopChild() const
{
    tCIDWnd::TWindowHandle hwndTop = GetWindow(__hwndThis, GW_CHILD);
    if (!hwndTop)
        return 0;

    // Extract the window object pointer out of it
    TWindow* pwndTmp = __pwndExtractPtr(__hwndThis);

    if (!pwndTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_NotCIDLibWindow
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    return pwndTmp;
}


tCIDLib::TVoid TWindow::SetBorder(TWndBorder* const pwndbNew)
{
    // Get a copy of the old border and put a janitor on it
    TWndBorder* const       pwndbOld = __pwndbThis;
    TJanitor<TWndBorder>    janBorder(pwndbOld);

    // And now set the new one
    __pwndbThis = pwndbNew;

    // If the window is not created yet, then do nothing
    if (__hwndThis == kCIDWnd::hwndInvalid)
        return;

    // Tell the world about it
    _BorderChanged(__pwndbThis, pwndbOld);

    // If the size changed, then we need to resize and redraw
    if (!pwndbOld || (__pwndbThis->c4Width() != pwndbOld->c4Width()))
    {
        _SizeChanged(areaWndSize(), areaWndSize());
        ForceRepaint();
    }
     else
    {
        TGraphicDevice ptbTemp(*this);
        __pwndbThis->DrawBorder(*this, ptbTemp, areaBorder());
    }
}


tCIDLib::TVoid TWindow::SetPos(const TPoint& pntNewOrg)
{
    if (!SetWindowPos
    (
        __hwndThis
        , kCIDWnd::hwndInvalid
        , pntNewOrg.i4X()
        , pntNewOrg.i4Y()
        , 0
        , 0
        , SWP_NOOWNERZORDER | SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_WindowPos
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , pntNewOrg
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid TWindow::SetPos( const   tCIDWnd::TWindowId  widChild
                                , const TPoint&             pntNewOrg)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->SetPos(pntNewOrg);
}

tCIDLib::TVoid
TWindow::SetSize(const tCIDLib::TCard4 c4CX, const tCIDLib::TCard4 c4CY)
{
    if (!SetWindowPos
    (
        __hwndThis
        , kCIDWnd::hwndInvalid
        , 0
        , 0
        , c4CX
        , c4CY
        , SWP_NOOWNERZORDER | SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_WindowPos
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TSize(c4CX, c4CY)
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid TWindow::SetSize(const   tCIDWnd::TWindowId  widChild
                                , const tCIDLib::TCard4     c4CX
                                , const tCIDLib::TCard4     c4CY)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->SetSize(c4CX, c4CY);
}

tCIDLib::TVoid TWindow::SetSizePos(const TArea& areaNew)
{
    if (!SetWindowPos
    (
        __hwndThis
        , kCIDWnd::hwndInvalid
        , areaNew.i4X()
        , areaNew.i4Y()
        , areaNew.c4Width()
        , areaNew.c4Height()
        , SWP_NOOWNERZORDER | SWP_NOMOVE | SWP_NOZORDER | SWP_NOACTIVATE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_WindowPos
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaNew
            , TCardinal(__widThis)
        );
    }
}

tCIDLib::TVoid
TWindow::SetSizePos(const tCIDWnd::TWindowId widChild, const TArea& areaNew)
{
    TWindow* pwndChild = pwndFromId(widChild);
    pwndChild->SetSizePos(areaNew);
}


//
// FUNCTION/METHOD NAME: Show
//
// DESCRIPTION:
//
//  This method will affect the visibility and activation state of the window,
//  and its min/max/restored state.
// ---------------------------------------
//   INPUT: eShowState is the new show state to set
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TWindow::Show(const tCIDWnd::EShowStates eShowState)
{
    // Convert our state into the the host state
    tCIDLib::TCard4 c4Flags = 0;

    if (eShowState == tCIDWnd::EShowState_Hide)
        c4Flags = SW_HIDE;
    else if (eShowState == tCIDWnd::EShowState_Maximize)
        c4Flags = SW_MAXIMIZE;
    else if (eShowState == tCIDWnd::EShowState_Minimize)
        c4Flags = SW_MINIMIZE;
    else if (eShowState == tCIDWnd::EShowState_RestoreNoActivate)
        c4Flags = SW_SHOWNOACTIVATE;
    else if (eShowState == tCIDWnd::EShowState_Show)
        c4Flags = SW_SHOW;
    else if (eShowState == tCIDWnd::EShowState_ShowMaximized)
        c4Flags = SW_SHOWMAXIMIZED;
    else if (eShowState == tCIDWnd::EShowState_ShowMinimized)
        c4Flags = SW_SHOWMINIMIZED;
    else if (eShowState == tCIDWnd::EShowState_ShowNoActivate)
        c4Flags = SW_SHOWNA;
    else if (eShowState == tCIDWnd::EShowState_ShowRestored)
        c4Flags = SW_SHOWNORMAL;
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_BadShowState
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_BadParms
            , TInteger(eShowState)
        );
    }
    ShowWindow(__hwndThis, c4Flags);
}


tCIDLib::TVoid
TWindow::SizeTo(const TSize& szNew)
{
    if (!SetWindowPos
    (
        __hwndThis
        , 0
        , 0
        , 0
        , szNew.c4Width()
        , szNew.c4Height()
        , SWP_NOOWNERZORDER | SWP_NOREPOSITION | SWP_NOZORDER | SWP_NOMOVE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_WindowPos
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , szNew
            , TCardinal(__widThis)
        );
    }
}


const TString& TWindow::strFlyoverText(const TString& strNewText)
{
    __strFlyoverText = strNewText;
    return __strFlyoverText;
}

const TString&
TWindow::strFlyoverText(const   tCIDWnd::TWindowId  widChild
                        , const TString&            strNewText)
{
    TWindow* const pwndChild = pwndFromId(widChild);
    return pwndChild->strFlyoverText(strNewText);
}


const TString& TWindow::strText() const
{
    return __strWndText;
}

const TString& TWindow::strText(const tCIDWnd::TWindowId widChild) const
{
    TWindow* const pwndChild = pwndFromId(widChild);
    return pwndChild->strText();
}

const TString& TWindow::strText(const TString& strNewText)
{
    TString  strOld(__strWndText);
    __strWndText = strNewText;
    _WndTextChanged(__strWndText, strOld);
    return __strWndText;
}

const TString& TWindow::strText(const   tCIDWnd::TWindowId  widChild
                                , const TString&            strNewText)
{
    TWindow* const pwndChild = pwndFromId(widChild);
    return pwndChild->strText(strNewText);
}


tCIDLib::TVoid TWindow::ValidateArea(const TArea& areaToValidate)
{
    tCIDLib::THostRectl rectlValidate;
    areaToValidate.ToRectl(rectlValidate, tCIDLib::ERectl_Inclusive);
    if (!ValidateRect(__hwndThis, (RECT*)&rectlValidate))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcWnd_Validate
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaToValidate
            , TCardinal(__widThis)
        );
    }
}
