//
//  FILE NAME: CIDWnd_Pen.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/27/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TPen and TGeometricPen classes. These are
//  classes that control the attributes of cosmetic and geometric lines
//  drawn onto a graphics output device.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDWnd_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TPen,TObject)
RTTIData(TPenJanitor,TObject)



// -----------------------------------------------------------------------------
//  CLASS: TPen
// PREFIX: pen
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TKrnlPen: Constructors and destructors
// -----------------------------------------------------------------------------

TPen::TPen() :

    __hpenThis(kCIDWnd::hpenInvalid)
{
    // Fill in a brush for a cosmetic line, with a solid black brush
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbColor = 0;
    BrushInfo.lbStyle = BS_SOLID;

    // Now create the pen, for solid and cosmetic
    tCIDWnd::TPenHandle hpenTmp = ExtCreatePen
    (
        PS_COSMETIC | PS_SOLID
        , 1
        , &BrushInfo
        , 0
        , 0
    );

    if (!hpenTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hpenThis = hpenTmp;

}

TPen::TPen(const TRGBClr& rgbFgn) :

    __hpenThis(kCIDWnd::hpenInvalid)
{
    // Fill in a brush for a cosmetic line, using the passed color
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbColor = rgbFgn.c4Color();
    BrushInfo.lbStyle = BS_SOLID;

    // Now create the pen, for solid and cosmetic
    tCIDWnd::TPenHandle hpenTmp = ExtCreatePen
    (
        PS_COSMETIC | PS_SOLID
        , 1
        , &BrushInfo
        , 0
        , 0
    );

    if (!hpenTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hpenThis = hpenTmp;
}

TPen::TPen(const tCIDWnd::ELineStyles eStyle, const TRGBClr& rgbFgn) :

    __hpenThis(kCIDWnd::hpenInvalid)
{
    // Fill in a brush for a cosmetic line, setting the passed color
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbColor = rgbFgn.c4Color();
    BrushInfo.lbStyle = BS_SOLID;

    //
    //  Set up styles for a cosmetic pen and put in the correct bit for
    //  the passed line style.
    //
    tCIDLib::TCard4 c4Flags = PS_COSMETIC;

    if (eStyle == tCIDWnd::ELineStyle_Solid)
        c4Flags |= PS_SOLID;
    else if (eStyle == tCIDWnd::ELineStyle_Dot)
        c4Flags |= PS_DOT;
    else if (eStyle == tCIDWnd::ELineStyle_Dash)
        c4Flags |= PS_DASH;
    else if (eStyle == tCIDWnd::ELineStyle_DashDot)
        c4Flags |= PS_DASHDOT;
    else if (eStyle == tCIDWnd::ELineStyle_DashDoubleDot)
        c4Flags |= PS_DASHDOTDOT;
    else if (eStyle == tCIDWnd::ELineStyle_Invisible)
        c4Flags |= PS_NULL;
    else
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_BadStyle
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TInteger(eStyle)
        );
    }

    // Now create the pen
    tCIDWnd::TPenHandle hpenTmp = ExtCreatePen(c4Flags, 1, &BrushInfo, 0, 0);
    if (!hpenTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hpenThis = hpenTmp;
}

TPen::TPen(const TGraphicDevice& gdevSource) :

    __hpenThis(kCIDWnd::hpenInvalid)
{
    // Get the current pen
    tCIDWnd::TPenHandle hpenTmp = (tCIDWnd::TPenHandle)GetCurrentObject
    (
        gdevSource.hdevThis()
        , OBJ_PEN
    );
    if (!hpenTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_GetPen
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Query the information for the current pen
    EXTLOGPEN PenInfo;
    if (!GetObject(hpenTmp, sizeof(PenInfo), &PenInfo))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_QueryInfo
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Fill in a brush for a cosmetic line
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbColor = PenInfo.elpColor;
    BrushInfo.lbStyle = BS_SOLID;

    // Make sure that the style is set up to be a cosmetic line.
    PenInfo.elpPenStyle = (PenInfo.elpPenStyle & ~PS_GEOMETRIC) | PS_COSMETIC;

    //
    //  Now create the pen. This is a cosmetic pen, so we ignore any stuff
    //  about geometric attributes and just set the width to 1 and the
    //  brush info to the correct values for a cosmetic line.
    //
    hpenTmp = ExtCreatePen
    (
        PenInfo.elpPenStyle
        , 1
        , &BrushInfo
        , PenInfo.elpNumEntries
        , PenInfo.elpStyleEntry
    );

    if (!hpenTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hpenThis = hpenTmp;
}

TPen::~TPen()
{
    if (__hpenThis != kCIDWnd::hpenInvalid)
    {
        if (!DeleteObject(__hpenThis))
        {
            // Log as a AppStatus so no exception is thrown out of dtor
            TKrnlError kerrToLog;
            facCIDWnd.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDWErrs::errcPen_Delete
                , kerrToLog
                , tCIDLib::ESev_Status
                , tCIDLib::EClass_Internal
            );
        }
        __hpenThis = kCIDWnd::hpenInvalid;
    }
}



// -----------------------------------------------------------------------------
//  CLASS: TPenJanitor
// PREFIX: jan
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TPenJanitor: Constructors and destructors
// -----------------------------------------------------------------------------

TPenJanitor::TPenJanitor(TGraphicDevice* const pgdevToSanitize) :

    __hpenSaved(kCIDWnd::hpenInvalid)
    , __pgdevSanitize(0)
{
    __hpenSaved = (tCIDWnd::TPenHandle)GetCurrentObject
    (
        pgdevToSanitize->hdevThis()
        , OBJ_PEN
    );

    if (!__hpenSaved)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_GetPen
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // It worked so store the device
    __pgdevSanitize = pgdevToSanitize;
}

TPenJanitor::~TPenJanitor()
{
    if (__pgdevSanitize)
    {
        if (!SelectObject(__pgdevSanitize->hdevThis(), __hpenSaved))
        {
            TKrnlError kerrToLog;
            facCIDWnd.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDWErrs::errcPen_Select
                , kerrToLog
                , tCIDLib::ESev_Status
                , tCIDLib::EClass_Internal
            );
        }
    }
}
