//
//  FILE NAME: CIDWnd_Device.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/27/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TGraphicDevice class, which is a wrapper class
//  for the host OS' graphics output device handle.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDWnd_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TGraphicDevice,TObject)



// -----------------------------------------------------------------------------
//  CLASS: TGraphicDevice
// PREFIX: gdev
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TGraphicDevice: Constructors and destructors
// -----------------------------------------------------------------------------

TGraphicDevice::TGraphicDevice( const   tCIDWnd::TDeviceHandle  hdevToUse
                                , const EDevType                eType
                                , const tCIDLib::EAdoptOpts     eAdopt) :
    __eAdopt(eAdopt)
    , __eType(eType)
    , __hdevThis(hdevToUse)
    , __phpiThis(0)
{
    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_GraphicDevCount, 1);
}

TGraphicDevice::TGraphicDevice( const   tCIDWnd::TDeviceHandle  hdevToUse
                                , const THostPaintInfo&         hpiToUse) :

    __eAdopt(tCIDLib::EAdoptOpt_Adopt)
    , __eType(EDevType_Paint)
    , __hdevThis(hdevToUse)
    , __phpiThis(0)
{
    __phpiThis = new THostPaintInfo;
    *__phpiThis = hpiToUse;
    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_GraphicDevCount, 1);
}

TGraphicDevice::TGraphicDevice(const TWindow& wndSource) :

    __eAdopt(tCIDLib::EAdoptOpt_Adopt)
    , __eType(EDevType_Gotten)
    , __hdevThis(0)
    , __phpiThis(0)
{
    __hdevThis = GetDCEx(wndSource.hwndThis(), 0, DCX_WINDOW);
    if (!__hdevThis)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_Get
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_GraphicDevCount, 1);

}

TGraphicDevice::~TGraphicDevice()
{
    _pmtrgCore->OffsetCard(tCIDWnd_::eCoreMetric_GraphicDevCount, -1);

    if ((__eAdopt == tCIDLib::EAdoptOpt_Adopt) && __hdevThis)
    {
        if (__eType == EDevType_Paint)
        {
            // Get the window associated with this paint DC
            tCIDWnd::TWindowHandle hwndDev = WindowFromDC(__hdevThis);
            if (!hwndDev)
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcDev_GetWindow
                    , kerrToLog
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                );
            }

            // And pass it to the end paint
            if (!EndPaint(hwndDev, __phpiThis))
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcDev_EndPaint
                    , kerrToLog
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                );
            }
        }
         else if (__eType == EDevType_Gotten)
        {
            // Get the window associated with this paint DC
            tCIDWnd::TWindowHandle hwndDev = WindowFromDC(__hdevThis);
            if (!hwndDev)
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcDev_GetWindow
                    , kerrToLog
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                );
            }

            if (!ReleaseDC(hwndDev, __hdevThis))
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcDev_Release
                    , kerrToLog
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                );
            }
        }
         else if (__eType == EDevType_Created)
        {
            if (!DeleteDC(__hdevThis))
            {
                TKrnlError kerrToLog;
                facCIDWnd.LogKrnlErr
                (
                    __FILE__
                    , __LINE__
                    , kCIDWErrs::errcDev_Delete
                    , kerrToLog
                    , tCIDLib::ESev_APIFailed
                    , tCIDLib::EClass_CantDo
                );
            }
        }
    }
    delete __phpiThis;
}


// -----------------------------------------------------------------------------
//  TGraphicDevice: Public, non-virtual methods
// -----------------------------------------------------------------------------

TArea TGraphicDevice::areaBounds(const tCIDLib::TBoolean bReset) const
{
    tCIDLib::TCard4 c4Flags = 0;
    if (bReset)
        c4Flags |= DCB_RESET;

    tCIDLib::THostRectl rectlTmp;
    tCIDLib::TCard4 c4Res = GetBoundsRect
    (
        __hdevThis
        , (RECT*)&rectlTmp
        , c4Flags
    );

    if (!c4Res)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_GetBounds
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    if ((c4Res & DCB_SET) == DCB_SET)
        return TArea(rectlTmp, tCIDLib::ERectl_Inclusive);

    return TArea();
}


TArea
TGraphicDevice::areaString( const   TString&            strText
                            , const tCIDLib::TCard4     c4Start
                            , const tCIDLib::TCard4     c4Len
                            , const tCIDLib::TBoolean   bSkipMnemonic) const
{
    //
    //  Pull in the indicated substring. Note that this guy will handle the
    //  c4Len parm being c4MaxCard or too longer for the source or
    //  destination strings.
    //
    TString strTmp(kCIDLib::pszEmptyZStr, strText.c4Length());
    strTmp.CopyInSubStr(strText, c4Start, c4Len);

    //
    //  If we need to pull out the mnemonic char, then do that now before
    //  we pull out the substring.
    //
    if (bSkipMnemonic)
    {
        tCIDLib::TCard4 c4Ind;
        if (strTmp.bFirstOccurrence(kCIDWnd::chMnemonic, c4Ind))
            strTmp.Cut(c4Ind, 1);
    }

    SIZE SizeInfo;
    if (!GetTextExtentPoint32
    (
        __hdevThis
        , strTmp.pszData()
        , strTmp.c4Length()
        , &SizeInfo))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_GetTextExtent
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strText
        );
    }

    //
    //  Return the area. Since we are returning a zero based area, we have
    //  to subtract 1 from the size values if they are not zero already.
    //
    return TArea
    (
        0
        , 0
        , SizeInfo.cx ? SizeInfo.cx-1 : 0
        , SizeInfo.cy ? SizeInfo.cy-1 : 0
    );
}


tCIDLib::TBoolean TGraphicDevice::bBoundsEnabled() const
{
    tCIDLib::THostRectl rectTmp;
    tCIDLib::TCard4 c4Res = GetBoundsRect
    (
        __hdevThis
        , (RECT*)&rectTmp
        , 0
    );

    if (!c4Res)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_GetBounds
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return ((c4Res & DCB_ENABLE) != 0);
}


tCIDLib::TVoid TGraphicDevice::DisableBoundsAccumulation()
{
    if (!(SetBoundsRect(__hdevThis, 0, DCB_DISABLE) & DCB_DISABLE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_EnableBounds
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TCard4
TGraphicDevice::c4DrawText(const TString& strText, const TPoint& pntAlign)
{
    // Get the length since we use it multiple times here
    tCIDLib::TCard4 c4Len = strText.c4Length();

    //
    //  This simple version just puts the text to the top and left, and
    //  draws at the current position.
    //
    if (!TextOut
    (
        __hdevThis
        , pntAlign.i4X()
        , pntAlign.i4Y()
        , strText.pszData()
        , c4Len))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_DisplayText
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strText
        );
    }

    // This one always draws the full string
    return c4Len;
}

tCIDLib::TCard4
TGraphicDevice::c4DrawText( const   TString&            strText
                            , const TArea&              areaFormat
                            , const tCIDLib::EHJustify  eHJustify
                            , const tCIDLib::EVJustify  eVJustify
                            , const tCIDWnd::ETextFmt   eFormat)
{
    //
    //  Convert our text formatting and justification values into the
    //  value required by DrawText().
    //
    tCIDLib::TCard4 c4Fmt = 0;

    if (eHJustify == tCIDLib::EHJustify_Left)
        c4Fmt |= DT_LEFT;
    else if (eHJustify == tCIDLib::EHJustify_Right)
        c4Fmt |= DT_RIGHT;
    else
        c4Fmt |= DT_CENTER;

    if (eVJustify == tCIDLib::EVJustify_Top)
        c4Fmt |= DT_TOP;
    else if (eVJustify == tCIDLib::EVJustify_Bottom)
        c4Fmt |= DT_BOTTOM;
    else
        c4Fmt |= DT_VCENTER;

    if (eFormat & tCIDWnd::ETextFmt_EndEllipsis)
        c4Fmt |= DT_END_ELLIPSIS;
    else if (eFormat & tCIDWnd::ETextFmt_PathEllipsis)
        c4Fmt |= DT_PATH_ELLIPSIS;

    if (eFormat & tCIDWnd::ETextFmt_ExpandTabs)
        c4Fmt |= DT_EXPANDTABS;

    if (eFormat & tCIDWnd::ETextFmt_NoClip)
        c4Fmt |= DT_NOCLIP;

    if (eFormat & tCIDWnd::ETextFmt_NoMnemonic)
        c4Fmt |= DT_NOPREFIX;

    if (eFormat & tCIDWnd::ETextFmt_RightToLeft)
        c4Fmt |= DT_RTLREADING;

    if (eFormat & tCIDWnd::ETextFmt_SingleLine)
        c4Fmt |= DT_RTLREADING;

    if (eFormat & tCIDWnd::ETextFmt_WordBreak)
        c4Fmt |= DT_WORDBREAK;

    //
    //  If the fill flag is on, then fill the area with the current
    //  background color.
    //
    if (eFormat & tCIDWnd::ETextFmt_FillArea)
        Fill(areaFormat, TRGBClr(GetBkColor(__hdevThis)));

    // Convert the area into a host rect
    tCIDLib::THostRectl rectlFormat;
    areaFormat.ToRectl(rectlFormat, tCIDLib::ERectl_Inclusive);

    DRAWTEXTPARAMS TextParams = {0};
    TextParams.cbSize = sizeof(TextParams);
    if (!DrawTextEx
    (
        __hdevThis
        , (tCIDLib::Tch*)strText.pszData()
        , -1
        , (RECT*)&rectlFormat
        , c4Fmt
        , &TextParams))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_DisplayText
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , strText
        );
    }
    return TextParams.uiLengthDrawn;
}


tCIDLib::TVoid
TGraphicDevice::Draw3DBorder(   const   tCIDWnd::E3DBorders eBorder
                                , const TArea&              areaPos
                                , const tCIDLib::TCard4     c4Width
                                , const TRGBClr&            rgbFgn
                                , const TRGBClr&            rgbBgn)
{
    tCIDLib::TInt4   i4Tmp;

    if (!eBorder)
        return;

    // Save the attributes that we might change here
    TGraphicDevJanitor janState(this);

    // Get a copy of the area we can modify
    TArea areaDraw(areaPos);

    if ((eBorder == tCIDWnd::E3DBorder_Raised)
    ||  (eBorder == tCIDWnd::E3DBorder_Sunken))
    {
        //
        //  Draw either a raised or sunken border.
        //
        __Draw3DBorder
        (
            (eBorder == tCIDWnd::E3DBorder_Raised)
            , areaDraw
            , c4Width
            , rgbFgn
            , rgbBgn
        );
    }
     else if (eBorder == tCIDWnd::E3DBorder_3D)
    {
        //
        //  This one is a raised border followed by a sunken border. Each
        //  one is half the width, with the raised side getting the odd
        //  pel if there is one.
        //
        tCIDLib::TCard4 c4ActualWidth = c4Width;
        if (c4ActualWidth < 2)
            c4ActualWidth = 2;

        // Get half the width, plus any odd pel
        i4Tmp = tCIDLib::TInt4(c4ActualWidth >> 1);
        if (c4ActualWidth & 1)
            i4Tmp++;

        //
        //  Draw the raised and sunken borders. The call bumps in the
        //  passed area by the size of the border, so we just call again
        //  to draw inside it.
        //
        __Draw3DBorder(kCIDLib::True, areaDraw, i4Tmp, rgbFgn, rgbBgn);
        i4Tmp = tCIDLib::TInt4(c4ActualWidth >> 1);
        __Draw3DBorder(kCIDLib::False, areaDraw, i4Tmp, rgbFgn, rgbBgn);
    }
     else
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_Bad3DBorder
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TInteger(eBorder)
        );
    }
}


tCIDLib::TVoid
TGraphicDevice::DrawBox(const   TPoint&         pntFrom
                        , const TPoint&         pntTo
                        , const tCIDLib::TCard4 c4Rounding)
{
    tCIDLib::TBoolean bRes = kCIDLib::True;
    if (c4Rounding)
    {
        if (!RoundRect
        (
            __hdevThis
            , pntFrom.i4X()
            , pntFrom.i4Y()
            , pntTo.i4X()
            , pntTo.i4Y()
            , c4Rounding
            , c4Rounding))
        {
            bRes = kCIDLib::False;
        }
    }
     else
    {
        if (!Rectangle
        (
            __hdevThis
            , pntFrom.i4X()
            , pntFrom.i4Y()
            , pntTo.i4X()
            , pntTo.i4Y()))
        {
            bRes = kCIDLib::False;
        }
    }

    if (!bRes)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_Box
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}

tCIDLib::TVoid
TGraphicDevice::DrawBox(const TArea& areaBox, const tCIDLib::TCard4 c4Rounding)
{
    TPoint pntUL, pntLR;
    areaBox.ToPoints(pntUL, pntLR);
    DrawBox(pntUL, pntLR, c4Rounding);
}


tCIDLib::TVoid TGraphicDevice::DrawFocusArea(const TArea& areaToDraw)
{
    tCIDLib::THostRectl rectlDraw;
    areaToDraw.ToRectl(rectlDraw, tCIDLib::ERectl_Inclusive);
    if (!DrawFocusRect(__hdevThis, (RECT*)&rectlDraw))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_DrawFocus
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaToDraw
        );
    }
}


tCIDWnd::EBackModes TGraphicDevice::eBackMode() const
{
    tCIDLib::TInt4 i4Mode = GetBkMode(__hdevThis);

    if (i4Mode == OPAQUE)
        return tCIDWnd::EBackMode_Opaque;
    else if (i4Mode == TRANSPARENT)
        return tCIDWnd::EBackMode_Transparent;

    TKrnlError kerrToLog;
    facCIDWnd.LogKrnlErr
    (
        __FILE__
        , __LINE__
        , kCIDWErrs::errcDev_SetBackMode
        , kerrToLog
        , tCIDLib::ESev_APIFailed
        , tCIDLib::EClass_CantDo
        , TInteger(i4Mode)
    );

    // Make the compiler happy, won't get here
    return tCIDWnd::EBackMode_Opaque;;
}


tCIDLib::TVoid TGraphicDevice::EnableBoundsAccumulation()
{
    if (!(SetBoundsRect(__hdevThis, 0, DCB_ENABLE) & DCB_ENABLE))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_EnableBounds
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid
TGraphicDevice::Fill(const TArea& areaToFill, const TRGBClr& rgbToUse)
{
    // Create a fill brush from the passed color
    tCIDWnd::TBrushHandle   hbrFill = __hbrCreateBrush(rgbToUse);

    //
    //  Convert the area to a host rectangle. Since the host fill op is
    //  non-inclusive, create a non-inclusive rectangle
    //
    tCIDLib::THostRectl  rectlFill;
    areaToFill.ToRectl(rectlFill, tCIDLib::ERectl_NonInclusive);

    if (!FillRect(__hdevThis, (RECT*)&rectlFill, hbrFill))
    {
        __DeleteObject(hbrFill, L"Brush");

        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_FillArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , areaToFill
        );
    }
    __DeleteObject(hbrFill, L"Brush");
}

tCIDLib::TVoid
TGraphicDevice::Fill(   const   TPoint&     pntFrom
                        , const TPoint&     pntTo
                        , const TRGBClr&    rgbToUse)
{
    // Create a fill brush from the passed color
    tCIDWnd::TBrushHandle   hbrFill = __hbrCreateBrush(rgbToUse);

    //
    //  Convert the points to a host rect. Since the fill operation is
    //  non-inclusive, we have to add 1 to the 'to' point to make the fill
    //  actually fill the whole requested area.
    //
    tCIDLib::THostRectl  rectlFill;
    rectlFill.i4Left = pntFrom.i4X();
    rectlFill.i4Top = pntFrom.i4Y();
    rectlFill.i4Right = pntTo.i4X()+1;
    rectlFill.i4Bottom = pntTo.i4Y()+1;

    if (!FillRect(__hdevThis, (RECT*)&rectlFill, hbrFill))
    {
        __DeleteObject(hbrFill, L"Brush");

        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_FillArea
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TArea(pntFrom, pntTo)
        );
    }
    __DeleteObject(hbrFill, L"Brush");
}


TRGBClr TGraphicDevice::rgbBackColor() const
{
    tCIDLib::TCard4 c4Color = GetBkColor(__hdevThis);

    if (c4Color == CLR_INVALID)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_GetBackColor
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return TRGBClr(c4Color);
}


tCIDLib::TVoid TGraphicDevice::ResetBoundsArea()
{
    if (!SetBoundsRect(__hdevThis, 0, DCB_RESET) & DCB_RESET)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_ResetBounds
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid TGraphicDevice::SetBackColor(const TRGBClr& rgbToSet)
{
    if (SetBkColor(__hdevThis, rgbToSet.c4Color()) == CLR_INVALID)
        throw TKrnlError();
}

tCIDLib::TVoid TGraphicDevice::SetBackMode(const tCIDWnd::EBackModes eMode)
{
    tCIDLib::TCard4 c4Mode = 0;

    if (eMode == tCIDWnd::EBackMode_Opaque)
        c4Mode = OPAQUE;
    else if (eMode == tCIDWnd::EBackMode_Transparent)
        c4Mode = TRANSPARENT;
    else
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_BadBackMode
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TInteger(eMode)
        );
    }

    if (!SetBkMode(__hdevThis, eMode))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_SetBackMode
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
}


tCIDLib::TVoid TGraphicDevice::SetTextColor(const TRGBClr& rgbToSet)
{
    if (::SetTextColor(__hdevThis, rgbToSet.c4Color()) == CLR_INVALID)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_SetTextColor
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , rgbToSet
        );
    }
}


tCIDLib::TVoid
TGraphicDevice::Stroke( const   TArea&          areaToStroke
                        , const TPen&           penToUse
                        , const tCIDLib::TCard4 c4Width)
{
    // Get the area to an array of host points
    THostPoint  aptPath[5];
    areaToStroke.ToPointArray(aptPath);

    TPenJanitor kjanSave(this);
    UsePen(penToUse);

    for (tCIDLib::TCard4 c4Count = 0; c4Count < c4Width; c4Count++)
    {
        if (!Polyline(__hdevThis, (POINT*)aptPath, 5))
        {
            TKrnlError kerrToLog;
            facCIDWnd.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDWErrs::errcDev_PolyLine
                , kerrToLog
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
                , TCardinal(5)
            );
        }

        //
        //  Adjust the points inwards for the next round. If it inverts
        //  because the user passed a width that's bigger than the area,
        //  then that's their fault.
        //
        aptPath[0].i4X++;
        aptPath[0].i4Y--;
        aptPath[1].i4X++;
        aptPath[1].i4Y++;
        aptPath[2].i4X--;
        aptPath[2].i4Y++;
        aptPath[3].i4X--;
        aptPath[3].i4Y--;
        aptPath[4].i4X++;
        aptPath[4].i4Y--;
    }
}


tCIDLib::TVoid TGraphicDevice::UseBrush(const TBrush& brToUse)
{
    __SelectObject(brToUse.hbrThis(), L"Brush");
}

tCIDLib::TVoid TGraphicDevice::UsePen(const TPen& penToUse)
{
    __SelectObject(penToUse.hpenThis(), L"Pen");
}


// -----------------------------------------------------------------------------
//  TGraphicDevice: Private, static methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TGraphicDevice::__DeleteObject( const   tCIDLib::THandle    hToDelete
                                , const tCIDLib::Tch* const pszType)
{
    if (!DeleteObject(hToDelete))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_DeleteObject
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(pszType)
        );
    }
}


tCIDWnd::TBrushHandle
TGraphicDevice::__hbrCreateBrush(const TRGBClr& rgbToUse)
{
    tCIDWnd::TBrushHandle hbrRet = CreateSolidBrush(rgbToUse.c4Color());
    if (!hbrRet)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return hbrRet;
}

tCIDWnd::TPenHandle
TGraphicDevice::__hpenCreatePen(const TRGBClr& rgbToUse)
{
    tCIDWnd::TPenHandle hpenRet = CreatePen(PS_SOLID, 0, rgbToUse.c4Color());
    if (!hpenRet)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcPen_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    return hpenRet;
}


// -----------------------------------------------------------------------------
//  TGraphicDevice: Private, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TGraphicDevice::__Draw3DBorder( const   tCIDLib::TBoolean   bRaised
                                ,       TArea&              areaPos
                                , const tCIDLib::TInt4      i4Width
                                , const TRGBClr&            rgbFgn
                                , const TRGBClr&            rgbBgn)
{
    THostPoint  aptPath[6], aptPathSave[6];

    // Get the area to an array of host points
    areaPos.ToPointArray(aptPath);

    // Copy the path list to a backup
    TRawMem::CopyMemBuf(aptPathSave, aptPath, sizeof(aptPath));

    tCIDWnd::TPenHandle     hpenFgn = kCIDWnd::hpenInvalid;
    tCIDWnd::TPenHandle     hpenBgn = kCIDWnd::hpenInvalid;
    tCIDWnd::TBrushHandle   hbrFgn  = kCIDWnd::hbrInvalid;
    tCIDWnd::TBrushHandle   hbrBgn  = kCIDWnd::hbrInvalid;
    try
    {
        //
        //  Handle the special cases of widths of 1 or 2 which are not thick
        //  enough to deal with via the normal code. We just do the left and
        //  upper sides as a line and the right and bottom sides as a line
        //  for a width of 1. If 2, then we just do the same thing inward
        //  one more pel.
        //
        tCIDLib::TInt4  i4Cnt = i4Width;
        if (i4Cnt < 3)
        {
            hpenFgn = __hpenCreatePen(rgbFgn);
            hpenBgn = __hpenCreatePen(rgbBgn);
            while (i4Cnt)
            {
                if (bRaised)
                    __SelectObject(hpenFgn, L"Pen");
                else
                    __SelectObject(hpenBgn, L"Pen");

                MoveToEx(__hdevThis, aptPath[0].i4X, aptPath[1].i4Y, 0);

                if (!Polyline(__hdevThis, (POINT*)&aptPath[1], 2))
                {
                    TKrnlError kerrToLog;
                    facCIDWnd.LogKrnlErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDWErrs::errcDev_PolyLine
                        , kerrToLog
                        , tCIDLib::ESev_APIFailed
                        , tCIDLib::EClass_CantDo
                        , TCardinal(2)
                    );
                }

                if (bRaised)
                    __SelectObject(hpenBgn, L"Pen");
                else
                    __SelectObject(hpenFgn, L"Pen");

                if (!Polyline(__hdevThis, (POINT*)&aptPath[3], 2))
                {
                    TKrnlError kerrToLog;
                    facCIDWnd.LogKrnlErr
                    (
                        __FILE__
                        , __LINE__
                        , kCIDWErrs::errcDev_PolyLine
                        , kerrToLog
                        , tCIDLib::ESev_APIFailed
                        , tCIDLib::EClass_CantDo
                        , TCardinal(2)
                    );
                }

                aptPath[0].i4X++;
                aptPath[0].i4Y--;
                aptPath[1].i4X++;
                aptPath[1].i4Y++;
                aptPath[2].i4X--;
                aptPath[2].i4Y++;
                aptPath[3].i4X--;
                aptPath[3].i4Y--;
                aptPath[4].i4X++;
                aptPath[4].i4Y--;

                i4Cnt--;
            }

            if (hpenFgn)
                __DeleteObject(hpenFgn, L"Pen");

            if (hpenBgn)
                __DeleteObject(hpenBgn, L"Pen");
        }
         else
        {
            hbrFgn = __hbrCreateBrush(rgbFgn);
            hbrBgn = __hbrCreateBrush(rgbBgn);

            // Build up a path for the upper left side of the border.
            aptPath[3].i4X = aptPath[2].i4X-(i4Width-1);
            aptPath[3].i4Y = aptPath[2].i4Y+(i4Width-1);
            aptPath[4].i4X = aptPath[1].i4X+(i4Width-1);
            aptPath[4].i4Y = aptPath[1].i4Y+(i4Width-1);
            aptPath[5].i4X = aptPath[0].i4X+(i4Width-1);
            aptPath[5].i4Y = aptPath[0].i4Y-(i4Width-1);

            // Set up for the upper left corner
            if (bRaised)
                __SelectObject(hbrFgn, L"Brush");
            else
                __SelectObject(hbrBgn, L"Brush");

            //
            //  Now begin a path which will consist of the points that
            //  we just set up.
            //
            if (!BeginPath(__hdevThis))
                throw TKrnlError();

            // Draw a polyline to create the path
            if (!Polyline(__hdevThis, (POINT*)aptPath, 6))
                throw TKrnlError();

            //
            //  Now end the path. Now we can fill the path as needed.
            //
            if (!EndPath(__hdevThis))
                throw TKrnlError();

            // Fill the upper left section with the current brush
            if (!FillPath(__hdevThis))
                throw TKrnlError();

            // Set up the line segments for the left and bottom sides
            aptPath[0].i4X = aptPathSave[2].i4X;
            aptPath[0].i4Y = aptPathSave[2].i4Y;
            aptPath[1].i4X = aptPathSave[3].i4X;
            aptPath[1].i4Y = aptPathSave[3].i4Y;
            aptPath[2].i4X = aptPathSave[0].i4X;
            aptPath[2].i4Y = aptPathSave[0].i4Y;
            aptPath[3].i4X = aptPathSave[0].i4X+(i4Width-1);
            aptPath[3].i4Y = aptPathSave[0].i4Y-(i4Width-1);
            aptPath[4].i4X = aptPathSave[3].i4X-(i4Width-1);
            aptPath[4].i4Y = aptPathSave[3].i4Y-(i4Width-1);
            aptPath[5].i4X = aptPathSave[2].i4X-(i4Width-1);
            aptPath[5].i4Y = aptPathSave[2].i4Y+(i4Width-1);

            // Set up for the lower right corner
            if (bRaised)
                __SelectObject(hbrBgn, L"Brush");
            else
                __SelectObject(hbrFgn, L"Brush");

            if (!BeginPath(__hdevThis))
                throw TKrnlError();

            // Draw a polyline to create the path
            if (!Polyline(__hdevThis, (POINT*)aptPath, 6))
                throw TKrnlError();

            // Now end the path. Now we can fill the path as needed.
            if (!EndPath(__hdevThis))
                throw TKrnlError();

            // Fill the lower right section with the current brush
            if (!FillPath(__hdevThis))
                throw TKrnlError();

            if (hbrFgn)
                __DeleteObject(hbrFgn, L"Brush");

            if (hbrBgn)
                __DeleteObject(hbrBgn, L"Brush");
        }
    }

    catch(...)
    {
        if (hpenFgn)
            __DeleteObject(hpenFgn, L"Pen");

        if (hpenBgn)
            __DeleteObject(hpenBgn, L"Pen");

        if (hbrFgn)
            __DeleteObject(hbrFgn, L"Brush");

        if (hbrBgn)
            __DeleteObject(hbrBgn, L"Brush");

        throw;
    }
    areaPos.Deflate(i4Width);
}


tCIDLib::TVoid
TGraphicDevice::__SelectObject( const   tCIDLib::THandle    hToSelect
                                , const tCIDLib::Tch* const pszType)
{
    if (!SelectObject(__hdevThis, hToSelect))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcDev_SelectObject
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
            , TString(pszType)
        );
    }
}
