//
//  FILE NAME: CIDWnd_Brush.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 05/27/97
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This module implements the TBrush class. This class controls the
//  attributes of filled areas on a graphics graphics output device.
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDWnd_.Hpp"


// ----------------------------------------------------------------------------
//  Do our RTTI macros
// ----------------------------------------------------------------------------
RTTIData(TBrush,TObject)
RTTIData(TBrushJanitor,TObject)



// -----------------------------------------------------------------------------
//  CLASS: TBrush
// PREFIX: br
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TBrush: Constructors and destructors
// -----------------------------------------------------------------------------

TBrush::TBrush() :

    __eType(TBrush::EBrushType_Solid)
    , __hbrThis(kCIDWnd::hbrInvalid)
{
    // Fill in a brush structure, for a default of black solid fill
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbStyle = BS_SOLID;

    // Now create the brush.
    tCIDWnd::TBrushHandle hbrTmp = CreateBrushIndirect(&BrushInfo);
    if (!hbrTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hbrThis = hbrTmp;
}

TBrush::TBrush(const TRGBClr& rgbFill) :

    __eType(TBrush::EBrushType_Solid)
    , __hbrThis(kCIDWnd::hbrInvalid)
{
    // Fill in a brush structure
    LOGBRUSH BrushInfo = {0};
    BrushInfo.lbColor = rgbFill.c4Color();
    BrushInfo.lbStyle = BS_SOLID;

    // Now create the brush.
    tCIDWnd::TBrushHandle hbrTmp = CreateBrushIndirect(&BrushInfo);
    if (!hbrTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }
    __hbrThis = hbrTmp;
}

TBrush::TBrush(const TGraphicDevice& gdevSource) :

    __eType(TBrush::EBrushType_Solid)
    , __hbrThis(kCIDWnd::hbrInvalid)
{
    // Get the current brush
    tCIDWnd::TBrushHandle hbrTmp = (tCIDWnd::TBrushHandle)GetCurrentObject
    (
        gdevSource.hdevThis()
        , OBJ_BRUSH
    );
    if (!hbrTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_GetBrush
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Query the information for the current brush
    LOGBRUSH BrushInfo;
    if (!GetObject(hbrTmp, sizeof(BrushInfo), &BrushInfo))
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_QueryInfo
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Now create our brush from it
    hbrTmp = CreateBrushIndirect(&BrushInfo);
    if (!hbrTmp)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_Create
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // Figure out what kind it is
    if ((BrushInfo.lbStyle == BS_SOLID)
    ||  (BrushInfo.lbStyle == BS_HOLLOW)
    ||  (BrushInfo.lbStyle == BS_NULL))
    {
        __eType = EBrushType_Solid;
    }
     else if (BrushInfo.lbStyle == BS_HATCHED)
    {
        __eType = EBrushType_Hatched;
    }
     else if (BrushInfo.lbStyle == BS_PATTERN)
    {
        __eType = EBrushType_Pattern;
    }
     else
    {
        facCIDWnd.LogErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_BadType
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // And store the handle
    __hbrThis = hbrTmp;
}

TBrush::~TBrush()
{
    if (__hbrThis != kCIDWnd::hbrInvalid)
    {
        if (!DeleteObject(__hbrThis))
        {
            // Log as a AppStatus so no exception is thrown out of dtor
            TKrnlError kerrToLog;
            facCIDWnd.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDWErrs::errcBrush_Delete
                , kerrToLog
                , tCIDLib::ESev_Status
                , tCIDLib::EClass_Internal
            );
        }
        __hbrThis = kCIDWnd::hbrInvalid;
    }
}



// -----------------------------------------------------------------------------
//  CLASS: TBrushJanitor
// PREFIX: jan
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TBrushJanitor: Constructors and destructors
// -----------------------------------------------------------------------------

TBrushJanitor::TBrushJanitor(TGraphicDevice* const pgdevToSanitize) :

    __hbrSaved(kCIDWnd::hbrInvalid)
    , __pgdevSanitize(0)
{
    __hbrSaved = (tCIDWnd::TBrushHandle)GetCurrentObject
    (
        pgdevToSanitize->hdevThis()
        , OBJ_BRUSH
    );

    if (!__hbrSaved)
    {
        TKrnlError kerrToLog;
        facCIDWnd.LogKrnlErr
        (
            __FILE__
            , __LINE__
            , kCIDWErrs::errcBrush_GetBrush
            , kerrToLog
            , tCIDLib::ESev_APIFailed
            , tCIDLib::EClass_CantDo
        );
    }

    // It worked so store the device
    __pgdevSanitize = pgdevToSanitize;
}

TBrushJanitor::~TBrushJanitor()
{
    if (__pgdevSanitize)
    {
        if (!SelectObject(__pgdevSanitize->hdevThis(), __hbrSaved))
        {
            TKrnlError kerrToLog;
            facCIDWnd.LogKrnlErr
            (
                __FILE__
                , __LINE__
                , kCIDWErrs::errcBrush_Select
                , kerrToLog
                , tCIDLib::ESev_APIFailed
                , tCIDLib::EClass_CantDo
            );
        }
    }
}
