//
//  FILE NAME: CIDWnd.Cpp
//
//     AUTHOR: Dean Roddey
//
//    CREATED: 11/23/96
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
//  DESCRIPTION:
//
//  This is the main module of the facility. This guy provides the DLL
//  init function in order to handle initialization of any modules that
//  require it
//
//  CAVEATS/GOTCHAS:
//

// ----------------------------------------------------------------------------
//  Includes
// ----------------------------------------------------------------------------
#include    "CIDWnd_.Hpp"
#include    "CIDLib_ModuleInit_.Hpp"


// ----------------------------------------------------------------------------
//  Global data
//
//  facCIDWnd
//      This is the facility object for CIDWnd.Dll. Its exported via the
//      main header.
// ----------------------------------------------------------------------------
TFacCIDWnd      facCIDWnd;


// ----------------------------------------------------------------------------
//  Intrafacility data
//
//  _pmtrgCore
//      This is a pointer to the main metric group. It is set up during the
//      main module init/term handler below.
// ----------------------------------------------------------------------------
TMetricGroup*   _pmtrgCore = 0;


// ----------------------------------------------------------------------------
//  Forward references
// ----------------------------------------------------------------------------
static tCIDLib::TVoid _InitTermMain
(
    const   tCIDLib::EInitTerm      eInitTerm
    , const tCIDLib::EGlobalStates  eGlobals
    , const TModule&                modInit
    , const tCIDLib::TCard4         c4MaxChars
    ,       tCIDLib::Tch* const     pszFailReason
);


// ----------------------------------------------------------------------------
//  Data for local use
//
//  __afnInitTerm
//      This is the list of init term functions for this facility.
//
//  __c4InitCount
//      This is the count of elements in the init/term list.
// ----------------------------------------------------------------------------
static tCIDLib::TModuleInitTermRec  __afnInitTerm[] =
{
        { _InitTermMain         , L"Main Module" }
    ,   { _InitTermWindow       , L"Base Window" }
};
static tCIDLib::TCard4  __c4InitCount = c4ArrayElems(__afnInitTerm);


// ----------------------------------------------------------------------------
//  Do some RTTI macros for classes that are all inline (i.e. they don't
//  have a Cpp file to do it in.
// ----------------------------------------------------------------------------
RTTIData(TWndBorder,TObject)


// ----------------------------------------------------------------------------
//  Local functions
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: _InitTermMain
//
// DESCRIPTION:
//
//  This is called from the DLL init below and it does any needed local
//  setup. This guy handles setting up the core windowing metrics.
// ---------------------------------------
//   INPUT: eInitTerm indicates what initialization phase we are in.
//          eGlobals indicates whether this is before constructors or
//              after destructors for globals.
//          modInit is the module object for initializing this module.
//          c4MaxChars is the max chars that the failure reason buffer
//              can hold.
//
//  OUTPUT: pszFailReason is filled with the reason for a failure.
//
//  RETURN: None
//
tCIDLib::TVoid _InitTermMain(   const   tCIDLib::EInitTerm      eInitTerm
                                , const tCIDLib::EGlobalStates  eGlobals
                                , const TModule&                modInit
                                , const tCIDLib::TCard4         c4MaxChars
                                ,       tCIDLib::Tch* const     pszFailReason)
{
    const tCIDLib::Tch* pszPhase = L"Uknown";

    try
    {
        if ((eInitTerm == tCIDLib::EInitTerm_Initialize)
        &&  (eGlobals == tCIDLib::EGlobalState_Before))
        {
            //
            //  First of all load up some primal text that will be used
            //  in the popups if any errors occur. These are constants
            //  here so we cast away the const'ness to set them. If the
            //  load fails, then the compiled in default text is there
            //  to use.
            //
            pszPhase = L"Loading primal popup text";
            const tCIDLib::Tch* pszMsg;
            const tCIDLib::Tch* pszTmp;

            pszMsg  = modInit.pszLoadCIDMsg(kCIDWMsgs::midFac_Title1);
            if (pszMsg)
            {
                pszTmp = const_cast<tCIDLib::Tch*>(kCIDWnd_::pszTitle1);
                pszTmp = pszMsg;
            }

            pszMsg  = modInit.pszLoadCIDMsg(kCIDWMsgs::midFac_Title2);
            if (pszMsg)
            {
                pszTmp = const_cast<tCIDLib::Tch*>(kCIDWnd_::pszTitle2);
                pszTmp = pszMsg;
            }

            //
            //  Create the core metrics group. It is declared and exported
            //  to all other facility modules that have metrics include, in
            //  the CIDWnd_.Hpp header.
            //
            pszPhase = L"Core Window metrics registration setup";
            TMetricSystem::TMetricInfo amtriCoreInfo[tCIDWnd_::eCoreMetric_Count];

            // Loop through the metrics and fill them in
            tCIDLib::TCard4 c4Index;
            for (c4Index = 0; c4Index < tCIDWnd_::eCoreMetric_Count; c4Index++)
            {
                TMetricSystem::TMetricInfo* pCur = &amtriCoreInfo[c4Index];

                if (c4Index == tCIDWnd_::eCoreMetric_WindowCount)
                {
                    pCur->strMetricName = L"Window Count";
                    pCur->strMetricHelp = L"The count of TWindow derived objects";
                    pCur->eType = tCIDLib::EMetric_Cardinal;
                }
                 else if (c4Index == tCIDWnd_::eCoreMetric_GraphicDevCount)
                {
                    pCur->strMetricName = L"Graphics Device Count";
                    pCur->strMetricHelp = L"The count of TGraphicDevice objects";
                    pCur->eType = tCIDLib::EMetric_Cardinal;
                }
            }

            //
            //  Register the new group and save away the pointer for internal
            //  use.
            //
            pszPhase = L"Core Window metrics registration";
            _pmtrgCore = TMetricSystem::pmtrgRegisterGroup
            (
                L"CIDWnd Core Metrics"
                , L"Fundamental CIDLib Window metrics"
                , amtriCoreInfo
                , tCIDWnd_::eCoreMetric_Count
            );
        }
         else if ((eInitTerm == tCIDLib::EInitTerm_Terminate)
              &&  (eGlobals == tCIDLib::EGlobalState_After))
        {
            pszPhase = L"Deletion of core metrics";
            delete _pmtrgCore;
            _pmtrgCore = 0;
        }
    }

    catch(...)
    {
        TRawStr::CopyCatStr
        (
            pszFailReason
            , c4MaxChars
            , pszPhase
            , L" failed"
        );
        throw;
    }
}


// ----------------------------------------------------------------------------
//  DLL Initialization and program entry point
// ----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: CIDWndInit
//
// DESCRIPTION:
//
//  This is the module entry point, which is set in the project settings.
// -------------------------------------
//   INPUT: hmodThis is the module handle of this DLL facility.
//          eReason is the reason we are getting called.
//          pReserved is not used of course.
//
//  OUTPUT: None
//
//  RETURN: 1 if successful, else 0. If a 0 is returned, then the load of
//              the client app will fail.
//
tCIDLib::TCard4 DLLINITAPI
CIDWndInit( tCIDLib::TModHandle     hmodThis
            , tCIDLib::EInitTerm    eInitTerm
            , tCIDLib::TVoid*       pDummy)
{
    if (!bCIDLib_DoModuleInitTerm
    (
        eInitTerm
        , hmodThis
        , tCIDLib::EModType_Dll
        , __c4InitCount
        , __afnInitTerm
        , kCIDWnd_::pszTitle1
        , kCIDWnd_::pszTitle2))
    {
        return 0;
    }
    return 1;
}
