//
// NAME: CIDTracer_ViewGeometry.Hpp
//
// DESCRIPTION: 
//
//  This is the header file for the CIDTracer_ViewGeometry.Cpp module. This
//  module implements the TViewGeometry class. This class provides viewing
//  geometry control for ray tracing applications. Viewing geometry includes
//  eye position, eye direction, distance from eye to view plane, up/right
//  vectors, view plane to physical coordinate translation.
//
//  This class also contains the objects that make up the scene. There are
//  APIs to insert objects into the scene, which are put into objects of the
//  specialized ray tracer object list template class, CIDRay_List.Hpp.
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/05/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//
//  1)  Equality for this class is just implemented as pointer equality,
//      since its complex and not very likely to be used interchangeably.
//


// -----------------------------------------------------------------------------
//  CLASS: TViewGeometry
// PREFIX: view
// -----------------------------------------------------------------------------
class CIDTRACEREXP TViewGeometry :

    public TObject, public MStreamable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TViewGeometry();

        TViewGeometry
        (
            const   tCIDLib::TCard4             c4ImageCX
            , const tCIDLib::TCard4             c4ImageCY
            , const tCIDTracer::EQualityLevels  eQuality
            , const tCIDTracer::ESampleModes    eSampleMode
            , const T3DVector&                  vecEyePos
            , const T3DVector&                  vecEyeDir
            , const T3DVector&                  vecRight
            , const T3DVector&                  vecUp
            , const TFRGBClr&                   frgbBgn
            , const tCIDLib::TCard4             c4MaxDepth
        );

        ~TViewGeometry();



        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TCard4 c4LightRaysCast() const;

        tCIDLib::TCard4 c4InitRayHits() const;

        tCIDLib::TCard4 c4LightCount
        (
        )   const;

        tCIDLib::TCard4 c4MaxReflections() const;

        tCIDLib::TCard4 c4MaxReflections
        (
            const   tCIDLib::TCard4         c4NewMax
        );

        tCIDLib::TCard4 c4ObjectCount() const;

        tCIDLib::TCard4 c4PelsProcessed() const;

        tCIDLib::TCard4 c4ImageCX() const;

        tCIDLib::TCard4 c4ImageCX
        (
            const   tCIDLib::TCard4         c4NewX
        );

        tCIDLib::TCard4 c4ImageCY() const;

        tCIDLib::TCard4 c4ImageCY
        (
            const   tCIDLib::TCard4         c4NewY
        );

        tCIDTracer::EQualityLevels eQuality() const;

        tCIDTracer::EQualityLevels eQuality
        (
            const   tCIDTracer::EQualityLevels eNewQuality
        );

        tCIDTracer::ESampleModes eSample() const;

        tCIDTracer::ESampleModes eSample
        (
            const   tCIDTracer::ESampleModes eNewSample
        );

        tCIDLib::TFloat8 f8StatSampleThreshold() const;

        tCIDLib::TFloat8 f8StatSampleThreshold
        (
            const   tCIDLib::TFloat8        f8Threshold
        );

        tCIDLib::TVoid InsertObj
        (
                    TRTObject* const        prtobjNew
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid LookAt
        (
            const   T3DVector&              vecFocusPnt
        );

        tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsrSrc
        );

        const TFRGBClr& frgbBgnColor() const;

        const TFRGBClr& frgbBgnColor
        (
            const   TFRGBClr&               frgbNewBgn
        );

        const TFRGBClr& frgbFogColor() const;

        const TFRGBClr& frgbFogColor
        (
            const   TFRGBClr&               frgbNewFog
        );

        tCIDLib::TVoid TracePel
        (
            const   tCIDLib::TInt4          i4XPel
            , const tCIDLib::TInt4          i4YPel
            ,       TRGBClr&                rgbToFill
        );

        tCIDLib::TVoid SetSceneDimensions
        (
            const   tCIDLib::TCard4         c4CXPels
            , const tCIDLib::TCard4         c4CYPels
        );

        tCIDLib::TVoid SetUpRight
        (
            const   T3DVector&              vecUp
            , const T3DVector&              vecRight
        );

        tCIDLib::TVoid Translate
        (
            const   T3DVector&              vecTrans
        );

        const T3DVector& vecEyeDir() const;

        const T3DVector& vecEyeDir
        (
            const   T3DVector&              vecEyeDir
        );

        const T3DVector& vecEyePos() const;

        const T3DVector& vecEyePos
        (
            const   T3DVector&               vecEyePos
        );

        const T3DVector& vecRight() const;

        const T3DVector& vecRight
        (
            const   T3DVector&              vecRight
        );

        const T3DVector& vecSky() const;

        const T3DVector& vecSky
        (
            const   T3DVector&              vecSky
        );

        const T3DVector& vecUp() const;

        const T3DVector& vecUp
        (
            const   T3DVector&              vecUp
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Unimplemented constructors and operators
        // ---------------------------------------------------------------------
        TViewGeometry
        (
            const   TViewGeometry&          viewToCopy
        );

        tCIDLib::TVoid operator=
        (
            const   TViewGeometry&          viewToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TViewGeometry&          viewToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TViewGeometry&          viewToTest
        )   const;


        // ---------------------------------------------------------------------
        //  Private, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean __bPointShadowed
        (
            const   TLightRay&              lrayNew
            , const tCIDLib::TFloat8        f8DistT
            , const TRTObject* const        prtobjSrc
        );

        tCIDLib::TVoid __BuildIntersectList
        (
            const   TLightRay&              lrayTest
            ,       TRTObjList&             rtlstObjects
            ,       TIntersectArray&        intraIntersects
            , const TRTGeometricObj* const  prtobjSkip = 0
        )   const;

        tCIDLib::TVoid __Init();

        tCIDLib::TVoid __MakeInitRay
        (   
            const   tCIDLib::TFloat8        f8XPos
            , const tCIDLib::TFloat8        f8YPos
            ,       TLightRay&              lrayTarget
        );

        tCIDLib::TVoid __ShadePnt
        (
            const   TRTGeometricObj* const  prtobjTarget
            , const T3DVector&              vecPoint
            , const T3DVector&              vecNormal
            , const TLightRay&              lrayTrace
            ,       TFRGBClr&               frgbColor
            , const tCIDLib::TCard4         c4RecursionLev
        );

        tCIDLib::TVoid __TraceRay
        (
            const   TLightRay&              lrayTrace
            ,       TFRGBClr&               frgbColor
            , const tCIDLib::TCard4         c4RecursionLev
        );


        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __c4ImageCX
        //  __c4ImageCY
        //      These are the physical limits of the output area in pels.
        //      These are used to convert view plane coordinates (which are
        //      in magical units) to physical units for the client code.
        //
        //  __c4LightRaysCast
        //      This is the number of light rays cast so far on this scene.
        //      It will include supersampled rays, since they are all initial
        //      light rays.
        //
        //  __c4InitRayHits
        //      This is the number of initial light rays that actually hit
        //      something in the scene.
        //
        //  __c4PelsProcessed
        //      This is the number of pixels processed so far on this scene.
        //
        //  __c4MaxReflections
        //      This is the maximum allowed recursion level.
        //
        //  __eQuality
        //      This is the quality level value.
        //
        //  __eSample
        //      This is the sampling stategy.
        //
        //  __f8StatSampleThreshold
        //      This is the threshold for statistical supersampling. Setting
        //      it higher makes the render faster and the image worse. Setting
        //      it lower (down to 1) makes the render longer and the image
        //      better. At 1, it is basically the same as doing stochastic
        //      supersampling, so use that method since there would be less
        //      overhead. The default is 32.0.
        //
        //  __frgbBgn
        //      The background color, for all rays that strike no object.
        //
        //  __frgbFog
        //      The fog color. It is used as the initial color, giving
        //      everything a kind of washed out effect.
        //
        //  __prtlstObjects
        //  __prtlstLights
        //      These are lists of scene objects and light source objects.
        //      Objects can be inserted into it by the InsertObj() method.
        //
        //  __vecEyeDir
        //      The vector that indicates the direction of the viewer's eye
        //      within the 3D space. The magnitude indicates distance from
        //      the view plane.
        //
        //  __vecEyePos
        //      The vector that indicates the position of the viewer's eye
        //      within the 3D space.
        //
        //  __vecREyeDir
        //      This is the eye direction reversed. This is used during
        //      specular reflection calculations. It is updated anytime that
        //      the eye direction is set.
        //
        //  __vecRight
        //      The vector that controls horizontal scaling. Assumed origin
        //      is the center of the view plane (0.0, 0.0).
        //
        //  __vecSky
        //      Controls rotation of the camera around the imaginary axis
        //      passing through it to the focus point. Defaults to the
        //      'absolute up' of (0,1,0).
        //
        //  __vecUp
        //      The vector that controls vertical scaling. Assumed origin is
        //      the center of the view plane (0.0, 0.0)
        // ---------------------------------------------------------------------
        tCIDLib::TCard4             __c4ImageCX;
        tCIDLib::TCard4             __c4ImageCY;
        tCIDLib::TCard4             __c4InitRayHits;
        tCIDLib::TCard4             __c4LightRaysCast;
        tCIDLib::TCard4             __c4PelsProcessed;
        tCIDLib::TCard4             __c4MaxReflections;
        tCIDTracer::EQualityLevels  __eQuality;
        tCIDTracer::ESampleModes    __eSample;
        tCIDLib::TFloat8            __f8StatSampleThreshold;

        TFRGBClr                    __frgbBgn;
        TFRGBClr                    __frgbFog;
        TRTObjList*                 __prtlstObjects;
        TRTObjList*                 __prtlstLights;
        T3DVector                   __vecEyeDir;
        T3DVector                   __vecEyePos;
        T3DVector                   __vecRight;
        T3DVector                   __vecSky;
        T3DVector                   __vecUp;


        // Just for runtime, not stored
        tCIDLib::TFloat8            __f8XPelDist;
        tCIDLib::TFloat8            __f8YPelDist;
        T3DVector                   __vecREyeDir;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TViewGeometry,TObject)
};


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TCard4 TViewGeometry::c4LightRaysCast() const
{
    return __c4LightRaysCast;
}

inline tCIDLib::TCard4 TViewGeometry::c4InitRayHits() const
{
    return __c4InitRayHits;
}

inline tCIDLib::TCard4 TViewGeometry::c4MaxReflections() const
{
    return __c4MaxReflections;
}

inline tCIDLib::TCard4
TViewGeometry::c4MaxReflections(const tCIDLib::TCard4 c4NewMax)
{
    __c4MaxReflections = c4NewMax;
    return c4NewMax;
}

inline tCIDLib::TCard4 TViewGeometry::c4PelsProcessed() const
{
    return __c4PelsProcessed;
}

inline tCIDLib::TCard4 TViewGeometry::c4ImageCX() const
{
    return __c4ImageCX;
}

inline tCIDLib::TCard4
TViewGeometry::c4ImageCX(const tCIDLib::TCard4 c4NewX)
{
    __c4ImageCX = c4NewX;
    return __c4ImageCX;
}

inline tCIDLib::TCard4 TViewGeometry::c4ImageCY() const
{
    return __c4ImageCY;
}

inline tCIDLib::TCard4 TViewGeometry::c4ImageCY(const tCIDLib::TCard4 c4NewY)
{
    __c4ImageCY = c4NewY;
    return __c4ImageCY;
}

inline tCIDTracer::EQualityLevels TViewGeometry::eQuality() const
{
    return __eQuality;
}

inline tCIDTracer::EQualityLevels
TViewGeometry::eQuality(const tCIDTracer::EQualityLevels eNewQuality)
{
    __eQuality = eNewQuality;
    return eNewQuality;
}

inline tCIDTracer::ESampleModes TViewGeometry::eSample() const
{
    return __eSample;
}

inline tCIDTracer::ESampleModes
TViewGeometry::eSample(const tCIDTracer::ESampleModes eNewSample)
{
    __eSample = eNewSample;
    return eNewSample;
}

inline tCIDLib::TFloat8 TViewGeometry::f8StatSampleThreshold() const
{
    return __f8StatSampleThreshold;
}

inline tCIDLib::TFloat8
TViewGeometry::f8StatSampleThreshold(const tCIDLib::TFloat8 f8Threshold)
{
    __f8StatSampleThreshold = f8Threshold;
    return f8Threshold;
}

inline const TFRGBClr& TViewGeometry::frgbBgnColor() const
{
    return __frgbBgn;
}

inline const TFRGBClr&
TViewGeometry::frgbBgnColor(const TFRGBClr& frgbNewBgn)
{
    __frgbBgn = frgbNewBgn;
    return __frgbBgn;
}

inline const TFRGBClr& TViewGeometry::frgbFogColor() const
{
    return __frgbFog;
}

inline const TFRGBClr&
TViewGeometry::frgbFogColor(const TFRGBClr& frgbNewFog)
{
    __frgbFog = frgbNewFog;
    return __frgbFog;
}

inline tCIDLib::TVoid TViewGeometry::Translate(const T3DVector& vecTrans)
{
    __vecEyePos += vecTrans;
}

inline const T3DVector& TViewGeometry::vecEyeDir() const
{
    return __vecEyeDir;
}

inline const T3DVector& TViewGeometry::vecEyePos() const
{
    return __vecEyePos;
}

inline const T3DVector& TViewGeometry::vecRight() const
{
    return __vecRight;
}

inline const T3DVector& TViewGeometry::vecSky() const
{
    return __vecSky;
}

inline const T3DVector& TViewGeometry::vecUp() const
{
    return __vecUp;
}

