//
// NAME: CIDTracer_ViewGeometry.Cpp
//
// DESCRIPTION:
//
//  This module implements the TViewGeometry class, which provides the
//  viewing geometry and overall scene attributes for a ray traced image.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//



// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI members macro
// -----------------------------------------------------------------------------
RTTIData(TViewGeometry,TObject)


// -----------------------------------------------------------------------------
//  Local constants
//
//  __c4SamplePosCnt
//      The number of elements in the sample position array, __af8SamplePos.
//
//  __af8SamplePos
//      This is a list of offsets that are used during supersampling.
//
//  __f8Limit
//      A local value used to demark distance limits that we won't even try
//      to deal with.
// -----------------------------------------------------------------------------
static const tCIDLib::TCard4    __c4SamplePosCnt = 9;
static const tCIDLib::TFloat8   __af8SamplePos[__c4SamplePosCnt][2] =
{
      {         0.0,  0.0        }
    , { -0.33333333, -0.33333333 }
    , { -0.33333333,  0.0        }
    , { -0.33333333,  0.33333333 }
    , {         0.0, -0.33333333 }
    , {         0.0,  0.33333333 }
    , {  0.33333333, -0.33333333 }
    , {  0.33333333,  0.0        }
    , {  0.33333333,  0.33333333 }
};
static const tCIDLib::TFloat8   __f8Limit = 1.0e17;



// -----------------------------------------------------------------------------
//  CLASS: TViewGeometry
// PREFIX: view
// -----------------------------------------------------------------------------

TViewGeometry::TViewGeometry(const   tCIDLib::TCard4            c4ImageCX
                             , const tCIDLib::TCard4            c4ImageCY
                             , const tCIDTracer::EQualityLevels eQuality
                             , const tCIDTracer::ESampleModes   eSample
                             , const T3DVector&                 vecEyePos
                             , const T3DVector&                 vecEyeDir
                             , const T3DVector&                 vecRight
                             , const T3DVector&                 vecUp
                             , const TFRGBClr&                  frgbBgn
                             , const tCIDLib::TCard4            c4MaxDepth) :
    __c4ImageCX(c4ImageCX)
    , __c4ImageCY(c4ImageCY)
    , __c4InitRayHits(0)
    , __c4LightRaysCast(0)
    , __c4MaxReflections(c4MaxDepth)
    , __c4PelsProcessed(0)
    , __eQuality(eQuality)
    , __eSample(eSample)
    , __f8StatSampleThreshold(1.2)
    , __prtlstObjects(0)
    , __prtlstLights(0)
    , __frgbBgn(frgbBgn)
    , __frgbFog(0,0,0)
    , __vecEyePos(vecEyePos)
    , __vecEyeDir(vecEyeDir)
    , __vecREyeDir(0,0,1)
    , __vecRight(vecRight)
    , __vecSky(0, 1, 0)
    , __vecUp(vecUp)
{
    // Create the lists
    __prtlstObjects = new TRTObjList;
    __prtlstLights = new TRTObjList;

    // Call the init method to set up data
    __Init();

    // Bump up the count of existing scenes
    _pmtrgTracerCore->OffsetCard(tCIDTracer_::eCorEMetric_ActiveScenes, 1);
}

TViewGeometry::TViewGeometry() :

    __c4ImageCX(128)
    , __c4ImageCY(128)
    , __c4LightRaysCast(0)
    , __c4InitRayHits(0)
    , __c4MaxReflections(3)
    , __c4PelsProcessed(0)
    , __eQuality(tCIDTracer::eQuality_5)
    , __eSample(tCIDTracer::ESampleMode_Normal)
    , __f8StatSampleThreshold(1.2)
    , __prtlstObjects(0)
    , __prtlstLights(0)
    , __frgbBgn(TFRGBClr(0, 0, 0))
    , __frgbFog(TFRGBClr(0, 0, 0))
    , __vecEyePos(0, 0, 0.0)
    , __vecEyeDir(0, 0, 1.0)
    , __vecRight(1.0, 0, 0)
    , __vecSky(0, 1, 0)
    , __vecUp(0, 1.0, 0)
{
    // Create the lists
    __prtlstObjects = new TRTObjList;
    __prtlstLights = new TRTObjList;

    // Call the init method to set up data
    __Init();

    // Bump up the count of existing scenes
    _pmtrgTracerCore->OffsetCard(tCIDTracer_::eCorEMetric_ActiveScenes, 1);
}

TViewGeometry::~TViewGeometry()
{
    // Flush the list and object lists and delete them
    __prtlstObjects->Flush();
    __prtlstLights->Flush();
    delete __prtlstObjects;
    delete __prtlstLights;

    // Bump down the count of scenes being traced
    _pmtrgTracerCore->OffsetCard(tCIDTracer_::eCorEMetric_ActiveScenes, -1);
}


// -----------------------------------------------------------------------------
//  TViewGeometry: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TCard4 TViewGeometry::c4ObjectCount() const
{
    return __prtlstObjects->c4NodeCount();
}

tCIDLib::TCard4 TViewGeometry::c4LightCount() const
{
    return __prtlstLights->c4NodeCount();
}


tCIDLib::TVoid
TViewGeometry::InsertObj(   TRTObject* const    prtobjNew
                            , const TString&    strName
                            , const TString&    strNameSpace)
{
    if (prtobjNew->bIsDescendantOf(L"TLightSource"))
        __prtlstLights->Add(prtobjNew, strName, strNameSpace);
     else
        __prtlstObjects->Add(prtobjNew, strName, strNameSpace);
}


tCIDLib::TVoid TViewGeometry::LookAt(const T3DVector& vecFocusPnt)
{
    tCIDLib::TFloat8    f8DirLen, f8UpLen, f8RightLen, f8Hand;

    // Get the lengths of the eye dir, up, and right vectors
    f8DirLen    = __vecEyeDir.f8Magnitude();
    f8UpLen     = __vecUp.f8Magnitude();
    f8RightLen  = __vecRight.f8Magnitude();

    T3DVector vecTmp(__vecEyeDir);
    vecTmp.Cross(__vecUp);
    f8Hand = vecTmp.f8Dot(__vecRight);

    // Copy the focus point to the direction vector
    __vecEyeDir = vecFocusPnt;

    // Subtract the eye position from the direction and normalize it
    __vecEyeDir -= __vecEyePos;
    __vecEyeDir.Normalize();

    __vecRight = vecCross(__vecEyeDir, __vecSky);
    __vecRight.Normalize();

    __vecUp = vecCross(__vecRight, __vecEyeDir);
    __vecEyeDir *= f8DirLen;

    if (f8Hand >= 0.0)
        __vecRight *= f8RightLen;
    else
        __vecRight *= -f8RightLen;

    __vecUp *= f8UpLen;


    // And set the inverse direction vector
    __vecREyeDir = __vecEyeDir;
    __vecREyeDir.Negate();
}


tCIDLib::TVoid TViewGeometry::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens eToken;
    TString             strName;
    TString             strNameSpace;
    TString             strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Next token has to be the open paren
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if ((eToken == TParseInfo::eToken_AddObject)
    ||  (eToken == TParseInfo::eToken_AddLight))
    {
        TRTObject* prtobjNew = prsiSrc.prtobjParseObject(strName);

        // Add this object to the list of scene objects
        InsertObj(prtobjNew, strName, prsiSrc.strCurNameSpace());
    }
     else if (eToken == TParseInfo::eToken_Color)
    {
        TFRGBClr frgbClr;
        prsiSrc.ParseColor(frgbClr);
        frgbBgnColor(frgbClr);
    }
     else if (eToken == TParseInfo::eToken_EyePosition)
    {
        T3DVector vecTmp;
        prsiSrc.ParseVector(vecTmp);
        vecEyePos(vecTmp);
    }
     else if (eToken == TParseInfo::eToken_FogColor)
    {
        TFRGBClr frgbClr;
        prsiSrc.ParseColor(frgbClr);
        frgbFogColor(frgbClr);
    }
     else if (eToken == TParseInfo::eToken_Lens)
    {
        // Get the lens type
        eToken = prsiSrc.eNextToken(strToken);

        if (strToken == L"FishEye")
            __vecEyeDir.Set(0, 0, 0.9);
        else if (strToken == L"Normal")
            __vecEyeDir.Set(0, 0, 1.0);
        else if (strToken == L"TelePhoto")
            __vecEyeDir.Set(0, 0, 1.4);
        else
            prsiSrc.ThrowErr(kTracerErrs::errcParse_LensType, strToken);
    }
     else if (eToken == TParseInfo::eToken_LookAt)
    {
        //
        //  The data is a position vector that we make the eye
        //  direction vector point to.
        //
        T3DVector vecTmp;
        prsiSrc.ParseVector(vecTmp);

        LookAt(vecTmp);
    }
     else if (eToken == TParseInfo::eToken_Depth)
    {
        tCIDLib::TCard4      c4Tmp;
        prsiSrc.ParseCard(c4Tmp);
        c4MaxReflections(c4Tmp);
    }
     else if (eToken == TParseInfo::eToken_Quality)
    {
        tCIDLib::TCard4  c4Tmp;
        prsiSrc.ParseCard(c4Tmp);

        if (!c4Tmp || (c4Tmp > tCIDTracer::EQualityLevels_Max))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcView_BadQuality
                , kCIDLib::pszEmptyZStr
                , TCardinal(c4Tmp)
                , TCardinal(tCIDTracer::EQualityLevels_Max)
            );
        }

        eQuality(tCIDTracer::EQualityLevels(c4Tmp));
    }
     else if (eToken == TParseInfo::eToken_Sample)
    {
        prsiSrc.eNextToken(strToken);

        if (strToken == "Normal")
            __eSample = tCIDTracer::ESampleMode_Normal;
        else if (strToken == "Stochastic")
            __eSample = tCIDTracer::ESampleMode_Stochastic;
        else if (strToken == "Statistical")
            __eSample = tCIDTracer::ESampleMode_Statistical;
        else
            prsiSrc.ThrowErr(kTracerErrs::errcView_UnknownSampling, strToken);

        //
        //  If statistical, get the threshold value if its there. If the
        //  next token is a comma, then the next one is the value.
        //
        if (__eSample == tCIDTracer::ESampleMode_Statistical)
        {
            eToken = prsiSrc.eNextToken(strToken);

            if (eToken == TParseInfo::eToken_Comma)
            {
                tCIDLib::TFloat8    f8Tmp;
                prsiSrc.ParseFloat(f8Tmp);

                if ((f8Tmp < 1.0) || (f8Tmp > 3.0))
                {
                    prsiSrc.ThrowErr
                    (
                        kTracerErrs::errcParse_InvalidValue
                        , TString(L"Statistical")
                        , TFloat(1.0)
                        , TFloat(3.0)
                    );
                }

                // Its valid so set it
                f8StatSampleThreshold(f8Tmp);
            }
             else
            {
                prsiSrc.UnGet(strToken);
            }
        }
    }
     else if (eToken == TParseInfo::eToken_UpVector)
    {
        T3DVector vecTmp;
        prsiSrc.ParseVector(vecTmp);

        vecUp(vecTmp);
    }
     else
    {
        prsiSrc.ThrowErr(kTracerErrs::errcParse_NotAllowedHere, strToken);
    }

    // Has to end with a close paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid
TViewGeometry::TracePel(const   tCIDLib::TInt4  i4XPel
                        , const tCIDLib::TInt4  i4YPel
                        ,       TRGBClr&        rgbToFill)
{
    tCIDLib::TFloat8    f8XPos, f8YPos;
    TFRGBClr            frgbClr;
    TLightRay           lrayThis;

    // Get the coordinates as floats
    f8XPos = i4XPel;
    f8YPos = i4YPel;

    //
    //  According to the sampling scheme that we are using, we do this part
    //  differently. If just normal, we do just the centered rays at exactly
    //  the x/y position. For stochastic we do a fixed set of 9 jittered
    //  offsets.
    //
    if (__eSample == tCIDTracer::ESampleMode_Normal)
    {
        __MakeInitRay(f8XPos, f8YPos, lrayThis);
        __TraceRay(lrayThis, frgbClr, 0);
    }
     else if (__eSample == tCIDTracer::ESampleMode_Stochastic)
    {
        tCIDLib::TFloat8    f8Offset;
        TFRGBClr            frgbSuper;

        // Compute the random noise for this pixel position
        f8Offset = 
        (
            _c2Random3D
            (
                tCIDLib::TInt2(i4XPel), tCIDLib::TInt2(i4YPel)) & 0x7FFF
            ) / 32768.0 * 0.33333333 - 0.16666666;

        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4SamplePosCnt; c4Ind++)
        {
            //
            //  Make the initial ray for this position, and trace it. Note
            //  that we don't have to clear the color we pass. __TraceRay
            //  will do that.
            //
            __MakeInitRay
            (
                f8XPos + f8Offset + __af8SamplePos[c4Ind][0]
                , f8YPos + f8Offset + __af8SamplePos[c4Ind][1]
                , lrayThis
            );
            __TraceRay(lrayThis, frgbSuper, 0);

            // Add in the new color amount
            frgbSuper.Clip();
            frgbClr.AddScaled(frgbSuper, 0.111111111);
        }
    }
     else if (__eSample == tCIDTracer::ESampleMode_Statistical)
    {
        tCIDLib::TFloat8    af8Mags[__c4SamplePosCnt];
        TFRGBClr            afrgbClr[__c4SamplePosCnt];
        tCIDLib::TFloat8    f8Lowest = 3.0;
        tCIDLib::TFloat8    f8Offset;
        tCIDLib::TFloat8    f8Mag;

        // Compute the random noise for this pixel position
        f8Offset = 
        (
            _c2Random3D(tCIDLib::TInt2(i4XPel), tCIDLib::TInt2(i4YPel)) & 0x7FFF
        ) / 32768.0 * 0.33333333 - 0.16666666;

        for (tCIDLib::TCard4 c4Ind = 0; c4Ind < __c4SamplePosCnt; c4Ind++)
        {
            //
            //  Make the initial ray for this position, and trace it. Note
            //  that we don't have to clear the color we pass. __TrayRay will
            //  do that.
            //
            __MakeInitRay
            (
                f8XPos + f8Offset + __af8SamplePos[c4Ind][0]
                , f8YPos + f8Offset + __af8SamplePos[c4Ind][1]
                , lrayThis
            );
            __TraceRay(lrayThis, afrgbClr[c4Ind], 0);

            //
            //  Get the magnitude of this color and store it if its
            //  the smallest or largest.
            //
            f8Mag = afrgbClr[c4Ind].f8Magnitude();
            if (f8Mag < f8Lowest)
                f8Lowest = f8Mag;
            af8Mags[c4Ind] = f8Mag;

            //
            //  If we've done 4 rays, then we need to start checking
            //  the average magnitude. If the difference between it and
            //  the center ray is less than the threshold, then we are
            //  done.
            //
            if (c4Ind == 3)
            {
                f8Mag = 0.0;
                for (tCIDLib::TCard4 c4Tmp = 0; c4Tmp < c4Ind; c4Tmp++)
                    f8Mag += (af8Mags[c4Tmp] - f8Lowest);

                // If its over the threshold, then do another
                if ((f8Mag / c4Ind) > __f8StatSampleThreshold)
                    continue;

                // It was under the threshold, so break out
                break;
            }
        }

        // Add in the number we got, scaled down to 1/xth
        for (tCIDLib::TCard4 c4MagInd = 0; c4MagInd < c4Ind; c4MagInd++)
            frgbClr.AddScaled(afrgbClr[c4MagInd], 1.0 / c4Ind);
    }

    // Bump up the count of pels processed.
    __c4PelsProcessed++;

    frgbClr.ConvertToRGB(rgbToFill);
}


tCIDLib::TVoid
TViewGeometry::SetSceneDimensions(  const   tCIDLib::TCard4 c4CXPels
                                    , const tCIDLib::TCard4 c4CYPels)
{
    __c4ImageCX = c4CXPels;
    __c4ImageCY = c4CYPels;

    //
    //  Calculate the range between the center points of a pel on our
    //  virtual view plane, which is always 1.0 units in both directions.
    //
    __f8XPelDist = 1.0 / tCIDLib::TFloat8(__c4ImageCX);
    __f8YPelDist = 1.0 / tCIDLib::TFloat8(__c4ImageCY);
}


tCIDLib::TVoid
TViewGeometry::SetUpRight(const T3DVector& vecUp, const T3DVector& vecRight)
{
    __vecUp = vecUp;
    __vecRight = vecRight;
}


const T3DVector& TViewGeometry::vecEyeDir(const T3DVector& vecEyeDir)
{
    __vecEyeDir = vecEyeDir;
    __vecREyeDir = vecEyeDir;
    __vecREyeDir.Negate();

    return __vecEyeDir;
}

const T3DVector& TViewGeometry::vecEyePos(const T3DVector& vecEyePos)
{
    __vecEyePos = vecEyePos;
    return __vecEyePos;
}

const T3DVector& TViewGeometry::vecRight(const T3DVector& vecRight)
{
    __vecRight = vecRight;
    return __vecRight;
}

const T3DVector& TViewGeometry::vecSky(const T3DVector& vecSky)
{
    __vecSky = vecSky;
    return __vecSky;
}

const T3DVector& TViewGeometry::vecUp(const T3DVector& vecUp)
{
    __vecUp = vecUp;
    return __vecUp;
}


// -----------------------------------------------------------------------------
//  TViewGeometry: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TViewGeometry::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the geometry info into the string
    strmToWriteTo   << L"Pels={" << __c4ImageCX
                    << L"," << __c4ImageCY
                    << L"}, Quality=" << __eQuality;
}


tCIDLib::TVoid
TViewGeometry::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Flush any existing lights or objects
    __prtlstLights->Flush();
    __prtlstObjects->Flush();

    // Get out the object members
    strmToReadFrom >> __frgbBgn;
    strmToReadFrom >> __frgbFog;
    strmToReadFrom >> __vecEyeDir;
    strmToReadFrom >> __vecEyePos;
    strmToReadFrom >> __vecRight;
    strmToReadFrom >> __vecSky;
    strmToReadFrom >> __vecUp;

    // Get fundamental members out
    strmToReadFrom >> __c4ImageCX;
    strmToReadFrom >> __c4ImageCY;
    strmToReadFrom >> __c4InitRayHits;
    strmToReadFrom >> __c4LightRaysCast;
    strmToReadFrom >> __c4PelsProcessed;
    strmToReadFrom >> __c4MaxReflections;
    strmToReadFrom >> __eQuality;
    strmToReadFrom >> __eSample;
    strmToReadFrom >> __f8StatSampleThreshold;

    //
    //  Validate the quality and sample enums, to make sure that we are not
    //  reading bad info.
    //
    if (!bIsValidEnum(__eQuality) || !bIsValidEnum(__eSample))
    {
        // <TBD> Log an exception here
    }

    // Get out the counts of objects and lights
    tCIDLib::TCard4 c4ObjCount;
    tCIDLib::TCard4 c4LightCount;
    strmToReadFrom >> c4ObjCount;
    strmToReadFrom >> c4LightCount;

    // Call the init method
    __Init();

    // Now stream in all of the lights and objects
    TRTObject* prtobjNew;
    for (tCIDLib::TCard4 c4Index = 0; c4Index < c4ObjCount; c4Index++)
    {
        ::PolymorphicRead(prtobjNew, strmToReadFrom);
        __prtlstObjects->Append(prtobjNew);
    }

    for (c4Index = 0; c4Index < c4LightCount; c4Index++)
    {
        ::PolymorphicRead(prtobjNew, strmToReadFrom);
        __prtlstLights->Append(prtobjNew);
    }
}

tCIDLib::TVoid
TViewGeometry::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the object members
    strmToWriteTo << __frgbBgn;
    strmToWriteTo << __frgbFog;
    strmToWriteTo << __vecEyeDir;
    strmToWriteTo << __vecEyePos;
    strmToWriteTo << __vecRight;
    strmToWriteTo << __vecSky;
    strmToWriteTo << __vecUp;

    // Write out the fundamental data members
    strmToWriteTo << __c4ImageCX;
    strmToWriteTo << __c4ImageCY;
    strmToWriteTo << __c4InitRayHits;
    strmToWriteTo << __c4LightRaysCast;
    strmToWriteTo << __c4PelsProcessed;
    strmToWriteTo << __c4MaxReflections;
    strmToWriteTo << __eQuality;
    strmToWriteTo << __eSample;
    strmToWriteTo << __f8StatSampleThreshold;

    // Write out the count of objects and lights we have
    strmToWriteTo << __prtlstObjects->c4NodeCount();
    strmToWriteTo << __prtlstLights->c4NodeCount();

    //
    //  Now iterate the list of objects and lights and stream them
    //  all out.
    //
    TRTObjNode* pnodeCur = __prtlstObjects->pnodeHead();
    while (pnodeCur)
    {
        ::PolymorphicWrite(pnodeCur->pobjData(), strmToWriteTo);
        pnodeCur = pnodeCur->pnodeNext();
    }

    pnodeCur = __prtlstLights->pnodeHead();
    while (pnodeCur)
    {
        ::PolymorphicWrite(pnodeCur->pobjData(), strmToWriteTo);
        pnodeCur = pnodeCur->pnodeNext();
    }
}


// -----------------------------------------------------------------------------
//  TViewGeometry: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __BuildIntersectList
//
// DESCRIPTION:
//
//  This method will build a list of all of the objects intersected by the
//  test ray. This method is recursive if there are bounding, CSG, or
//  composite objects in the scene.
// ---------------------------------------
//   INPUT: lrayTest is the ray to test for intersections
//          rtlstObjects is the list of objects to test
//          prtobjSkip is an optional object pointer. This object (if not
//              null) will not be added to the list.
//
//  OUTPUT: slstIntersect is updated with an TIntersect object for every
//              object intersected.
//
//  RETURN: None
//
tCIDLib::TVoid
TViewGeometry::__BuildIntersectList(const   TLightRay&          lrayTrace
                                    ,       TRTObjList&         rtlstObjects
                                    ,       TIntersectArray&    intraHits
                                    , const TRTGeometricObj*    const prtobjSkip) const
{
    TRTObjNode* pnodeCur = rtlstObjects.pnodeHead();
    while (pnodeCur)
    {
        // Get the object data
        TRTObject* const prtobjCur = pnodeCur->pobjData();

        // If it is the skip object, then move on
        if (prtobjCur == (TRTGeometricObj*)prtobjSkip)
        {
            pnodeCur = pnodeCur->pnodeNext();
            continue;
        }

        // Get all of our intersections into the list
        prtobjCur->AllIntersects(lrayTrace, intraHits);

        // Move to the next node
        pnodeCur = pnodeCur->pnodeNext();
    }
}


//
// FUNCTION/METHOD NAME: __bPointShadowed
//
// DESCRIPTION:
//
//  This method will see if the point, from which the ray is eminating,
//  is shadowed by an object.
// ---------------------------------------
//   INPUT: lrayTest is the ray that is being shot from the object surface
//              to the light source.
//          f8LightDistj is the distance to the light source. If anything has
//              a smaller distance, then the ray origin is in shadow from
//              this light source.
//          prtobjSrc is the source object from which the ray is eminating.
//              We have to skip this object.
//
//  OUTPUT: None
//
//  RETURN: eTrue if the point is shadowed, else eFalse.
//
tCIDLib::TBoolean
TViewGeometry::__bPointShadowed(const   TLightRay&          lrayTest
                                , const tCIDLib::TFloat8    f8LightDist
                                , const TRTObject* const    prtobjSrc)
{
    tCIDLib::TFloat8    f8Dist1, f8Dist2;

    TRTObjNode* pnodeCur = __prtlstObjects->pnodeHead();
    while (pnodeCur)
    {
        TRTObject* const prtobjCur = pnodeCur->pobjData();

        if (prtobjCur != prtobjSrc)
        {
            if (prtobjCur->bTestIntersect(lrayTest, f8Dist1, f8Dist2))
            {
                if ((f8Dist1 > kCIDTracer_::f8Epsilon)
                &&  (f8Dist1 < f8LightDist))
                {
                    return kCIDLib::True;
                }

                if (f8Dist2 != f8Dist1)
                {
                    if ((f8Dist2 > kCIDTracer_::f8Epsilon)
                    &&  (f8Dist2 < f8LightDist))
                    {
                        return kCIDLib::True;
                    }
                }
            }
        }

        // Move to the next node
        pnodeCur = pnodeCur->pnodeNext();
    }
    return kCIDLib::False;
}


//
// FUNCTION/METHOD NAME: __ShadePnt
//
// DESCRIPTION:
//
//  This method will shade the point at which the ray intersects the passed
//  object.
// ---------------------------------------
//   INPUT: prtobjTarget is a pointer to the object being struck
//          vecPoint is the point of intersection
//          vecNormal is the normal at the point of intersection
//          lrayTrace is the ray striking the object
//          frgbColor, on input, is the current color component
//          c4RecursionLev is the current level of recursion
//
//  OUTPUT: frgbColor is updated as each level of recursion is executed
//          vecNormal might be modified via the object's texture
//
//  RETURN: None
//
tCIDLib::TVoid
TViewGeometry::__ShadePnt(  const  TRTGeometricObj* const   prtobjTarget
                            , const T3DVector&              vecPoint
                            , const T3DVector&              vecNormal
                            , const TLightRay&              lrayTrace
                            ,       TFRGBClr&               frgbColor
                            , const tCIDLib::TCard4         c4RecursionLev)
{
    tCIDLib::TFloat8    f8Dist;
    TLightRay           lrayNew;
    TLightRay           lrayReflected;
    TFRGBClr            frgbLight;
    TFRGBClr            frgbNew;
    TFRGBClr            frgbOrig;
    TFRGBClr            frgbTmp;

    // If the point is too distant to worry about, just return
    if (!vecPoint.bWithinLimits(__f8Limit))
        return;

    //
    //  Loop through all of the textures associated with this object
    //  and add up their contributions.
    //
    TRTTxtrNode* pnodeCur = prtobjTarget->rtlstTextures().pnodeHead();

    // If no textures, then we are done
    if (!pnodeCur)
        return;

    frgbColor.FadeToBlack();
    while (pnodeCur)
    {
        // Get a pointer to the texture
        TRTTexture* const ptxtrCur = pnodeCur->pobjData();

        // Get the texture's basic color as a default
        frgbTmp.FadeToBlack();

        // Get the texture's color at the intersection
        if (__eQuality > tCIDTracer::eQuality_3)
            ptxtrCur->QueryColorAt(vecPoint, frgbTmp);
         else
            frgbTmp = ptxtrCur->frgbClr();

        // Save a copy of this original color
        frgbOrig = frgbTmp;

        // Scale by the ambient color factor
        frgbTmp *= ptxtrCur->f8Ambient();

        // Loop through the light sources and get contributions
        TRTObjNode* pnodeCurLight = __prtlstLights->pnodeHead();
        while (pnodeCurLight)
        {
            TLightSource* const plsrcCur
                                    = (TLightSource*)pnodeCurLight->pobjData();

            // Create a ray from the object to this light
            f8Dist = plsrcCur->f8MakeLightRay(vecPoint, lrayNew);

            // See if it is in shadow
            if (__eQuality >= tCIDTracer::eQuality_2)
            {
                if (__bPointShadowed(lrayNew, f8Dist, prtobjTarget))
                    frgbLight = facCIDTracer.frgbBlack;
                 else
                    frgbLight = plsrcCur->frgbClr();
            }
             else
            {
                frgbLight = plsrcCur->frgbClr();
            }

            // Calculate the phong reflection
            if ((ptxtrCur->f8Phong() != 0.0)
            &&  (__eQuality >= tCIDTracer::eQuality_4))
            {
                ptxtrCur->CalcPhong
                (
                    lrayNew
                    , frgbOrig
                    , frgbLight
                    , vecNormal
                    , __vecEyeDir
                    , frgbTmp
                );
            }

            // Calculate the specular reflection
            if ((__eQuality >= tCIDTracer::eQuality_4)
            &&  (ptxtrCur->f8Specular() != 0.0))
            {
                ptxtrCur->CalcSpecular
                (
                    lrayNew
                    , frgbOrig
                    , frgbLight
                    , __vecREyeDir
                    , vecNormal
                    , frgbTmp
                );
            }

            if ((__eQuality >= tCIDTracer::eQuality_1)
            &&  (ptxtrCur->f8Diffuse() != 0.0))
            {
                ptxtrCur->CalcDiffuse
                (
                    lrayNew
                    , frgbOrig
                    , frgbLight
                    , vecNormal
                    , frgbTmp
                );
            }

            // Move to the next light source
            pnodeCurLight = pnodeCurLight->pnodeNext();
        }

        if (__eQuality >= tCIDTracer::eQuality_5)
        {
            //
            //  Handle the reflection ray, if this texture has any reflective
            //  properties that would contribute.
            //
            tCIDLib::TFloat8 f8Tmp = ptxtrCur->f8Reflection();
            if (f8Tmp > kCIDTracer_::f8TinyValue)
            {
                lrayTrace.CalcReflectedRay(vecPoint, vecNormal, lrayReflected);
                __TraceRay(lrayReflected, frgbNew, c4RecursionLev+1);
                frgbTmp.AddScaled(frgbNew, f8Tmp);
            }

            //
            //  Handle the refracted ray, if this texture has any refractive
            //  properties that would contribute.
            //
            f8Tmp = ptxtrCur->f8Refraction();
            if (f8Tmp > kCIDTracer_::f8TinyValue)
            {
                //
                //  Create a light ray that we will set up as the refracted
                //  ray. We check the in/out state of the 
                TLightRay   lrayReflected;
                
            }
        }

        //
        //  Add the accumulated color value for this texture to the
        //  return color. Scale it by its transparency value.
        //
        frgbColor.AddScaled(frgbTmp, 1.0 - frgbTmp.f8Alpha());

        // Move to the next node
        pnodeCur = pnodeCur->pnodeNext();
    }
}


//
// FUNCTION/METHOD NAME: __Init
//
// DESCRIPTION:
//
//  This method is called from the constructors to set up some data so that
//  the code does not have to be duplicated.
// ---------------------------------------
//   INPUT: None
//
//  OUTPUT: None
//
//  RETURN: None
//
tCIDLib::TVoid TViewGeometry::__Init()
{
    // Set up the reversed eye direction
    __vecREyeDir = __vecEyeDir;
    __vecREyeDir.Negate();

    //
    //  Calculate the range between the center points of a pel on our
    //  virtual view plane, which is always 1.0 units in both directions.
    //
    __f8XPelDist = 1.0 / tCIDLib::TFloat8(__c4ImageCX);
    __f8YPelDist = 1.0 / tCIDLib::TFloat8(__c4ImageCY);
}


//
// FUNCTION/METHOD NAME: __MakeInitRay
//
// DESCRIPTION:
//
//  This method will create the initial ray from the user's eye through the
//  indicated pixel.
// ---------------------------------------
//   INPUT: f8XPos, f8YPos is the position to shoot through
//
//  OUTPUT: lrayTarget is filled in with the ray info
//
//  RETURN: None
//
tCIDLib::TVoid
TViewGeometry::__MakeInitRay(   const   tCIDLib::TFloat8    f8XPos
                                , const tCIDLib::TFloat8    f8YPos
                                ,       TLightRay&          lrayTarget)
{
    // Calculate the scaled positions within the viewing plane
    tCIDLib::TFloat8 f8XReal =
    (
        f8XPos - ((tCIDLib::TFloat8)__c4ImageCX / 2.0)
    ) / (tCIDLib::TFloat8)__c4ImageCX;

    tCIDLib::TFloat8 f8YReal =
    (
        f8YPos - ((tCIDLib::TFloat8)__c4ImageCY / 2.0)
    ) / (tCIDLib::TFloat8)__c4ImageCY;

    // Calculate the direction of the ray
    T3DVector  vecTmp1(__vecUp);
    vecTmp1 *= f8YReal;
    T3DVector  vecTmp2(__vecRight);
    vecTmp2 *= f8XReal;

    vecTmp1 += vecTmp2;
    vecTmp1 += __vecEyeDir;
    vecTmp1.Normalize();
    lrayTarget.vecDir(vecTmp1);

    // Set the origin
    lrayTarget.vecOrg(__vecEyePos);

    // Bump up the number of light rays cast
    __c4LightRaysCast++;
}


//
// FUNCTION/METHOD NAME: __TraceRay
//
// DESCRIPTION:
//
//  This method will trace the ray from the eye to the 
// ---------------------------------------
//   INPUT: lrayTrace is the ray being traced
//          c4RecursionLev is the current level of recursion
//
//  OUTPUT: frgbColor is set to the color contribution.
//
//  RETURN: None
//
tCIDLib::TVoid
TViewGeometry::__TraceRay(  const   TLightRay&      lrayTrace
                            ,       TFRGBClr&       frgbColor
                            , const tCIDLib::TCard4 c4RecursionLev)
{
    tCIDLib::TCard4     c4ItemCnt;
    tCIDLib::TFloat8    f8NormalDir;
    TIntersectArray     intraHits;

    // Init the color to the fog color
    frgbColor = __frgbFog;

    if (c4RecursionLev > __c4MaxReflections)
        return;

    //
    //  Loop through the object lists and create a list of TIntersect
    //  objects with all of the intersected objects and their distances.
    //
    __BuildIntersectList(lrayTrace, *__prtlstObjects, intraHits);

    // If no intersections, then just return
    c4ItemCnt = intraHits.c4HitCount();
    if (!c4ItemCnt)
    {
        frgbColor += __frgbBgn;
        return;
    }

    // If this is recursion level 0, then record an initial object hit
    if (!c4RecursionLev)
        __c4InitRayHits++;

    // If we hit more than one object sort the list by distance.
    if (c4ItemCnt > 1)
        intraHits.Sort();

    //
    //  Now iterate the list and add the contribution of each intersected
    //  object. Stop if we hit an object whose texture is totally opaque.
    //
    T3DVector vecPoint, vecNormal;

    // Get a pointer to the intersection
    TIntersect* const pintrCur = intraHits[0];

    // Get a local copy of the intersect point
    vecPoint = pintrCur->vecIntersect;

    // Get the surface normal at that point
    pintrCur->prtobjInter->CalcSurfaceNormal(pintrCur, vecNormal);

    //
    //  Allow all of the associated textures to adjust the normal if they
    //  want to.
    //
    if (__eQuality >= tCIDTracer::eQuality_3)
    {
        TRTTxtrNode* pnodeTxtr
                    = pintrCur->prtobjInter->rtlstTextures().pnodeHead();
        while (pnodeTxtr)
        {
            TRTTexture* const ptxtrCur = pnodeTxtr->pobjData();
            ptxtrCur->bAdjustNormalAt(vecPoint, vecNormal);
            pnodeTxtr = pnodeTxtr->pnodeNext();
        }
    }

    // See if we need to reverse the direction
    f8NormalDir = vecNormal.f8Dot(lrayTrace.vecDir());
    if (f8NormalDir >= 0.0)
        vecNormal.Negate();

    //
    //  Shade the point. Don't bump up the recursion level here. This
    //  guy will bump it up before calling us back (if needed.) Otherwise
    //  it would be going up on both ends and only recurse half the
    //  desired levels.
    //
    __ShadePnt
    (
        pintrCur->prtobjInter
        , vecPoint
        , vecNormal
        , lrayTrace
        , frgbColor
        , c4RecursionLev
    );
}
