//
// NAME: CIDTracer_Light.Hpp
//
// DESCRIPTION: 
//
//  This is the header file for the CIDTracer_Light.Cpp module. This module
//  module implements the TLightRay and TLightSource classes.
//
//  The TLightRay class provides all of the smarts to handle ray calculations.
//  It does not handle ray/object intersections though. Each object class
//  understands how to check its intersection with a ray (and to calculate
//  the surface normal and angle of reflection vectors thereof.)
//
//  The TLightSource class implements a light source. It is a very simple
//  derivative of the TTRTObject class, which is the base for all objects in
//  the scene. The light source just needs a vector to indicate its center
//  point, since light sources are considered to be perfect point sources.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 03/05/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  CLASS: TLightRay
// PREFIX: lray
// -----------------------------------------------------------------------------
class CIDTRACEREXP TLightRay :

    public TObject, public MStreamable, public MDuplicable, public MFormattable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TLightRay();

        TLightRay
        (
            const   TLightRay&              lrayToCopy
        );

        TLightRay
        (
            const   T3DVector&              vecDir
            , const T3DVector&              vecOrg
            , const tCIDTracer::EInOut      eInOut = tCIDTracer::EInOut_Neither
        );

        ~TLightRay() {}



        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TLightRay& operator=
        (
            const   TLightRay&              lrayToAssign
        );

        tCIDLib::TBoolean operator==
        (
            const   TLightRay&              lrayToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TLightRay&              lrayToTest
        )   const   {return !operator==(lrayToTest);}


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid CalcRayPoint
        (
            const   tCIDLib::TFloat8        f8Mag
            ,       T3DVector&              vecTarget
        )   const;

        tCIDLib::TVoid CalcReflectedRay
        (
            const   T3DVector&              vecPoint
            , const T3DVector&              vecNormal
            ,       TLightRay&              lrayTarget
        )   const;

        tCIDTracer::EInOut eInOut() const;

        tCIDTracer::EInOut eInOut
        (
            const   tCIDTracer::EInOut      eInOut
        );

        tCIDLib::TVoid SetVectors
        (
            const   T3DVector&              vecOrg
            , const T3DVector&              vecDir
        );

        const T3DVector& vecDir
        (
            const   T3DVector&              vecNew
        );

        const T3DVector& vecDir() const;

        const T3DVector& vecDir2() const;

        const T3DVector& vecOrg
        (
            const   T3DVector&              vecNew
        );

        const T3DVector& vecOrg() const;

        const T3DVector& vecOrg2() const;

        const T3DVector& vecOrgDir() const;

        const T3DVector& vecMixOrgDir() const;

        const T3DVector& vecMixDir() const;

        const T3DVector& vecMixOrg() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __eInOut
        //      This value is used to track the state of the rays as they
        //      are refracted through objects. If the ray is a refracted
        //      ray going through an object, its In. If its a refracted ray
        //      coming out of an object, its Out. If its not a refracted ray
        //      at all, then its Neither.
        //
        //  __vecOrg
        //  __vecOrg2
        //      The origin of the ray in 3D space, and its squared term
        //      sibling.
        //
        //  __vecDir
        //  __vecDir2
        //      The direction/magnitude of the ray, and its squared term
        //      sibling.
        // ---------------------------------------------------------------------
        tCIDTracer::EInOut      __eInOut;
        T3DVector               __vecOrg, __vecOrg2;
        T3DVector               __vecDir, __vecDir2;

        T3DVector               __vecOrgDir, __vecMixOrgDir;
        T3DVector               __vecMixOrg, __vecMixDir;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TLightRay,TObject)
        DefPolyDup(TLightRay)
};


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------

inline tCIDTracer::EInOut TLightRay::eInOut() const
{
    return __eInOut;
}

inline tCIDTracer::EInOut TLightRay::eInOut(const tCIDTracer::EInOut eInOut)
{
    __eInOut = eInOut;
    return __eInOut;
}

inline const T3DVector& TLightRay::vecDir() const
{
    return __vecDir;
}

inline const T3DVector& TLightRay::vecDir2() const
{
    return __vecDir2;
}

inline const T3DVector& TLightRay::vecOrg() const
{
    return __vecOrg;
}

inline const T3DVector& TLightRay::vecOrg2() const
{
    return __vecOrg2;
}

inline const T3DVector& TLightRay::vecOrgDir() const
{
    return __vecOrgDir;
}

inline const T3DVector& TLightRay::vecMixOrgDir()const
{
    return __vecMixOrgDir;
}

inline const T3DVector& TLightRay::vecMixDir() const
{
    return __vecMixDir;
}

inline const T3DVector& TLightRay::vecMixOrg() const
{
    return __vecMixOrg;
}



// -----------------------------------------------------------------------------
//  CLASS: TLightSource
// PREFIX: lsrc
// -----------------------------------------------------------------------------
class CIDTRACEREXP TLightSource : public TRTObject
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TLightSource();

        TLightSource
        (
            const   TLightSource&           lsrcSrc
        );

        TLightSource
        (
            const   T3DVector&              vecCenter
            , const TFRGBClr&               frgbColor
        );

        ~TLightSource() {}


        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TLightSource& operator=
        (
            const   TLightSource&           lsrcSrc
        );

        tCIDLib::TBoolean operator==
        (
            const   TLightSource&           lsrcToTest
        )   const;

        tCIDLib::TBoolean operator!=
        (
            const   TLightSource&           lsrcToTest
        )   const;


        // ---------------------------------------------------------------------
        //  Public, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid AppendTexture
        (
                    TRTTexture* const       ptxtrAppend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Invert();

        tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsiSource
        );

        tCIDLib::TVoid PrependTexture
        (
                    TRTTexture* const       ptxtrPrepend
            , const TString&                strName = NUL_TString
            , const TString&                strNameSpace = NUL_TString
        );

        tCIDLib::TVoid Transform
        (
            const   TTransform&             trnsApply
        );

        tCIDLib::TVoid TransformTextures
        (
            const   TTransform&             trnsApply
        );

        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TFloat8 f8MakeLightRay
        (
            const   T3DVector&              vecPoint
            ,       TLightRay&              lrayTarget
        )   const;

        const TFRGBClr& frgbClr() const;

        const TFRGBClr& frgbClr
        (
            const   TFRGBClr&               frgbNewClr
        );

        const T3DVector& vecCenter() const;

        const T3DVector& vecCenter
        (
            const   T3DVector&              vecNewSrc
        );


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _AllIntersects
        (
            const   TLightRay&              lrayTest
            ,       TIntersectArray&        intraIntersects
        )   const;

        tCIDLib::TBoolean _bPointInObj
        (
            const   T3DVector&              vecPoint
        )   const;

        tCIDLib::TBoolean _bTestIntersect
        (
            const   TLightRay&              lrayTest
            ,       tCIDLib::TFloat8&       f8Dist1
            ,       tCIDLib::TFloat8&       f8Dist2
        )   const;

        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _Rotate
        (
            const   T3DVector&              vecRotate
            , const TTransform&             trnsRotate
        );

        tCIDLib::TVoid _Scale
        (
            const   T3DVector&              vecScale
            , const TTransform&             trnsScale
        );

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;

        tCIDLib::TVoid _Translate
        (
            const   T3DVector&              vecTrans
            , const TTransform&             trnsTranslate
        );

    private :
        // ---------------------------------------------------------------------
        //  Private data members
        //
        //  __frgbClr
        //      The color of the light.
        //
        //  __vecCenter
        //      The vector that represents the center point of the light
        //      source.
        // ---------------------------------------------------------------------
        TFRGBClr                __frgbClr;
        T3DVector                __vecCenter;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TLightSource,TRTObject)
        DefPolyDup(TLightSource)
};

// -----------------------------------------------------------------------------
//  Public operators
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TLightSource::operator!=(const TLightSource& lsrcToTest) const
{
    return !operator==(lsrcToTest);
}

// -----------------------------------------------------------------------------
//  Public, inherited methods
// -----------------------------------------------------------------------------
inline tCIDLib::TVoid TLightSource::Invert()
{}

inline tCIDLib::TVoid
TLightSource::TransformTextures(const TTransform& trnsApply)
{
}

// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline const TFRGBClr& TLightSource::frgbClr() const
{
    return __frgbClr;
}

inline const
TFRGBClr& TLightSource::frgbClr(const TFRGBClr& frgbNewClr)
{
    __frgbClr = frgbNewClr;
    return __frgbClr;
}

inline const T3DVector& TLightSource::vecCenter() const
{
    return __vecCenter;
}

inline const T3DVector& TLightSource::vecCenter(const T3DVector& vecNewSrc)
{
    __vecCenter = vecNewSrc;
    return __vecCenter;
}


// -----------------------------------------------------------------------------
//  Protected, inherited methods
// -----------------------------------------------------------------------------

inline tCIDLib::TVoid
TLightSource::_AllIntersects(const TLightRay&, TIntersectArray&) const
{
}

inline tCIDLib::TBoolean TLightSource::_bPointInObj(const T3DVector&) const
{
    return kCIDLib::False;
}

inline tCIDLib::TBoolean
TLightSource::_bTestIntersect(  const   TLightRay&
                                ,       tCIDLib::TFloat8&
                                ,       tCIDLib::TFloat8&) const
{
    return kCIDLib::False;
}

inline tCIDLib::TVoid TLightSource::_Scale(const T3DVector&, const TTransform&)
{
}
