//
// NAME: CIDTracer_Light.Cpp
//
// DESCRIPTION:
//
//  This module implements the TLightRay and TLightSource classes. TLightRay
//  provides the smarts to handle simulated light rays projected into a 3D
//  space. The TLightSource class is a small derivative of TRTObject that
//  presents a light source in the 3D space. It just has a center point,
//  plus the common attributes provided by its parent class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TLightRay,TObject)
RTTIData2(TLightSource,TRTObject)


// -----------------------------------------------------------------------------
//  CLASS: TLightRay
// PREFIX: lray
// -----------------------------------------------------------------------------

TLightRay::TLightRay() :

    __eInOut(tCIDTracer::EInOut_Neither)
{
}

TLightRay::TLightRay(const TLightRay& lrayToCopy) :

    __eInOut(lrayToCopy.__eInOut)
    , __vecDir(lrayToCopy.__vecDir)
    , __vecDir2(lrayToCopy.__vecDir2)
    , __vecOrg(lrayToCopy.__vecOrg)
    , __vecOrg2(lrayToCopy.__vecOrg2)
    , __vecOrgDir(lrayToCopy.__vecOrgDir)
    , __vecMixOrg(lrayToCopy.__vecMixOrg)
    , __vecMixDir(lrayToCopy.__vecMixDir)
    , __vecMixOrgDir(lrayToCopy.__vecMixOrgDir)
{
}

TLightRay::TLightRay(   const   T3DVector&          vecDir
                        , const T3DVector&          vecOrg
                        , const tCIDTracer::EInOut  eInOut) :
    __eInOut(eInOut)
    , __vecDir(vecDir)
    , __vecDir2(vecDir)
    , __vecOrg(vecOrg)
    , __vecOrg2(vecOrg)
    , __vecOrgDir(vecOrg * vecDir)
    , __vecMixOrg(vecMix(vecOrg, vecOrg))
    , __vecMixDir(vecMix(vecDir, vecDir))
    , __vecMixOrgDir(vecMix(vecDir, vecOrg))
{
    // Square the squared term versions of the origin and direction
    __vecDir2.Square();
    __vecOrg2.Square();

    // Finish off the mixed org/dir vector
    __vecMixOrgDir += vecMix(__vecOrg, __vecDir);
}


// -----------------------------------------------------------------------------
//  TLightRay: Public operators
// -----------------------------------------------------------------------------

TLightRay&  TLightRay::operator=(const TLightRay& lrayToAssign)
{
    if (this == &lrayToAssign)
        return *this;

    __eInOut        = lrayToAssign.__eInOut;
    __vecDir        = lrayToAssign.__vecDir;
    __vecDir2       = lrayToAssign.__vecDir2;
    __vecOrg        = lrayToAssign.__vecOrg;
    __vecOrg2       = lrayToAssign.__vecOrg2;
    __vecOrgDir     = lrayToAssign.__vecOrgDir;
    __vecMixOrg     = lrayToAssign.__vecMixOrg;
    __vecMixDir     = lrayToAssign.__vecMixDir;
    __vecMixOrgDir  = lrayToAssign.__vecMixOrgDir;

    return *this;
}


tCIDLib::TBoolean TLightRay::operator==(const TLightRay& lrayToTest) const
{
    //
    //  Compare the values of the objects. We don't bother to check any
    //  of the vectors other than the origin and direction. All of the
    //  others are calculated from these.
    //
    if ((__eInOut != lrayToTest.__eInOut)
    ||  (__vecDir != lrayToTest.__vecDir)
    ||  (__vecOrg != lrayToTest.__vecOrg))
    {
        return kCIDLib::False;
    }
    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TLightRay: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TLightRay::CalcRayPoint(const   tCIDLib::TFloat8    f8Mag
                        ,       T3DVector&          vecTarget) const
{
    vecTarget.Set
    (
        (f8Mag * __vecDir.f8X() + __vecOrg.f8X())
        , (f8Mag * __vecDir.f8Y() + __vecOrg.f8Y())
        , (f8Mag * __vecDir.f8Z() + __vecOrg.f8Z())
    );
}


tCIDLib::TVoid 
TLightRay::CalcReflectedRay(const   T3DVector&  vecPoint
                            , const T3DVector&  vecNormal
                            ,       TLightRay&  lrayTarget) const
{
    tCIDLib::TFloat8 f8K;

    // The origin is the point of intersection of course
    lrayTarget.__vecOrg = vecPoint;

    f8K = __vecDir.f8Dot(vecNormal);
    f8K *= -2.0;

    //  Calculate the direction
    lrayTarget.__vecDir.Set
    (
        f8K * vecNormal.f8X() + __vecDir.f8X()
        , f8K * vecNormal.f8Y() + __vecDir.f8Y()
        , f8K * vecNormal.f8Z() + __vecDir.f8Z()
    );
}


const T3DVector&  TLightRay::vecDir(const T3DVector& vecNew)
{
    __vecDir = vecNew;
    __vecDir2 = vecNew;
    __vecDir2.Square();

    __vecOrgDir     = __vecOrg * __vecDir;
    __vecMixOrg     = vecMix(__vecOrg, __vecOrg);
    __vecMixDir     = vecMix(__vecDir, __vecDir);
    __vecMixOrgDir  = vecMix(__vecOrg, __vecDir);
    __vecMixOrgDir += vecMix(__vecDir, __vecOrg);

    return __vecDir;
}

const T3DVector&  TLightRay::vecOrg(const T3DVector& vecNew)
{
    __vecOrg = vecNew;
    __vecOrg2 = vecNew;
    __vecOrg2.Square();

    __vecOrgDir     = __vecOrg * __vecDir;
    __vecMixOrg     = vecMix(__vecOrg, __vecOrg);
    __vecMixDir     = vecMix(__vecDir, __vecDir);
    __vecMixOrgDir  = vecMix(__vecOrg, __vecDir);
    __vecMixOrgDir += vecMix(__vecDir, __vecOrg);

    return __vecOrg;
}


tCIDLib::TVoid 
TLightRay::SetVectors(const T3DVector& vecOrg, const T3DVector& vecDir)
{
    __vecDir = vecDir;
    __vecDir2 = vecDir;
    __vecDir2.Square();

    __vecOrg = vecOrg;
    __vecOrg2 = vecOrg;
    __vecOrg2.Square();

    __vecOrgDir     = __vecOrg * __vecDir;
    __vecMixOrg     = vecMix(__vecOrg, __vecOrg);
    __vecMixDir     = vecMix(__vecDir, __vecDir);
    __vecMixOrgDir  = vecMix(__vecOrg, __vecDir);
    __vecMixOrgDir += vecMix(__vecDir, __vecOrg);
    
}

// -----------------------------------------------------------------------------
//  TLightRay: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TLightRay::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the geometry info into the string
    strmToWriteTo << L"Dir=" << __vecDir << L", Org=" << __vecOrg;
}


tCIDLib::TVoid  TLightRay::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Just ask all of vectors to stream back in
    strmToReadFrom >> __eInOut;
    strmToReadFrom >> __vecOrg;
    strmToReadFrom >> __vecOrg2;
    strmToReadFrom >> __vecDir;
    strmToReadFrom >> __vecDir2;
    strmToReadFrom >> __vecOrgDir;
    strmToReadFrom >> __vecMixOrgDir;
    strmToReadFrom >> __vecMixOrg;
    strmToReadFrom >> __vecMixDir;
}

tCIDLib::TVoid  TLightRay::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << __eInOut;
    strmToWriteTo << __vecOrg;
    strmToWriteTo << __vecOrg2;
    strmToWriteTo << __vecDir;
    strmToWriteTo << __vecDir2;
    strmToWriteTo << __vecOrgDir;
    strmToWriteTo << __vecMixOrgDir;
    strmToWriteTo << __vecMixOrg;
    strmToWriteTo << __vecMixDir;
}




// -----------------------------------------------------------------------------
//  CLASS: TLightSource
// PREFIX: lsrc
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TLightSource: Constructors and Destructors
// -----------------------------------------------------------------------------

TLightSource::TLightSource( const   T3DVector&  vecCenter
                            , const TFRGBClr&   frgbColor):
    __frgbClr(frgbColor)
    , __vecCenter(vecCenter)
{
}

TLightSource::TLightSource(const TLightSource& lsrcSrc) :

    TRTObject(lsrcSrc)
    , __frgbClr(lsrcSrc.__frgbClr)
    , __vecCenter(lsrcSrc.__vecCenter)
{
}

TLightSource::TLightSource() :

    __frgbClr(1.0, 1.0, 1.0)
    , __vecCenter(0.0, 0.0, 0.0)
{
}


// -----------------------------------------------------------------------------
//  TLightSource: Public, Inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TLightSource::AppendTexture(        TRTTexture* const ptxtrAppend
                            , const TString&
                            , const TString&)
{
    delete ptxtrAppend;
    facCIDTracer.LogMsg
    (
        __FILE__
        , __LINE__
        , L"This should not be called"
        , tCIDLib::ESev_Status
        , tCIDLib::EClass_Internal
    );
}


tCIDLib::TVoid  TLightSource::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strName(kCIDLib::pszEmptyZStr, 64);
    TString                 strToken(kCIDLib::pszEmptyZStr, 128);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Must be followed by an open parens
    prsiSrc.CheckOpenParen();

    if (eToken == TParseInfo::eToken_Center)
    {
        prsiSrc.ParseVector(__vecCenter);
    }
     else if (eToken == TParseInfo::eToken_Color)
    {
        prsiSrc.ParseColor(__frgbClr);
    }
     else
    {
        // UnGet and pass it on to our parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Must end with a closing paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid 
TLightSource::PrependTexture(           TRTTexture* const   ptxtrPrepend
                                , const TString&
                                , const TString&)
{
    delete ptxtrPrepend;
    facCIDTracer.LogMsg
    (
        __FILE__
        , __LINE__
        , L"This should not be called"
        , tCIDLib::ESev_Status
        , tCIDLib::EClass_Internal
    );
}


tCIDLib::TVoid  TLightSource::Transform(const TTransform& trnsApply)
{
    // Transform ourself
    trnsApply.mtrxTrans.Transform(__vecCenter);
}


// -----------------------------------------------------------------------------
//  TLightSource: Public, virtual methods
// -----------------------------------------------------------------------------

TLightSource&  TLightSource::operator=(const TLightSource& lsrcSrc)
{
    if (this == &lsrcSrc)
        return *this;

    // Copy our parent class' members
    TParent::operator=(lsrcSrc);

    __vecCenter = lsrcSrc.__vecCenter;
    __frgbClr   = lsrcSrc.__frgbClr;

    return *this;
}


tCIDLib::TBoolean 
TLightSource::operator==(const TLightSource& lsrcToTest) const
{
    // Call our parent's version first
    if (!TParent::operator==(lsrcToTest))
        return kCIDLib::False;

    // Compare the values of the objects
    if (__frgbClr != lsrcToTest.__frgbClr)
        return kCIDLib::False;

    if (__vecCenter != lsrcToTest.__vecCenter)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TLightSource: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TFloat8 
TLightSource::f8MakeLightRay(const T3DVector& vecPoint, TLightRay& lrayTarget) const
{
    tCIDLib::TFloat8 f8Distance;

    T3DVector vecTmp(__vecCenter);
    vecTmp -= vecPoint;
    f8Distance = vecTmp.f8Magnitude();
    vecTmp.Normalize();

    // Set the origin and direction
    lrayTarget.SetVectors(vecPoint, vecTmp);

    return f8Distance;
}


// -----------------------------------------------------------------------------
//  TLightSource: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TLightSource::_FormatTo(TTextStream& strmToWriteTo) const
{
    //
    //  Only the color attribute is used for light sources, so don't just
    //  call the parent class, Just format the center point then the color
    //  from TRTObject.
    //
    strmToWriteTo   << L"Center: (" << __vecCenter 
                    << L"), Color: (" << __frgbClr << L")";
}


tCIDLib::TVoid 
TLightSource::_Rotate(  const   T3DVector&   vecRotate
                        , const TTransform& trnsRotate)
{
    // Apply it to the center origin
    trnsRotate.mtrxTrans.Transform(__vecCenter);
}


tCIDLib::TVoid  TLightSource::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the center point vector out
    strmToReadFrom >> __frgbClr;
    strmToReadFrom >> __vecCenter;
}

tCIDLib::TVoid 
TLightSource::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the center point vector
    strmToWriteTo << __frgbClr;
    strmToWriteTo << __vecCenter;
}


tCIDLib::TVoid 
TLightSource::_Translate(   const   T3DVector&   vecTrans
                            , const TTransform& trnsTranslate)
{
    // Apply it to the center origin.
    __vecCenter += vecTrans;
}
