//
// NAME: CIDTracer_Transform.Cpp
//
// DESCRIPTION:
//
//  This module implements the TTransform class, which provides standard
//  transformation support for all objects and textures.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 05/14/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI stuff for our classes
// -----------------------------------------------------------------------------
RTTIData(TTransform,TObject)


// -----------------------------------------------------------------------------
//  CLASS: TTransform
// PREFIX: trns
// -----------------------------------------------------------------------------

TTransform::TTransform( const   T4By4Matrix&    mtrxNormal
                        , const T4By4Matrix&    mtrxInverse) :
    mtrxTrans(mtrxNormal)
    , mtrxInvTrans(mtrxInverse)
{
}

TTransform::TTransform(const TTransform& trnsToCopy) :

    mtrxTrans(trnsToCopy.mtrxTrans)
    , mtrxInvTrans(trnsToCopy.mtrxInvTrans)
{
}


// -----------------------------------------------------------------------------
//  TTransform: Public operators
// -----------------------------------------------------------------------------

TTransform& TTransform::operator=(const TTransform& trnsToAssign)
{
    if (this == &trnsToAssign)
        return *this;

    mtrxTrans = trnsToAssign.mtrxTrans;
    mtrxInvTrans = trnsToAssign.mtrxInvTrans;

    return *this;
}


TTransform operator*(const TTransform& trns1, const TTransform& trns2)
{
    // Note that inverse is in opposite direction!
    return TTransform
    (
        trns1.mtrxTrans * trns2.mtrxTrans
        ,  trns2.mtrxInvTrans * trns1.mtrxInvTrans
    );
}

tCIDLib::TVoid TTransform::operator*=(const TTransform& trnsSrc)
{
    // Note that inverse is in opposite direction!
    mtrxTrans *= trnsSrc.mtrxTrans;
    mtrxInvTrans = trnsSrc.mtrxInvTrans * mtrxInvTrans;
}


tCIDLib::TBoolean TTransform::operator==(const TTransform& trnsToTest) const
{
    // Compare the values of the objects
    if (mtrxInvTrans != trnsToTest.mtrxInvTrans)
        return kCIDLib::False;

    if (mtrxTrans != trnsToTest.mtrxTrans)
        return kCIDLib::False;

    return kCIDLib::True;
}


// -----------------------------------------------------------------------------
//  TTransform: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTransform::Reset()
{
    mtrxTrans.SetToIdentity();
    mtrxInvTrans.SetToIdentity();
}


tCIDLib::TVoid TTransform::Rotate(const T3DVector& vecRotate)
{
    T4By4Matrix mtrxTmp;

    // Get a copy of the vector and convert to radians
    T3DVector vecTmp(vecRotate);
    vecTmp.ToRadians();

    // Get the sine and cosine of each value
    tCIDLib::TFloat8 f8SinX, f8SinY, f8SinZ;
    tCIDLib::TFloat8 f8CosX, f8CosY, f8CosZ;
    f8SinX = TMathLib::f8Sine(vecTmp.f8X());
    f8CosX = TMathLib::f8Cosine(vecTmp.f8X());
    f8SinY = TMathLib::f8Sine(vecTmp.f8Y());
    f8CosY = TMathLib::f8Cosine(vecTmp.f8Y());
    f8SinZ = TMathLib::f8Sine(vecTmp.f8Z());
    f8CosZ = TMathLib::f8Cosine(vecTmp.f8Z());

    //
    //  We can put the x rotation straight in, since we are starting from
    //  the identity matrix
    //
    mtrxTrans.SetToIdentity();
    mtrxTrans[tCIDLib::EMatrix_22] = f8CosX;
    mtrxTrans[tCIDLib::EMatrix_33] = f8CosX;
    mtrxTrans[tCIDLib::EMatrix_23] = f8SinX;
    mtrxTrans[tCIDLib::EMatrix_32] = -f8SinX;

    // And create the inverse for this rotation
    mtrxInvTrans.SetToTransposed(mtrxTrans);

    // Concatenate the Y rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_33] = f8CosY;
    mtrxTmp[tCIDLib::EMatrix_13] = -f8SinY;
    mtrxTmp[tCIDLib::EMatrix_31] = f8SinY;
    mtrxTrans *= mtrxTmp;

    // And do the inverse
    mtrxTmp.Transpose();
    mtrxInvTrans = mtrxTmp * mtrxInvTrans;

    // Concatenate the Z rotation
    mtrxTmp.SetToIdentity();
    mtrxTmp[tCIDLib::EMatrix_11] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_22] = f8CosZ;
    mtrxTmp[tCIDLib::EMatrix_12] = f8SinZ;
    mtrxTmp[tCIDLib::EMatrix_21] = -f8SinZ;
    mtrxTrans *= mtrxTmp;

    // And do the inverse
    mtrxTmp.Transpose();
    mtrxInvTrans = mtrxTmp * mtrxInvTrans;
}


tCIDLib::TVoid TTransform::Scale(const T3DVector& vecScale)
{
    mtrxTrans.SetToIdentity();
    mtrxTrans[tCIDLib::EMatrix_11] = vecScale.f8X();
    mtrxTrans[tCIDLib::EMatrix_22] = vecScale.f8Y();
    mtrxTrans[tCIDLib::EMatrix_33] = vecScale.f8Z();

    mtrxInvTrans[tCIDLib::EMatrix_11] = 1.0 / vecScale.f8X();
    mtrxInvTrans[tCIDLib::EMatrix_22] = 1.0 / vecScale.f8Y();
    mtrxInvTrans[tCIDLib::EMatrix_33] = 1.0 / vecScale.f8Z();
}


tCIDLib::TVoid TTransform::Translate(const T3DVector& vecTrans)
{
    mtrxTrans[tCIDLib::EMatrix_Xt] = vecTrans.f8X();
    mtrxTrans[tCIDLib::EMatrix_Yt] = vecTrans.f8Y();
    mtrxTrans[tCIDLib::EMatrix_Zt] = vecTrans.f8Z();

    mtrxInvTrans[tCIDLib::EMatrix_Xt] = 0.0 - vecTrans.f8X();
    mtrxInvTrans[tCIDLib::EMatrix_Yt] = 0.0 - vecTrans.f8Y();
    mtrxInvTrans[tCIDLib::EMatrix_Zt] = 0.0 - vecTrans.f8Z();
}


// -----------------------------------------------------------------------------
//  TTransform: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TTransform::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << L"Trans:" << mtrxTrans
                    << L"\nInv:" << mtrxInvTrans;
}


tCIDLib::TVoid TTransform::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    strmToReadFrom >> mtrxTrans;
    strmToReadFrom >> mtrxInvTrans;
}

tCIDLib::TVoid TTransform::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    strmToWriteTo << mtrxTrans;
    strmToWriteTo << mtrxInvTrans;
}
