//
// NAME: CIDTracer_Texture.Hpp
//
// DESCRIPTION: 
//
//  This is the header file for the CIDTracer_Texture.Cpp module. This module
//  implements the TRTTexture class, which is the basic class for applying
//  textures to objects. Each object has an optional pointer to a texture.
//  If a texture pointer is present, the shader will call the texture
//  object's with the intersection point and surface normal, and the texture
//  object will return the color and potentially adjust the surface normal
//  if the text uses bump mapping.
//
//  If no texture pointer is present, then the shader just starts off with
//  the ambient color of the object at the intersection point.
//
//  TRTTexture holds a TTransform object that is used to transform the texture
//  to the desired location and rotation. It provides the methods Scale(),
//  Rotate(), and Translate() to apply transforms to the texture. It also
//  provides the trnsTexture() method to allow derived texture classes to
//  get a reference to the transform.
//
//
// AUTHOR: Dean Roddey
//
// CREATE DATE: 05/08/94
//
// COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS: 
//


// -----------------------------------------------------------------------------
//  Forward references
// -----------------------------------------------------------------------------
class   TParseInfo;


#pragma pack(push, CIDLIBPACK)

// -----------------------------------------------------------------------------
//  CLASS: TRTTexture
// PREFIX: txtr
// -----------------------------------------------------------------------------
class CIDTRACEREXP TRTTexture :

    public TObject, public MDuplicable, public MFormattable, public MStreamable
{
    public  :
        // ---------------------------------------------------------------------
        // Constructors and Destructors
        // ---------------------------------------------------------------------
        TRTTexture();

        TRTTexture
        (
            const   TRTTexture&             txtrToCopy
        );

        ~TRTTexture();



        // ---------------------------------------------------------------------
        //  Public operators
        // ---------------------------------------------------------------------
        TRTTexture& TRTTexture::operator=
        (
            const   TRTTexture&             txtrToAssign
        );


        // ---------------------------------------------------------------------
        //  Public, virtual methods
        // ---------------------------------------------------------------------
        virtual tCIDLib::TBoolean bAdjustNormalAt
        (
            const   T3DVector&              vecPoint
            ,       T3DVector&              vecNormal
        )   const;

        virtual tCIDLib::TVoid ParseMethod
        (
                    TParseInfo&             prsiSource
        );

        virtual tCIDLib::TVoid QueryColorAt
        (
            const   T3DVector&              vecPoint
            ,       TFRGBClr&               frgbClr
        )   const;


        // ---------------------------------------------------------------------
        //  Public, non-virtual methods
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean bMetallic() const;

        tCIDLib::TBoolean bMetallic
        (
            const   tCIDLib::TBoolean       bNew
        );

        tCIDLib::TVoid CalcDiffuse
        (
            const   TLightRay&              lrayLight
            , const TFRGBClr&               frgbSurface
            , const TFRGBClr&               frgbLight
            , const T3DVector&              vecNormal
            ,       TFRGBClr&               frgbColor
        )   const;

        tCIDLib::TVoid CalcPhong
        (
            const   TLightRay&              lrayLight
            , const TFRGBClr&               frgbSurface
            , const TFRGBClr&               frgbLight
            , const T3DVector&              vecNormal
            , const T3DVector&              vecEyePos
            ,       TFRGBClr&               frgbColor
        )   const;

        tCIDLib::TVoid CalcSpecular
        (
            const   TLightRay&              lraySrc
            , const TFRGBClr&               frgbSurface
            , const TFRGBClr&               frgbLight
            , const T3DVector&              vecRevEye
            , const T3DVector&              vecNormal
            ,       TFRGBClr&               rgbColor
        )   const;

        tCIDLib::TFloat8 f8Ambient
        (
        )   const;

        tCIDLib::TFloat8 f8Ambient
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Diffuse
        (
        )   const;

        tCIDLib::TFloat8 f8Diffuse
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Brilliance() const;

        tCIDLib::TFloat8 f8Brilliance
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Phong() const;

        tCIDLib::TFloat8 f8Phong
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8PhongSize() const;

        tCIDLib::TFloat8 f8PhongSize
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Randomness() const;

        tCIDLib::TFloat8 f8Randomness
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Roughness() const;

        tCIDLib::TFloat8 f8Roughness
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Reflection() const;

        tCIDLib::TFloat8 f8Reflection
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Refraction() const;

        tCIDLib::TFloat8 f8Refraction
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8RefracIndex() const;

        tCIDLib::TFloat8 f8RefracIndex
        (
            const   tCIDLib::TFloat8        f8New
        );

        tCIDLib::TFloat8 f8Specular() const;

        tCIDLib::TFloat8 f8Specular
        (
            const   tCIDLib::TFloat8        f8New
        );

        const TFRGBClr& frgbClr() const;

        const TFRGBClr& frgbClr
        (
            const   TFRGBClr&               frgbNew
        );

        tCIDLib::TVoid Rotate
        (
            const   T3DVector&              vecRotate
        );

        tCIDLib::TVoid Rotate
        (
            const   tCIDLib::TFloat8        f8X
            , const tCIDLib::TFloat8        f8Y
            , const tCIDLib::TFloat8        f8Z
        );

        tCIDLib::TVoid Scale
        (
            const   T3DVector&              vecScale
        );

        tCIDLib::TVoid Scale
        (
            const   tCIDLib::TFloat8        f8X
            , const tCIDLib::TFloat8        f8Y
            , const tCIDLib::TFloat8        f8Z
        );

        tCIDLib::TVoid SetAttribs
        (
            const   TFRGBClr&               frgbClr
            , const tCIDLib::TFloat8        f8Ambient
            , const tCIDLib::TFloat8        f8Diffuse
            , const tCIDLib::TFloat8        f8Brilliance
            , const tCIDLib::TFloat8        f8Phong
            , const tCIDLib::TFloat8        f8PhongSize
            , const tCIDLib::TFloat8        f8Roughness
            , const tCIDLib::TFloat8        f8Reflection
            , const tCIDLib::TFloat8        f8Refraction
            , const tCIDLib::TFloat8        f8RefracIndex
            , const tCIDLib::TFloat8        f8Specular
        );

        tCIDLib::TVoid Translate
        (
            const   T3DVector&               vecTrans
        );

        tCIDLib::TVoid Translate
        (
            const   tCIDLib::TFloat8        f8X
            , const tCIDLib::TFloat8        f8Y
            , const tCIDLib::TFloat8        f8Z
        );

        tCIDLib::TVoid Transform
        (
            const   TTransform&             trnsApply
        );

        const TTransform& trnsTexture() const;


    protected   :
        // ---------------------------------------------------------------------
        //  Protected, inherited methods
        // ---------------------------------------------------------------------
        tCIDLib::TVoid _FormatTo
        (
                    TTextStream&            strmToWriteTo
        )   const;

        tCIDLib::TVoid _StreamFrom
        (
                    TBinaryStream&          strmToReadFrom
        );

        tCIDLib::TVoid _StreamTo
        (
                    TBinaryStream&          strmToWriteTo
        )   const;


    private :
        // ---------------------------------------------------------------------
        //  Priaate data members
        //
        //  __bMetallic
        //      This flag will cause the color calculations to add a metallic
        //      effect.
        //
        //  __f8Ambient
        //      The amount of ambient lighting
        //
        //  __f8Diffuse
        //      The amount of diffuse lighting
        //
        //  __f8Brilliance
        //      The brilliance factor
        //
        //  __f8Phong
        //  __f8PhongSize
        //      These members control phong reflections during the calculation
        //      of the reflected light value.
        //
        //  __f8Randomness
        //      This is a measure of randomness that is applied to the diffuse
        //      color contribution to give a more realistic effect.
        //
        //  __f8Roughness
        //      The roughness of the surface of the object.
        //
        //  __f8Reflection
        //      The reflectivity of the object. 1.0 is a mirror, 0.0 is a
        //      solid black object.
        //
        //  __f8Refraction
        //      The refractivity of an object. 1.0 is transparent, 0.0 is
        //      opaque.
        //
        //  __f8RefracIndex
        //      This is a measure of the speed at which light travels through
        //      an object.
        //
        //  __f8Specular
        //      The amount of specular reflection this object has.
        //
        //  __frgbClr
        //      This is the color of the surface of this object.
        //
        //  __trnsTexture
        //      This is the tranformations that are applied to the texture.
        // ---------------------------------------------------------------------
        tCIDLib::TBoolean       __bMetallic;
        tCIDLib::TFloat8        __f8Ambient;
        tCIDLib::TFloat8        __f8Diffuse;
        tCIDLib::TFloat8        __f8Brilliance;
        tCIDLib::TFloat8        __f8Phong;
        tCIDLib::TFloat8        __f8PhongSize;
        tCIDLib::TFloat8        __f8Randomness;
        tCIDLib::TFloat8        __f8Roughness;
        tCIDLib::TFloat8        __f8Reflection;
        tCIDLib::TFloat8        __f8Refraction;
        tCIDLib::TFloat8        __f8RefracIndex;
        tCIDLib::TFloat8        __f8Specular;
        TFRGBClr                __frgbClr;
        TTransform              __trnsTexture;


        // --------------------------------------------------------------------
        //  Do any needed magic macros
        // --------------------------------------------------------------------
        RTTIMacros(TRTTexture,TObject)
        DefPolyDup(TRTTexture)
};

#pragma pack(pop)



// -----------------------------------------------------------------------------
//  Public, virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean
TRTTexture::bAdjustNormalAt(const   T3DVector&  vecPoint
                            ,       T3DVector&  vecNormal) const
{
    return kCIDLib::False;
}

inline tCIDLib::TVoid
TRTTexture::QueryColorAt(const  T3DVector&  vecPoint
                        ,       TFRGBClr&   frgbClr) const
{
    frgbClr = __frgbClr;
}


// -----------------------------------------------------------------------------
//  Public, non-virtual methods
// -----------------------------------------------------------------------------
inline tCIDLib::TBoolean TRTTexture::bMetallic() const
{
    return __bMetallic;
}

inline tCIDLib::TBoolean
TRTTexture::bMetallic(const tCIDLib::TBoolean bNew)
{
    __bMetallic = bNew;
    return bNew;
}

inline tCIDLib::TFloat8 TRTTexture::f8Ambient() const
{
    return __f8Ambient;
}

inline tCIDLib::TFloat8 TRTTexture::f8Diffuse() const
{
    return __f8Diffuse;
}

inline tCIDLib::TFloat8 TRTTexture::f8Brilliance() const
{
    return __f8Brilliance;
}

inline tCIDLib::TFloat8 TRTTexture::f8Brilliance(const tCIDLib::TFloat8 f8New)
{
    __f8Brilliance = f8New;
    return __f8Brilliance;
}

inline tCIDLib::TFloat8 TRTTexture::f8Phong() const
{
    return __f8Phong;
}

inline tCIDLib::TFloat8
TRTTexture::f8Phong(const tCIDLib::TFloat8 f8New)
{
    __f8Phong = f8New;
    return __f8Phong;
}

inline tCIDLib::TFloat8 TRTTexture::f8PhongSize() const
{
    return __f8PhongSize;
}

inline tCIDLib::TFloat8
TRTTexture::f8PhongSize(const tCIDLib::TFloat8 f8New)
{
    __f8PhongSize = f8New;
    return __f8PhongSize;
}

inline tCIDLib::TFloat8 TRTTexture::f8Randomness() const
{
    return __f8Randomness;
}

inline tCIDLib::TFloat8
TRTTexture::f8Randomness(const tCIDLib::TFloat8 f8New)
{
    __f8Randomness = f8New;
    return f8New;
}

inline tCIDLib::TFloat8 TRTTexture::f8Roughness() const
{
    return __f8Roughness;
}

inline tCIDLib::TFloat8 TRTTexture::f8Roughness(const tCIDLib::TFloat8 f8New)
{
    __f8Roughness = f8New;
    return __f8Roughness;
}

inline tCIDLib::TFloat8 TRTTexture::f8Reflection() const
{
    return __f8Reflection;
}

inline tCIDLib::TFloat8
TRTTexture::f8Reflection(const tCIDLib::TFloat8 f8New)
{
    __f8Reflection = f8New;
    return __f8Reflection;
}

inline tCIDLib::TFloat8 TRTTexture::f8Refraction() const
{
    return __f8Refraction;
}

inline tCIDLib::TFloat8
TRTTexture::f8Refraction(const tCIDLib::TFloat8 f8New)
{
    __f8Refraction = f8New;
    return __f8Refraction;
}

inline tCIDLib::TFloat8 TRTTexture::f8RefracIndex() const
{
    return __f8RefracIndex;
}

inline tCIDLib::TFloat8
TRTTexture::f8RefracIndex(const tCIDLib::TFloat8 f8New)
{
    __f8RefracIndex = f8New;
    return __f8RefracIndex;
}

inline tCIDLib::TFloat8 TRTTexture::f8Specular() const
{
    return __f8Specular;
}

inline tCIDLib::TFloat8
TRTTexture::f8Specular(const tCIDLib::TFloat8 f8New)
{
    __f8Specular = f8New;
    return __f8Specular;
}

inline const TFRGBClr& TRTTexture::frgbClr() const
{
    return __frgbClr;
}

inline const TFRGBClr&
TRTTexture::frgbClr(const TFRGBClr& frgbNew)
{
    __frgbClr = frgbNew;
    return __frgbClr;
}

inline tCIDLib::TVoid
TRTTexture::Rotate( const   tCIDLib::TFloat8    f8X
                    , const tCIDLib::TFloat8    f8Y
                    , const tCIDLib::TFloat8    f8Z)
{
    Rotate(T3DVector(f8X, f8Y, f8Z));
}

inline tCIDLib::TVoid
TRTTexture::Scale(  const   tCIDLib::TFloat8    f8X
                    , const tCIDLib::TFloat8    f8Y
                    , const tCIDLib::TFloat8    f8Z)
{
    Scale(T3DVector(f8X, f8Y, f8Z));
}

inline tCIDLib::TVoid
TRTTexture::Translate(  const   tCIDLib::TFloat8        f8X
                        , const tCIDLib::TFloat8        f8Y
                        , const tCIDLib::TFloat8        f8Z)
{
    Translate(T3DVector(f8X, f8Y, f8Z));
}

inline const TTransform& TRTTexture::trnsTexture() const
{
    return __trnsTexture;
}


// -----------------------------------------------------------------------------
//  Typedef a list of texture objects
// -----------------------------------------------------------------------------
typedef TRTListNode<TRTTexture>     TRTTxtrNode;
typedef TRTList<TRTTexture>         TRTTxtrList;
