//
// NAME: CIDTracer_Texture.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTTexture class, which is the basis for all
//  textures.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 05/12/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our RTTI stuff for our classes
// -----------------------------------------------------------------------------
RTTIData(TRTTexture,TObject)



// -----------------------------------------------------------------------------
//  CLASS: TRTTexture
// PREFIX: txtr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTTexture: Constructors and Destructors.
// -----------------------------------------------------------------------------

TRTTexture::TRTTexture() :

    __bMetallic(kCIDLib::False)
    , __f8Ambient(0.3)
    , __f8Diffuse(0.7)
    , __f8Brilliance(1.0)
    , __f8Phong(0.0)
    , __f8PhongSize(50)
    , __f8Randomness(0.0)
    , __f8Roughness(0.05)
    , __f8Reflection(0.0)
    , __f8Refraction(0.0)
    , __f8RefracIndex(1.0)
    , __f8Specular(0.0)
    , __frgbClr(0.5, 0.5, 0.5)
{
}

TRTTexture::TRTTexture(const TRTTexture& txtrToCopy) :

    __bMetallic(txtrToCopy.__bMetallic)
    , __f8Ambient(txtrToCopy.__f8Ambient)
    , __f8Diffuse(txtrToCopy.__f8Diffuse)
    , __f8Brilliance(txtrToCopy.__f8Brilliance)
    , __f8Phong(txtrToCopy.__f8Phong)
    , __f8PhongSize(txtrToCopy.__f8PhongSize)
    , __f8Randomness(txtrToCopy.__f8Randomness)
    , __f8Roughness(txtrToCopy.__f8Roughness)
    , __f8Reflection(txtrToCopy.__f8Reflection)
    , __f8Refraction(txtrToCopy.__f8Refraction)
    , __f8RefracIndex(txtrToCopy.__f8RefracIndex)
    , __f8Specular(txtrToCopy.__f8Specular)
    , __frgbClr(txtrToCopy.__frgbClr)
    , __trnsTexture(txtrToCopy.__trnsTexture)
{
}

TRTTexture::~TRTTexture()
{
}


// -----------------------------------------------------------------------------
//  TRTTexture: Public operators
// -----------------------------------------------------------------------------

TRTTexture& TRTTexture::operator=(const TRTTexture& txtrToAssign)
{
    if (&txtrToAssign == this)
        return *this;

    __bMetallic = txtrToAssign.__bMetallic;
    __f8Ambient = txtrToAssign.__f8Ambient;
    __f8Diffuse = txtrToAssign.__f8Diffuse;
    __f8Brilliance = txtrToAssign.__f8Brilliance;
    __f8Phong = txtrToAssign.__f8Phong;
    __f8PhongSize = txtrToAssign.__f8PhongSize;
    __f8Randomness = txtrToAssign.__f8Randomness;
    __f8Roughness = txtrToAssign.__f8Roughness;
    __f8Reflection = txtrToAssign.__f8Reflection;
    __f8Refraction = txtrToAssign.__f8Refraction;
    __f8RefracIndex = txtrToAssign.__f8RefracIndex;
    __f8Specular = txtrToAssign.__f8Specular;
    __frgbClr = txtrToAssign.__frgbClr;
    __trnsTexture = txtrToAssign.__trnsTexture;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTTexture: Public, virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRTTexture::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens eToken;
    tCIDLib::TFloat8    f8Tmp;
    TString             strToken;

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Must be followed by an open parens
    prsiSrc.CheckOpenParen();

    // Deal with the valid methods at this level
    if (eToken == TParseInfo::eToken_Ambient)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Ambient(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Brilliance)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 1) || (f8Tmp > 20.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(1.0)
                , TFloat(20.0)
            );
        }
        f8Brilliance(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Color)
    {
        prsiSrc.ParseColor(__frgbClr);
    }
     else if (eToken == TParseInfo::eToken_Diffuse)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Diffuse(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Metallic)
    {
        prsiSrc.ParseBoolean(__bMetallic);
    }
     else if (eToken == TParseInfo::eToken_Phong)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Phong(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_PhongSize)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 1.0) || (f8Tmp > 100.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(1.0)
                , TFloat(100.0)
            );
        }
        f8PhongSize(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Randomness)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Randomness(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Reflection)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Reflection(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Refraction)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Refraction(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_RefractionIndex)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 2.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(2.0)
            );
        }
        f8RefracIndex(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Roughness)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 100.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(100.0)
            );
        }
        f8Roughness(f8Tmp);
    }
     else if (eToken == TParseInfo::eToken_Specular)
    {
        prsiSrc.ParseFloat(f8Tmp);

        if ((f8Tmp < 0) || (f8Tmp > 1.0))
        {
            prsiSrc.ThrowErr
            (
                kTracerErrs::errcParse_InvalidValue
                , strToken
                , TFloat(0.0)
                , TFloat(1.0)
            );
        }
        f8Specular(f8Tmp);
    }
     else if ((eToken == TParseInfo::eToken_Scale)
          ||  (eToken == TParseInfo::eToken_Rotate)
          ||  (eToken == TParseInfo::eToken_Translate))
    {
        T3DVector vecTmp;
        prsiSrc.ParseVector(vecTmp);
        vecTmp.ClipToDegrees();

        if (eToken == TParseInfo::eToken_Scale)
            Scale(vecTmp);
        else if (eToken == TParseInfo::eToken_Rotate)
            Rotate(vecTmp);
        else if (eToken == TParseInfo::eToken_Translate)
            Translate(vecTmp);
    }
     else
    {
        // Not a valid method
        prsiSrc.ThrowErr(kTracerErrs::errcParse_UnknownMethod, strToken);
    }

    // Gotta be a close parens
    prsiSrc.CheckCloseParen();
}



// -----------------------------------------------------------------------------
//  TRTTexture: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid
TRTTexture::CalcDiffuse(const   TLightRay&  lrayLight
                        , const TFRGBClr&   frgbSurface
                        , const TFRGBClr&   frgbLight
                        , const T3DVector&  vecNormal
                        ,       TFRGBClr&   frgbColor) const
{
	tCIDLib::TFloat8 f8CosIncidence, f8Intensity, f8RandomNumber;

    f8CosIncidence = vecNormal.f8Dot(lrayLight.vecDir());
	if (f8CosIncidence < 0.0)
		f8CosIncidence = -f8CosIncidence;

	if (__f8Brilliance != 1.0)
    {
        if (f8CosIncidence != 0.0)
		    f8Intensity = TMathLib::f8Power(f8CosIncidence, __f8Brilliance);
         else
            f8Intensity = 0.0;
    }
	 else
    {
		f8Intensity = f8CosIncidence;
    }

	f8Intensity *= __f8Diffuse;

    if (__f8Randomness != 0.0)
    {
	    f8RandomNumber = tCIDLib::TFloat8(TMathLib::c4RandomNum() & 0x7FFF)
                                                / (tCIDLib::TFloat8)0x7FFF;
	    f8Intensity -= (f8RandomNumber * __f8Randomness);
    }

    TFRGBClr frgbTmp(frgbSurface);
    frgbTmp *= frgbLight;
    frgbTmp *= f8Intensity;

    frgbColor += frgbTmp;
}


tCIDLib::TVoid
TRTTexture::CalcPhong(  const   TLightRay&  lrayLight
                        , const TFRGBClr&   frgbSurface
                        , const TFRGBClr&   frgbLight
                        , const T3DVector&  vecNormal
                        , const T3DVector&  vecEyePos
                        ,       TFRGBClr&   frgbColor) const
{
    tCIDLib::TFloat8 f8CosIncidence, f8NormalLen, f8Intensity;
    T3DVector         vecNormProj(vecNormal);
    T3DVector         vecReflectDir(vecEyePos);
    T3DVector         vecTmpNormal(vecNormal);

    f8CosIncidence = vecEyePos.f8Dot(vecNormal);

    if (f8CosIncidence < 0.0)
        f8CosIncidence = -f8CosIncidence;
     else
        vecTmpNormal *= -1.0;

    vecNormProj *= f8CosIncidence;
    vecNormProj *= 2.0;
    vecReflectDir += vecNormProj;

    f8CosIncidence = vecReflectDir.f8Dot(lrayLight.vecDir());
    f8NormalLen = lrayLight.vecDir().f8Magnitude();

    if (f8NormalLen < kCIDTracer_::f8TinyValue)
        f8CosIncidence = 0.0;
     else
        f8CosIncidence /= f8NormalLen;

    if (f8CosIncidence < 0.0)
        f8CosIncidence = 0.0;

    if (__f8PhongSize != 1.0)
        f8Intensity = TMathLib::f8Power(f8CosIncidence, __f8PhongSize);
     else
        f8Intensity = f8CosIncidence;

    // If it would not make any noticable contribution, then just return
    if (f8Intensity < kCIDTracer_::f8TinyValue)
        return;

    f8Intensity *= __f8Phong;

    if (__bMetallic)
        frgbColor.AddScaled(frgbSurface, f8Intensity);
     else
        frgbColor.AddScaled(frgbLight, f8Intensity);
}


tCIDLib::TVoid
TRTTexture::CalcSpecular(const   TLightRay& lraySrc
                         , const TFRGBClr&  frgbSurface
                         , const TFRGBClr&  frgbLight
                         , const T3DVector& vecRevEye
                         , const T3DVector& vecNormal
                         ,       TFRGBClr&  frgbColor) const
{
	tCIDLib::TFloat8 f8CosIncidence, f8NormalLen, f8Intensity;
    tCIDLib::TFloat8 f8HalfwayLen, f8RoughTmp;

    // Get a vector halfway between the light and reversed eye
	T3DVector vecHalfway(vecRevEye);
    vecHalfway.HalfBetween(lraySrc.vecDir());

    // Get its magnitude and that of the surface normal
    f8HalfwayLen = vecHalfway.f8Magnitude();
    f8NormalLen  = vecNormal.f8Magnitude();

    // Calc the cosine of the angle of incidence
    f8CosIncidence = vecHalfway.f8Dot(vecNormal);

    if ((f8NormalLen == 0.0) || (f8HalfwayLen == 0.0))
    {
        f8CosIncidence = 0.0;
    }
	 else
    {
        tCIDLib::TFloat8 f8Tmp = (f8NormalLen * f8HalfwayLen);
        if (f8Tmp < kCIDTracer_::f8TinyValue)
            f8CosIncidence = 0.0;
         else
            f8CosIncidence /= f8Tmp;

        if (f8CosIncidence < 0.0)
            f8CosIncidence = 0.0;
    }

    //
    //  Calculate the roughness value and use it to calculate the intensity
    //  of specular reflection.
    //
    f8RoughTmp = 1.0 / __f8Roughness;

    if (f8RoughTmp != 1.0)
        f8Intensity = TMathLib::f8Power(f8CosIncidence, f8RoughTmp);
	 else
        f8Intensity = f8CosIncidence;

    // If it would not make any noticable contribution, then just return
    if (f8Intensity < kCIDTracer_::f8TinyValue)
        return;

    // Scale the intensity by the specular value.
    f8Intensity *= __f8Specular;

    if (__bMetallic)
        frgbColor.AddScaled(frgbSurface, f8Intensity);
     else
        frgbColor.AddScaled(frgbLight, f8Intensity);
}


tCIDLib::TFloat8 TRTTexture::f8Ambient(const tCIDLib::TFloat8 f8New)
{
    tCIDLib::TFloat8    f8Actual = f8New;
    if (f8Actual > 1.0)
        f8Actual = 1.0;

    __f8Ambient = f8Actual;

    if (f8Actual + __f8Diffuse > 1.0)
        __f8Diffuse = 1.0 - f8Actual;

    return __f8Ambient;
}


tCIDLib::TFloat8 TRTTexture::f8Diffuse(const tCIDLib::TFloat8 f8New)
{
    tCIDLib::TFloat8    f8Actual = f8New;

    if (f8Actual > 1.0)
        f8Actual = 1.0;

    __f8Diffuse = f8Actual;

    if (f8Actual + __f8Ambient > 1.0)
        __f8Ambient = 1.0 - f8Actual;

    return __f8Diffuse;
}


tCIDLib::TVoid TRTTexture::Rotate(const T3DVector& vecRotate)
{
    TTransform   trnsTmp;

    // Set the temp tranformation to the indicated scaling tranform
    trnsTmp.Rotate(vecRotate);

    // And concate that tranform to the texture transform
    __trnsTexture *= trnsTmp;
}


tCIDLib::TVoid TRTTexture::Scale(const T3DVector& vecScale)
{
    TTransform  trnsTmp;

    // Set the temp tranformation to the indicated scaling tranform
    trnsTmp.Scale(vecScale);

    // And concate that tranform to the texture transform
    __trnsTexture *= trnsTmp;
}
 

tCIDLib::TVoid
TRTTexture::SetAttribs( const   TFRGBClr&           frgbClr
                        , const tCIDLib::TFloat8    f8Ambient
                        , const tCIDLib::TFloat8    f8Diffuse
                        , const tCIDLib::TFloat8    f8Brilliance
                        , const tCIDLib::TFloat8    f8Phong
                        , const tCIDLib::TFloat8    f8PhongSize
                        , const tCIDLib::TFloat8    f8Roughness
                        , const tCIDLib::TFloat8    f8Reflection
                        , const tCIDLib::TFloat8    f8Refraction
                        , const tCIDLib::TFloat8    f8RefracIndex
                        , const tCIDLib::TFloat8    f8Specular)
{
    __frgbClr       = frgbClr;
    __f8Ambient     = f8Ambient;
    __f8Diffuse     = f8Diffuse;
    __f8Phong       = f8Phong;
    __f8PhongSize   = f8PhongSize;
    __f8Brilliance  = f8Brilliance;
    __f8Roughness   = f8Roughness;
    __f8Reflection  = f8Reflection;
    __f8Refraction  = f8Refraction;
    __f8RefracIndex = f8RefracIndex;
    __f8Specular    = f8Specular;
}


tCIDLib::TVoid TRTTexture::Translate(const T3DVector& vecTrans)
{
    TTransform  trnsTmp;

    // Set the temp tranformation to the indicated translation
    trnsTmp.Translate(vecTrans);

    // And concate that tranform to the texture transform
    __trnsTexture *= trnsTmp;
}


tCIDLib::TVoid TRTTexture::Transform(const TTransform& trnsApply)
{
    __trnsTexture *= trnsApply;
}


// -----------------------------------------------------------------------------
//  TRTTexture: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid TRTTexture::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << L"Amb:" << __f8Ambient
                    << L", Dif:" << __f8Diffuse
                    << L", Brl:" << __f8Brilliance
                    << L", Phng:" << __f8Phong
                    << L", PhSz:" << __f8PhongSize
                    << L", Rof:" << __f8Roughness
                    << L", Rfl:" << __f8Reflection
                    << L", Rrf:" << __f8Refraction
                    << L", RrfI:" << __f8RefracIndex
                    << L", Spc:" << __f8Specular;
}


tCIDLib::TVoid TRTTexture::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Get the object data members out
    strmToReadFrom >> __frgbClr;
    strmToReadFrom >> __trnsTexture;

    // Get the values out
    strmToReadFrom >> __f8Ambient;
    strmToReadFrom >> __f8Diffuse;
    strmToReadFrom >> __f8Brilliance;
    strmToReadFrom >> __f8Phong;
    strmToReadFrom >> __f8PhongSize;
    strmToReadFrom >> __f8Randomness;
    strmToReadFrom >> __f8Roughness;
    strmToReadFrom >> __f8Reflection;
    strmToReadFrom >> __f8Refraction;
    strmToReadFrom >> __f8RefracIndex;
    strmToReadFrom >> __f8Specular;
}

tCIDLib::TVoid TRTTexture::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Write out the object data members
    strmToWriteTo << __frgbClr;
    strmToWriteTo << __trnsTexture;

    // Write out the fundamental data members
    strmToWriteTo << __f8Ambient;
    strmToWriteTo << __f8Diffuse;
    strmToWriteTo << __f8Brilliance;
    strmToWriteTo << __f8Phong;
    strmToWriteTo << __f8PhongSize;
    strmToWriteTo << __f8Randomness;
    strmToWriteTo << __f8Roughness;
    strmToWriteTo << __f8Reflection;
    strmToWriteTo << __f8Refraction;
    strmToWriteTo << __f8RefracIndex;
    strmToWriteTo << __f8Specular;
}
