//
// NAME: CIDTracer_Sphere.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTSphere class, which (of course) represents
//  a sphere in the 3D space. It has a center point vector and a radius,
//  along with the standard attributes provided by the parent class.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/05/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//


// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TRTSphere,TRTGeometricObj)


// -----------------------------------------------------------------------------
//  CLASS: TRTSphere
// PREFIX: sphr
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTSphere: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTSphere::TRTSphere() :

    __vecCenter(0.0, 0.0, 0.0)
    , __f8Radius(0.5)
    , __f8RadiusSqr(__f8Radius * __f8Radius)
{
}

TRTSphere::TRTSphere(const TRTSphere& sphrSrc) :

    TRTGeometricObj(sphrSrc)
    , __vecCenter(sphrSrc.__vecCenter)
    , __f8Radius(sphrSrc.__f8Radius)
    , __f8RadiusSqr(sphrSrc.__f8Radius * sphrSrc.__f8Radius)
{
}


// -----------------------------------------------------------------------------
//  TRTSphere: Public operators
// -----------------------------------------------------------------------------

TRTSphere&  TRTSphere::operator=(const TRTSphere& sphrSrc)
{
    if (this == &sphrSrc)
        return *this;

    // Copy our parent's members
    ((TParent&)*this) = (const TParent&)sphrSrc;

    __vecCenter = sphrSrc.__vecCenter;
    __f8Radius = sphrSrc.__f8Radius;
    __f8RadiusSqr = __f8Radius * __f8Radius;

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTSphere: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTSphere::CalcSurfaceNormal(   const   TIntersect* const   pintrSrc
                                ,       T3DVector&          vecNormal) const
{
    vecNormal = pintrSrc->vecIntersect - __vecCenter;
    vecNormal *= (1.0 / __f8Radius);
}


tCIDLib::TVoid  TRTSphere::Invert()
{
    TParent::Invert();
}


tCIDLib::TVoid  TRTSphere::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strToken(kCIDLib::pszEmptyZStr, 256);

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Next has to be an open paren
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_Center)
    {
        prsiSrc.ParseVector(__vecCenter);
    }
     else if (eToken == TParseInfo::eToken_Radius)
    {
        prsiSrc.ParseFloat(__f8Radius);
    }
     else
    {
        // Unget the token and pass it to parent for one shot
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Must end with a close paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid  TRTSphere::Transform(const TTransform& trnsApply)
{
    // Call our parent
    TParent::Transform(trnsApply);

    // Transform ourself
    trnsApply.mtrxTrans.Transform(__vecCenter);

    //
    //  Make a vector from the radius and transform it. Since a sphere can
    //  only be scaled uniformly, we just get the x back out of the vector.
    //
    T3DVector vecTmp(__f8Radius, 0, 0);
    trnsApply.mtrxTrans.Transform(vecTmp);
    __f8Radius = vecTmp.f8X();
    __f8RadiusSqr = __f8Radius * __f8Radius;
}


// -----------------------------------------------------------------------------
//  TRTSphere: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTSphere::_AllIntersects(  const   TLightRay&          lrayTest
                            ,       TIntersectArray&    intraHits) const
{
    tCIDLib::TFloat8 f8Dist1, f8Dist2;

    if (_bTestIntersect(lrayTest, f8Dist1, f8Dist2))
    {
        intraHits.Add(new TIntersect(lrayTest, this, f8Dist1));

        if (f8Dist1 != f8Dist2)
            intraHits.Add(new TIntersect(lrayTest, this, f8Dist2));
    }
}


tCIDLib::TBoolean 
TRTSphere::_bPointInObj(const T3DVector& vecPoint) const
{
    tCIDLib::TFloat8    f8Tmp;
    T3DVector           vecTmp(__vecCenter);

    vecTmp -= vecPoint;
    f8Tmp = vecTmp.f8Dot(vecTmp);

    if (bInverted())
    {
        if (f8Tmp - __f8RadiusSqr > kCIDTracer_::f8SmallTolerance)
            return kCIDLib::True;
    }
     else
    {
        if (f8Tmp - __f8RadiusSqr < kCIDTracer_::f8SmallTolerance)
            return kCIDLib::True;
    }
    return kCIDLib::False;
}


tCIDLib::TBoolean 
TRTSphere::_bTestIntersect( const   TLightRay&          lrayTest
                            ,       tCIDLib::TFloat8&   f8Dist1
                            ,       tCIDLib::TFloat8&   f8Dist2) const
{
    tCIDLib::TBoolean   bInside;
    tCIDLib::TFloat8    f8Squared, f8Closest, f8Half, f8HalfSqr;
    T3DVector           vecOrgToCenter(__vecCenter);

    vecOrgToCenter -= lrayTest.vecOrg();
    f8Squared = vecOrgToCenter.f8Dot(vecOrgToCenter);

    bInside = kCIDLib::False;
    if (f8Squared < __f8RadiusSqr)
        bInside = kCIDLib::True;

    f8Closest = vecOrgToCenter.f8Dot(lrayTest.vecDir());

    if (!bInside && (f8Closest < kCIDTracer_::f8SmallTolerance))
        return kCIDLib::False;

    f8HalfSqr = __f8RadiusSqr - f8Squared + (f8Closest * f8Closest);

    if (f8HalfSqr < kCIDTracer_::f8SmallTolerance)
        return kCIDLib::False;

    f8Half = TMathLib::f8SqrRoot(f8HalfSqr);
    f8Dist1 = f8Closest + f8Half;
    f8Dist2 = f8Closest - f8Half;

    if ((f8Dist1 < kCIDTracer_::f8SmallTolerance)
    ||  (f8Dist1 > kCIDTracer_::f8MaxDistance))
    {
        if ((f8Dist2 < kCIDTracer_::f8SmallTolerance)
        ||  (f8Dist2 > kCIDTracer_::f8MaxDistance))
        {
            return kCIDLib::False;
        }
         else
        {
            f8Dist1 = f8Dist2;
        }
    }
     else
    {
        if ((f8Dist2 < kCIDTracer_::f8SmallTolerance)
        ||  (f8Dist2 > kCIDTracer_::f8MaxDistance))
        {
            f8Dist2 = f8Dist1;
        }
    }
    return kCIDLib::True;
}


tCIDLib::TVoid  TRTSphere::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << L"CenterPnt: (" << __vecCenter << L"), "
                    << L"Radius: " << __f8Radius;
}


tCIDLib::TVoid  TRTSphere::_Rotate( const   T3DVector&   vecRotate
                                    , const TTransform& trnsRotate)
{
    // Call the parent version
    TParent::_Rotate(vecRotate, trnsRotate);

    // Apply it to the center origin
    trnsRotate.mtrxTrans.Transform(__vecCenter);
}


tCIDLib::TVoid  TRTSphere::_Scale(  const   T3DVector&  vecScale
                                    , const TTransform& trnsScale)
{
    #if CID_DEBUG_ON
    if ((vecScale.f8X() != vecScale.f8Y())
    ||  (vecScale.f8X() != vecScale.f8Z()))
    {
        facCIDTracer.LogErr
        (
            __FILE__
            , __LINE__
            , kTracerErrs::errcSphere_NonUniformScale
            , tCIDLib::ESev_ProcessFatal
            , tCIDLib::EClass_AppError
            , vecScale
        );
    }
    #endif

    // Call the parent version
    TParent::_Scale(vecScale, trnsScale);

    //
    //  Since spheres can only be scaled uniformly, just multiple the
    //  radius times the x value and scale the centerpoint by it.
    //
    __f8Radius *= vecScale.f8X();
    __f8RadiusSqr = __f8Radius * __f8Radius;
    __vecCenter *= vecScale.f8X();
}


tCIDLib::TVoid  TRTSphere::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the center point vector out
    strmToReadFrom >> __vecCenter;

    // Get the radius out
    strmToReadFrom >> __f8Radius;
}

tCIDLib::TVoid  TRTSphere::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the center point vector
    strmToWriteTo << __vecCenter;

    // Write out the radius
    strmToWriteTo << __f8Radius;
}


tCIDLib::TVoid 
TRTSphere::_Translate(  const   T3DVector&   vecTrans
                        , const TTransform& trnsTranslate)
{
    // Call the parent version
    TParent::_Translate(vecTrans, trnsTranslate);

    // Apply it to the center origin.
    __vecCenter += vecTrans;
}
