//
// NAME: CIDTracer_Quadric.Cpp
//
// DESCRIPTION:
//
//  This module implements the TRTQuadric class, which represents a
//  quadric shape in the 3D space.
//
//
//  AUTHOR: Dean Roddey
//
//  CREATE DATE: 03/13/94
//
//  COPYRIGHT: 1992..1997, 'CIDCorp
//
// CAVEATS/GOTCHAS:
//



// -----------------------------------------------------------------------------
//  Facility specific includes
// -----------------------------------------------------------------------------
#include    "CIDTracer_.Hpp"


// -----------------------------------------------------------------------------
//  Do our standard members macros
// -----------------------------------------------------------------------------
RTTIData2(TRTQuadric,TRTGeometricObj)


// -----------------------------------------------------------------------------
//  CLASS: TRTQuadric
// PREFIX: quad
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  TRTQuadric: Constructors and Destructors
// -----------------------------------------------------------------------------

TRTQuadric::TRTQuadric() :

    __bNonZero(kCIDLib::True)
    , __f8Constant(-1.0)
    , __pplnClip1(0)
    , __pplnClip2(0)
    , __vecABC(1.0, 1.0, 1.0)
    , __vecDEF(0.0, 0.0, 0.0)
    , __vecGHI(0.0, 0.0, 0.0)
{
}

TRTQuadric::TRTQuadric( const   T3DVector&          vecABC
                        , const T3DVector&          vecDEF
                        , const T3DVector&          vecGHI
                        , const tCIDLib::TFloat8    f8Constant) :

    __bNonZero(kCIDLib::True)
    , __f8Constant(f8Constant)
    , __pplnClip1(0)
    , __pplnClip2(0)
    , __vecABC(vecABC)
    , __vecDEF(vecDEF)
    , __vecGHI(vecGHI)
{
    if (__vecABC.bAtOrg() && __vecDEF.bAtOrg())
        __bNonZero = kCIDLib::False;
}

TRTQuadric::TRTQuadric(const TRTQuadric& quadSrc) :

    TRTGeometricObj(quadSrc)
    , __bNonZero(quadSrc.__bNonZero)
    , __f8Constant(quadSrc.__f8Constant)
    , __pplnClip1(0)
    , __pplnClip2(0)
    , __vecABC(quadSrc.__vecABC)
    , __vecDEF(quadSrc.__vecDEF)
    , __vecGHI(quadSrc.__vecGHI)
{
    if (quadSrc.__pplnClip1)
        __pplnClip1 = new TRTPlane(*quadSrc.__pplnClip1);

    if (quadSrc.__pplnClip2)
        __pplnClip2 = new TRTPlane(*quadSrc.__pplnClip2);
}

TRTQuadric::~TRTQuadric()
{
    delete __pplnClip1;
    delete __pplnClip2;
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Public operators
// -----------------------------------------------------------------------------

TRTQuadric&  TRTQuadric::operator=(const TRTQuadric& quadSrc)
{
    if (this == &quadSrc)
        return *this;

    // Copy our parent class' members
    ((TParent&)*this) = (const TParent&)quadSrc;

    __bNonZero = quadSrc.__bNonZero;
    __vecABC = quadSrc.__vecABC;
    __vecDEF = quadSrc.__vecDEF;
    __vecGHI = quadSrc.__vecGHI;
    __f8Constant = quadSrc.__f8Constant;

    delete __pplnClip1;
    __pplnClip1 = 0;
    if (quadSrc.__pplnClip1)
        __pplnClip1 = new TRTPlane(*quadSrc.__pplnClip1);

    delete __pplnClip2;
    __pplnClip2 = 0;
    if (quadSrc.__pplnClip2)
        __pplnClip2 = new TRTPlane(*quadSrc.__pplnClip2);

    return *this;
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Public, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTQuadric::AppendTexture(  TRTTexture* const   ptxtrToAppend
                            , const TString&    strName
                            , const TString&    strNameSpace)
{
    TParent::AppendTexture(ptxtrToAppend, strName, strNameSpace);

    if (__pplnClip1)
    {
        __pplnClip1->AppendTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToAppend)
            , strName
            , strNameSpace
        );
    }

    if (__pplnClip2)
    {
        __pplnClip2->AppendTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToAppend)
            , strName
            , strNameSpace
        );
    }
}


tCIDLib::TVoid 
TRTQuadric::CalcSurfaceNormal(  const   TIntersect* const pintrSrc
                                ,       T3DVector&         vecNormal) const
{
    vecNormal = __vecABC;
    vecNormal *= 2.0;
    vecNormal *= pintrSrc->vecIntersect;
    vecNormal += __vecGHI;

    vecNormal.Adjust
    (
        __vecDEF.f8X() * pintrSrc->vecIntersect.f8Y()
            + __vecDEF.f8Y() * pintrSrc->vecIntersect.f8Z()
        , __vecDEF.f8X() * pintrSrc->vecIntersect.f8X()
            + __vecDEF.f8Z() * pintrSrc->vecIntersect.f8Z()
        , __vecDEF.f8Y() * pintrSrc->vecIntersect.f8X()
            + __vecDEF.f8Z() * pintrSrc->vecIntersect.f8Y()
    );

    tCIDLib::TFloat8 f8Len = vecNormal.f8Magnitude();
    if (f8Len == 0.0)
        vecNormal.Set(1.0, 0.0, 0.0);
     else
        vecNormal /= f8Len;
}


tCIDLib::TVoid  TRTQuadric::Invert()
{
    TParent::Invert();

    __vecABC *= -1;
    __vecDEF *= -1;
    __vecGHI *= -1;
    __f8Constant *= -1;

    //
    //  No need to invert the clipping planes!! They still need to
    //  contain the object just as before but only the object gets
    //  flipped.
    //
}


tCIDLib::TVoid  TRTQuadric::ParseMethod(TParseInfo& prsiSrc)
{
    TParseInfo::eTokens     eToken;
    TString                 strName;
    TString                 strNameSpace;
    TString                 strToken;

    // Get the next token
    eToken = prsiSrc.eNextToken(strToken);

    // Next token has to be an open paren
    prsiSrc.CheckOpenParen();

    // Deal with the valid tokens at this level
    if (eToken == TParseInfo::eToken_QuadricValues)
    {
        // Parse the three vectors out
        prsiSrc.ParseVector(__vecABC);
        prsiSrc.CheckComma();
        prsiSrc.ParseVector(__vecDEF);
        prsiSrc.CheckComma();
        prsiSrc.ParseVector(__vecGHI);
    }
     else if (eToken == TParseInfo::eToken_Constant)
    {
        prsiSrc.ParseFloat(__f8Constant);
    }
     else if ((eToken == TParseInfo::eToken_Clip1)
          ||  (eToken == TParseInfo::eToken_Clip2))
    {
        // Save the incoming token here because we are going to use it again
        TParseInfo::eTokens eOrgToken = eToken;

        TRTPlane*   pplnNew = 0;
        try
        {
            //
            //  We can go one of two ways here. One is an inline description
            //  of the plane (a vector and distance) or a named plane object
            //  to copy.
            //
            eToken = prsiSrc.eNextToken(strToken
                                        , TParseInfo::eTokenMode_UnknownOk);

            // Unget it, regardless of which it will be
            prsiSrc.UnGet(strToken);

            if (eToken == TParseInfo::eToken_Vector)
            {
                tCIDLib::TFloat8    f8Tmp;
                T3DVector            vecTmp;

                prsiSrc.ParseVector(vecTmp);
                prsiSrc.CheckComma();
                prsiSrc.ParseFloat(f8Tmp);

                // Create a new plane object
                pplnNew = new TRTPlane(vecTmp, f8Tmp);
            }
             else
            {
                //
                //  Next token must be the name of the plane object to copy. It
                //  can be a template or instance, we don't care.
                //
                prsiSrc.GetName(strName, strNameSpace);
                TRTObject* const prtobjTmp
                                    = prsiSrc.prtobjFind(strName, strNameSpace);
                if (!prtobjTmp)
                {
                    prsiSrc.ThrowErr
                    (
                        kTracerErrs::errcParse_ObjectNotFound
                        , strName
                        , strNameSpace
                    );
                }

                if (!prtobjTmp->bIsDescendantOf(TRTPlane::clsThis))
                {
                    prsiSrc.ThrowErr
                    (
                        kTracerErrs::errcQuadric_MustBePlane
                        , strName
                        , strNameSpace
                    );
                }

                // Create a new object that is a copy of the template
                pplnNew = ::pDupObject<TRTPlane>(*prtobjTmp);
            }

            // And install it as the correct clip plane
            if (eOrgToken == TParseInfo::eToken_Clip1)
                SetClipPlane1(*pplnNew);
             else
                SetClipPlane2(*pplnNew);
        }

        catch(...)
        {
            delete pplnNew;
            throw;
        }

        // And delete our temp since he just copied it
        delete pplnNew;
    }
     else
    {
        // Unget the token and pass it to our parent
        prsiSrc.UnGet("(");
        prsiSrc.UnGet(strToken);
        TParent::ParseMethod(prsiSrc);
        return;
    }

    // Has to end with a closing paren
    prsiSrc.CheckCloseParen();
}


tCIDLib::TVoid 
TRTQuadric::PrependTexture( TRTTexture* const   ptxtrToPrepend
                            , const TString&    strName
                            , const TString&    strNameSpace)
{
    // Append duplicates to the clip planes if present
    if (__pplnClip1)
    {
        __pplnClip1->PrependTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToPrepend)
            , strName
            , strNameSpace
        );
    }

    if (__pplnClip2)
    {
        __pplnClip2->PrependTexture
        (
            ::pDupObject<TRTTexture>(*ptxtrToPrepend)
            , strName
            , strNameSpace
        );
    }

    // Pass it on to my parent
    TParent::PrependTexture(ptxtrToPrepend, strName, strNameSpace);
}


tCIDLib::TVoid  TRTQuadric::Transform(const TTransform& trnsApply)
{
    // Call the parent version
    TParent::Transform(trnsApply);

    T4By4Matrix mtrxQuadric, mtrxTmp;

    // Transform the clip planes that are set
    if (__pplnClip1)
        __pplnClip1->Transform(trnsApply);

    if (__pplnClip2)
        __pplnClip2->Transform(trnsApply);

    // Copy the quadric to a matrix
    _ToMatrix(mtrxQuadric);

    //
    //  Multiply by transform. Note that we CANNOT do mtrxQuadric *= mtrxTmp
    //  because the multiply operation order is important. The inverse matrix
    //  must be first.
    //
    mtrxQuadric = trnsApply.mtrxInvTrans * mtrxQuadric;
    mtrxTmp.SetToTransposed(trnsApply.mtrxInvTrans);
    mtrxQuadric *= mtrxTmp;

    // Copy the matrix back to the quadric
    _FromMatrix(mtrxQuadric);
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Public, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TRTQuadric::SetClipPlane1(const TRTPlane& plnClip)
{
    delete __pplnClip1;

    // If they just want to turn it off, we are done
    if (!(&plnClip))
    {
        __pplnClip1 = 0;
        return;
    }

    __pplnClip1 = new TRTPlane(plnClip);
}

tCIDLib::TVoid  TRTQuadric::SetClipPlane2(const TRTPlane& plnClip)
{
    delete __pplnClip2;

    // If they just want to turn it off, we are done
    if (!(&plnClip))
    {
        __pplnClip2 = 0;
        return;
    }

    __pplnClip2 = new TRTPlane(plnClip);
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Protected, inherited methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid 
TRTQuadric::_AllIntersects( const   TLightRay&          lrayTest
                            ,       TIntersectArray&    intraHits) const
{
    tCIDLib::TFloat8 f8Dist1, f8Dist2;
    T3DVector         vecTmp;

    // Do a raw test of the object without clipping
    if (!__bTestIntersect(lrayTest, f8Dist1, f8Dist2))
        return;

    // If it falls within the clip are, then add it
    lrayTest.CalcRayPoint(f8Dist1, vecTmp);
    if (__bPointInClipArea(vecTmp))
        intraHits.Add(new TIntersect(lrayTest, this, f8Dist1));

    // If the second intersect is different, then add it if within clip area
    if (f8Dist1 != f8Dist2)
    {
        lrayTest.CalcRayPoint(f8Dist2, vecTmp);
        if (__bPointInClipArea(vecTmp))
            intraHits.Add(new TIntersect(lrayTest, this, f8Dist2));
    }

    //
    //  If there are clip planes, get their intersections. If the intersect
    //  point is within the quadric area, then append this intersection.
    //
    if (__pplnClip1)
    {
        if (__pplnClip1->bTestIntersect(lrayTest, f8Dist1, f8Dist2))
        {
            lrayTest.CalcRayPoint(f8Dist1, vecTmp);
            if (__bPointInObject(vecTmp))
                intraHits.Add(new TIntersect(lrayTest, __pplnClip1, f8Dist1));

            if (f8Dist2 != f8Dist1)
            {
                lrayTest.CalcRayPoint(f8Dist2, vecTmp);
                if (__bPointInObject(vecTmp))
                    intraHits.Add(new TIntersect(lrayTest, __pplnClip1, f8Dist2));
            }
        }
    }

    if (__pplnClip2)
    {
        if (__pplnClip2->bTestIntersect(lrayTest, f8Dist1, f8Dist2))
        {
            lrayTest.CalcRayPoint(f8Dist1, vecTmp);
            if (__bPointInObject(vecTmp))
                intraHits.Add(new TIntersect(lrayTest, __pplnClip2, f8Dist1));

            if (f8Dist2 != f8Dist1)
            {
                lrayTest.CalcRayPoint(f8Dist2, vecTmp);
                if (__bPointInObject(vecTmp))
                    intraHits.Add(new TIntersect(lrayTest, __pplnClip2, f8Dist2));
            }
        }
    }
}


tCIDLib::TBoolean 
TRTQuadric::_bTestIntersect(const   TLightRay&          lrayTest
                            ,       tCIDLib::TFloat8&   f8Dist1
                            ,       tCIDLib::TFloat8&   f8Dist2) const
{
    TIntersectArray intraTmp;

    // Get all intersections
    _AllIntersects(lrayTest, intraTmp);

    // Get the number of intersections
    tCIDLib::TCard4 c4Cnt = intraTmp.c4HitCount();

    if (!c4Cnt)
        return kCIDLib::False;

    // Sort by the distance
    if (c4Cnt > 2)
        intraTmp.Sort();

    // Return the distance of the smallest distance
    TIntersect*  pintrTmp = intraTmp[0];
    f8Dist1 = pintrTmp->f8Distance;

    if (f8Dist1 < kCIDTracer_::f8SmallTolerance)
        return kCIDLib::False;

    if (c4Cnt > 1)
    {
        // Do the tail as the exit point
        pintrTmp = intraTmp[c4Cnt-1];
        f8Dist2 = pintrTmp->f8Distance;
    }

    return kCIDLib::True;
}


tCIDLib::TBoolean 
TRTQuadric::_bPointInObj(const T3DVector& vecPoint) const
{
    if (!__bPointInClipArea(vecPoint))
        return kCIDLib::False;

    if (!__bPointInObject(vecPoint))
        return kCIDLib::False;

    return kCIDLib::True;
}


tCIDLib::TVoid  TRTQuadric::_FormatTo(TTextStream& strmToWriteTo) const
{
    // Format the attribute info into the string
    strmToWriteTo   << L"ABC: (" << __vecABC
                    << L"), DEF: (" << __vecDEF
                    << L"), GHI: (" << __vecGHI
                    << L"), Const: " << __f8Constant;
}


tCIDLib::TVoid  TRTQuadric::_Rotate( const   T3DVector&   vecRotate
                                            , const TTransform& trnsRotate)
{
    // Call the parent version
    TParent::_Rotate(vecRotate, trnsRotate);

    T4By4Matrix mtrxQuadric, mtrxTmp;

    // Rotate the clip planes that are set
    if (__pplnClip1)
        __pplnClip1->Rotate(vecRotate);

    if (__pplnClip2)
        __pplnClip2->Rotate(vecRotate);

    // Copy the quadric to a matrix
    _ToMatrix(mtrxQuadric);

    mtrxQuadric = trnsRotate.mtrxInvTrans * mtrxQuadric;
    mtrxTmp.SetToTransposed(trnsRotate.mtrxInvTrans);
    mtrxQuadric *= mtrxTmp;

    // Copy the matrix back to the quadric
    _FromMatrix(mtrxQuadric);
}


tCIDLib::TVoid  TRTQuadric::_Scale(  const   T3DVector&   vecScale
                                            , const TTransform& trnsScale)
{
    // Call the parent version
    TParent::_Scale(vecScale, trnsScale);

    T4By4Matrix mtrxQuadric, mtrxTmp;

    // scale the clip planes that are set
    if (__pplnClip1)
        __pplnClip1->Scale(vecScale);

    if (__pplnClip2)
        __pplnClip2->Scale(vecScale);

    // Copy the quadric to a matrix
    _ToMatrix(mtrxQuadric);

    mtrxQuadric = trnsScale.mtrxInvTrans * mtrxQuadric;
    mtrxTmp.SetToTransposed(trnsScale.mtrxInvTrans);
    mtrxQuadric *= mtrxTmp;

    // Copy the matrix back to the quadric
    _FromMatrix(mtrxQuadric);
}


tCIDLib::TVoid  TRTQuadric::_StreamFrom(TBinaryStream& strmToReadFrom)
{
    // Call our parent's version
    TParent::_StreamFrom(strmToReadFrom);

    // Get the quadratic terms
    strmToReadFrom >> __vecABC;
    strmToReadFrom >> __vecDEF;
    strmToReadFrom >> __vecGHI;

    // Get the constant value
    strmToReadFrom >> __f8Constant;
}

tCIDLib::TVoid 
TRTQuadric::_StreamTo(TBinaryStream& strmToWriteTo) const
{
    // Call our parent's version
    TParent::_StreamTo(strmToWriteTo);

    // Write out the center point vector
    strmToWriteTo << __vecABC;
    strmToWriteTo << __vecDEF;
    strmToWriteTo << __vecGHI;

    // Write out the constant value
    strmToWriteTo << __f8Constant;
}


tCIDLib::TVoid 
TRTQuadric::_Translate( const   T3DVector&   vecTrans
                        , const TTransform& trnsTranslate)
{
    // Call the parent version
    TParent::_Translate(vecTrans, trnsTranslate);

    T4By4Matrix mtrxQuadric, mtrxTmp;

    // Translate the clip planes that are set
    if (__pplnClip1)
        __pplnClip1->Translate(vecTrans);

    if (__pplnClip2)
        __pplnClip2->Translate(vecTrans);

    // Copy the quadric to a matrix
    _ToMatrix(mtrxQuadric);

    mtrxQuadric = trnsTranslate.mtrxInvTrans * mtrxQuadric;
    mtrxTmp.SetToTransposed(trnsTranslate.mtrxInvTrans);
    mtrxQuadric *= mtrxTmp;

    // Copy the matrix back to the quadric
    _FromMatrix(mtrxQuadric);
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Protected, non-virtual methods
// -----------------------------------------------------------------------------

tCIDLib::TVoid  TRTQuadric::_ToMatrix(T4By4Matrix& mtrxTarget)
{
    // Zero out the matrix
    mtrxTarget.Zero();

    mtrxTarget[tCIDLib::EMatrix_11] = __vecABC.f8X();
    mtrxTarget[tCIDLib::EMatrix_22] = __vecABC.f8Y();
    mtrxTarget[tCIDLib::EMatrix_33] = __vecABC.f8Z();

    mtrxTarget[tCIDLib::EMatrix_12] = __vecDEF.f8X();
    mtrxTarget[tCIDLib::EMatrix_13] = __vecDEF.f8Y();
    mtrxTarget[tCIDLib::EMatrix_23] = __vecDEF.f8Z();

    mtrxTarget[tCIDLib::EMatrix_14] = __vecGHI.f8X();
    mtrxTarget[tCIDLib::EMatrix_24] = __vecGHI.f8Y();
    mtrxTarget[tCIDLib::EMatrix_34] = __vecGHI.f8Z();

    mtrxTarget[tCIDLib::EMatrix_44] = __f8Constant;
}


tCIDLib::TVoid  TRTQuadric::_FromMatrix(const T4By4Matrix& mtrxSrc)
{
    __vecABC.Set
    (
        mtrxSrc[tCIDLib::EMatrix_11]
        , mtrxSrc[tCIDLib::EMatrix_22]
        , mtrxSrc[tCIDLib::EMatrix_33]
    );

    __vecDEF.Set
    (
        mtrxSrc[tCIDLib::EMatrix_12]+mtrxSrc[tCIDLib::EMatrix_21]
        , mtrxSrc[tCIDLib::EMatrix_13]+mtrxSrc[tCIDLib::EMatrix_31]
        , mtrxSrc[tCIDLib::EMatrix_23]+mtrxSrc[tCIDLib::EMatrix_32]
    );

    __vecGHI.Set
    (
        mtrxSrc[tCIDLib::EMatrix_14]+mtrxSrc[tCIDLib::EMatrix_Xt]
        , mtrxSrc[tCIDLib::EMatrix_24]+mtrxSrc[tCIDLib::EMatrix_Yt]
        , mtrxSrc[tCIDLib::EMatrix_34]+mtrxSrc[tCIDLib::EMatrix_Zt]
    );

    __f8Constant = mtrxSrc[tCIDLib::EMatrix_44];
}


// -----------------------------------------------------------------------------
//  TRTQuadric: Private, non-virtual methods
// -----------------------------------------------------------------------------

//
// FUNCTION/METHOD NAME: __bPointInClipArea
//
// DESCRIPTION:
//
//  This method will test to see if the passed point is within the clip
//  planes.
// ---------------------------------------
//   INPUT: vecTest is the postion vector to test.
//
//  OUTPUT: None
//
//  RETURN: eTrue if inside, else eFalse.
//
tCIDLib::TBoolean
TRTQuadric::__bPointInClipArea(const T3DVector& vecTest) const
{
    // Make sure the point is with any clip planes
    if (__pplnClip1)
    {
        if (!__pplnClip1->bPointInObj(vecTest))
            return kCIDLib::False;
    }

    if (__pplnClip2)
    {
        if (!__pplnClip2->bPointInObj(vecTest))
            return kCIDLib::False;
    }

    return kCIDLib::True;
}


//
// FUNCTION/METHOD NAME: __bPointInObject
//
// DESCRIPTION:
//
//  This will test the passed position vector to see if it is within the
//  object, without worrying about the clip planes.
// ---------------------------------------
//   INPUT: vecTest is the vector to test
//
//  OUTPUT: None
//
//  RETURN: eTrue if the point is in the object, else eFalse.
//
tCIDLib::TBoolean
TRTQuadric::__bPointInObject(const T3DVector& vecTest) const
{
    tCIDLib::TFloat8 f8Res, f8Linear, f8Square;
    T3DVector         vecTmp(vecTest);

    f8Linear = vecTest.f8Dot(__vecGHI);
    f8Res = f8Linear + __f8Constant;
    vecTmp.Square();
    f8Square = vecTmp.f8Dot(__vecABC);
    f8Res += f8Square;

    f8Res += __vecDEF.f8X() * vecTest.f8X() * vecTest.f8Y()
             + __vecDEF.f8Y() * vecTest.f8X() * vecTest.f8Z()
             + __vecDEF.f8Z() * vecTest.f8Y() * vecTest.f8Z();

    if (f8Res >= kCIDTracer_::f8SmallTolerance)
        return kCIDLib::False;

    return kCIDLib::True;
}


//
// FUNCTION/METHOD NAME: __bTestIntersect
//
// DESCRIPTION:
//
//  This method will test the light ray intersection with this object,
//  without any check for the intersection points being within any clipping
//  planes.
// ---------------------------------------
//   INPUT: lrayTest is the light ray to test
//
//  OUTPUT: f8Dist1, f8Dist2 are both filled in 
//
//  RETURN: eTrue if an intersection, else eFalse.
//
tCIDLib::TBoolean 
TRTQuadric::__bTestIntersect(   const   TLightRay&          lrayTest
                                ,       tCIDLib::TFloat8&   f8Dist1
                                ,       tCIDLib::TFloat8&   f8Dist2) const
{
    tCIDLib::TFloat8 f8LinearTerm, f8ConstantTerm, f8SquareTerm;
    tCIDLib::TFloat8 f8Determinant, f8Determinant2, f8A2, f8BMinus;

    if (__bNonZero)
    {
        f8SquareTerm = __vecABC.f8Dot(lrayTest.vecDir2())
                        + __vecDEF.f8Dot(lrayTest.vecMixDir());
    }
     else
    {
        f8SquareTerm = 0.0;
    }

    // Calculate the linear term
    f8LinearTerm = __vecABC.f8Dot(lrayTest.vecOrgDir()) * 2.0;
    f8LinearTerm += __vecGHI.f8Dot(lrayTest.vecDir());
    f8LinearTerm += __vecDEF.f8Dot(lrayTest.vecMixOrgDir());

    // Calculate the constant term
    f8ConstantTerm = __vecABC.f8Dot(lrayTest.vecOrg2());
    f8ConstantTerm += __vecGHI.f8Dot(lrayTest.vecOrg());
    f8ConstantTerm += __f8Constant;
    f8ConstantTerm += __vecDEF.f8Dot(lrayTest.vecMixOrg());

    if (f8SquareTerm != 0.0)
    {
        f8Determinant2 = f8LinearTerm * f8LinearTerm -
                         4.0 * f8SquareTerm * f8ConstantTerm;

        if (f8Determinant2 < 0.0)
            return kCIDLib::False;

        f8Determinant = TMathLib::f8SqrRoot(f8Determinant2);
        f8A2 = f8SquareTerm * 2.0;
        f8BMinus = f8LinearTerm * -1.0;

        f8Dist1 = (f8BMinus + f8Determinant) / f8A2;
        f8Dist2 = (f8BMinus - f8Determinant) / f8A2;
    }
     else
    {
        if (f8LinearTerm == 0.0)
            return kCIDLib::False;

        f8Dist1 = f8ConstantTerm * -1.0 / f8LinearTerm;
        f8Dist2 = f8Dist1;
    }

    if ((f8Dist1 < kCIDTracer_::f8SmallTolerance)
    ||  (f8Dist1 > kCIDTracer_::f8MaxDistance))
    {
        if ((f8Dist2 < kCIDTracer_::f8SmallTolerance)
        ||  (f8Dist2 > kCIDTracer_::f8MaxDistance))
        {
            return kCIDLib::False;
        }
        f8Dist1 = f8Dist2;
    }
     else
    {
        if ((f8Dist2 < kCIDTracer_::f8SmallTolerance)
        ||  (f8Dist2 > kCIDTracer_::f8MaxDistance))
        {
            f8Dist2 = f8Dist1;
        }
    }
    return kCIDLib::True;
}
